/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "fixtures/tasktest.h"
#include "socialtestutilities.h"
#include "testutilities.h"
#include "twitchsdk/social/internal/task/socialfriendrequeststask.h"
#include "twitchsdk/social/internal/task/socialpresencesettingstask.h"
#include "twitchsdk/social/internal/task/socialrecommendedfriendstask.h"
#include "twitchsdk/social/internal/task/socialupdatefriendtask.h"

#include <algorithm>
#include <iostream>
#include <numeric>

using namespace ttv;
using namespace ttv::test;
using namespace ttv::social;

namespace {
static const UserId kTestUserId = 9001;
static const UserId kTestUserId2 = 9002;
static const std::string kTestOAuthToken = "test_oauth_token";
}  // namespace

TEST_F(TaskTest, SocialFriendRequestsTask_Fetch) {
  uint32_t limit = 2;

  auto response = mHttpRequest
                    ->AddResponse(std::string("https://api.twitch.tv/kraken/users/") + std::to_string(kTestUserId) +
                                  "/friends/requests")
                    .AddRequestParam("direction", "ASC")
                    .AddRequestParam("limit", limit)
                    .AddRequestHeader("Accept", "application/vnd.twitchtv.v3+json")
                    .AddRequestHeader("Authorization", std::string("OAuth ") + kTestOAuthToken)
                    .SetResponseBodyFromFile("social/socialfriendrequeststask_fetch.json")
                    .Done();

  bool callbackCalled = false;
  std::shared_ptr<SocialFriendRequestsTask> task;

  SocialFriendRequestsTask::Callback callback = [&callbackCalled, &task](SocialFriendRequestsTask* source,
                                                  TTV_ErrorCode ec,
                                                  std::shared_ptr<SocialFriendRequestsTask::Result> result) {
    callbackCalled = true;

    ASSERT_EQ(task.get(), source);
    ASSERT_TRUE(TTV_SUCCEEDED(ec));
    ASSERT_TRUE(result != nullptr);

    ASSERT_EQ(result->cursor, "test_cursor");
  };

  auto checkFunction = [&callbackCalled]() -> bool { return callbackCalled; };

  task = std::make_shared<SocialFriendRequestsTask>(kTestUserId, kTestOAuthToken, callback);
  task->FetchRequests(limit, SocialFriendRequestsTask::SortDirection::Ascending, "");
  mTaskRunner->AddTask(task);

  ASSERT_TRUE(WaitUntilResultWithPollTask(1000, checkFunction, GetDefaultUpdateFunc()));

  response->AssertRequestsMade();

  mHttpRequest->RemoveResponse(response);

  // Try with a cursor
  response = mHttpRequest
               ->AddResponse(
                 std::string("https://api.twitch.tv/kraken/users/") + std::to_string(kTestUserId) + "/friends/requests")
               .AddRequestParam("direction", "ASC")
               .AddRequestParam("limit", limit)
               .AddRequestParam("cursor", "test_cursor")
               .AddRequestHeader("Accept", "application/vnd.twitchtv.v3+json")
               .AddRequestHeader("Authorization", std::string("OAuth ") + kTestOAuthToken)
               .SetResponseBodyFromFile("social/socialfriendrequeststask_fetch.json")
               .Done();

  callbackCalled = false;

  task = std::make_shared<SocialFriendRequestsTask>(kTestUserId, kTestOAuthToken, callback);
  task->FetchRequests(limit, SocialFriendRequestsTask::SortDirection::Ascending, "test_cursor");
  mTaskRunner->AddTask(task);

  ASSERT_TRUE(WaitUntilResultWithPollTask(3000, checkFunction, GetDefaultUpdateFunc()));
  response->AssertRequestsMade();
}

TEST_F(TaskTest, SocialFriendRequestsTask_MarkAllRead) {
  auto response = mHttpRequest
                    ->AddResponse(std::string("https://api.twitch.tv/kraken/users/") + std::to_string(kTestUserId) +
                                  "/friends/notifications")
                    .SetType(HTTP_DELETE_REQUEST)
                    .AddRequestHeader("Authorization", std::string("OAuth ") + kTestOAuthToken)
                    .AddRequestHeader("Accept", "application/vnd.twitchtv.v3+json")
                    .Done();

  bool callbackCalled = false;
  std::shared_ptr<SocialFriendRequestsTask> task;

  SocialFriendRequestsTask::Callback callback = [&callbackCalled, &task](SocialFriendRequestsTask* source,
                                                  TTV_ErrorCode ec,
                                                  std::shared_ptr<SocialFriendRequestsTask::Result> result) {
    callbackCalled = true;

    ASSERT_EQ(task.get(), source);
    ASSERT_TRUE(TTV_SUCCEEDED(ec));
    ASSERT_TRUE(result != nullptr);
  };

  auto checkFunction = [&callbackCalled]() -> bool { return callbackCalled; };

  task = std::make_shared<SocialFriendRequestsTask>(kTestUserId, kTestOAuthToken, callback);
  task->MarkAllRead();
  mTaskRunner->AddTask(task);

  ASSERT_TRUE(WaitUntilResultWithPollTask(1000, checkFunction, GetDefaultUpdateFunc()));
  response->AssertRequestsMade();
}

TEST_F(TaskTest, SocialFriendRequestsTask_GetUnreadCount) {
  auto response = mHttpRequest
                    ->AddResponse(std::string("https://api.twitch.tv/kraken/users/") + std::to_string(kTestUserId) +
                                  "/friends/notifications")
                    .AddRequestHeader("Accept", "application/vnd.twitchtv.v3+json")
                    .AddRequestHeader("Authorization", std::string("OAuth ") + kTestOAuthToken)
                    .SetResponseBodyFromFile("social/socialfriendrequeststask_getunreadcount.json")
                    .Done();

  bool callbackCalled = false;
  std::shared_ptr<SocialFriendRequestsTask> task;

  SocialFriendRequestsTask::Callback callback = [&callbackCalled, &task](SocialFriendRequestsTask* source,
                                                  TTV_ErrorCode ec,
                                                  std::shared_ptr<SocialFriendRequestsTask::Result> result) {
    callbackCalled = true;

    ASSERT_EQ(task.get(), source);
    ASSERT_TRUE(TTV_SUCCEEDED(ec));
    ASSERT_TRUE(result != nullptr);

    ASSERT_EQ(result->total, 6);
  };

  auto checkFunction = [&callbackCalled]() -> bool { return callbackCalled; };

  task = std::make_shared<SocialFriendRequestsTask>(kTestUserId, kTestOAuthToken, callback);
  task->GetUnreadCount();
  mTaskRunner->AddTask(task);

  ASSERT_TRUE(WaitUntilResultWithPollTask(1000, checkFunction, GetDefaultUpdateFunc()));
  response->AssertRequestsMade();
}

TEST_F(TaskTest, SocialGetFriendsPresenceTask) {
  auto response =
    mHttpRequest
      ->AddResponse(std::string("https://api.twitch.tv/v5/users/") + std::to_string(kTestUserId) + "/status/friends")
      .AddRequestHeader("Accept", "application/vnd.twitchtv.v3+json")
      .AddRequestHeader("Authorization", std::string("OAuth ") + kTestOAuthToken)
      .SetResponseBodyFromFile("social/socialfriendspresencetask.json")
      .Done();

  bool callbackCalled = false;
  std::shared_ptr<SocialGetFriendsPresenceTask> task;

  SocialGetFriendsPresenceTask::Callback callback = [&callbackCalled, &task](SocialGetFriendsPresenceTask* source,
                                                      TTV_ErrorCode ec,
                                                      std::shared_ptr<SocialGetFriendsPresenceTask::Result> result) {
    callbackCalled = true;

    ASSERT_EQ(task.get(), source);
    ASSERT_TRUE(TTV_SUCCEEDED(ec));
    ASSERT_TRUE(result != nullptr);
    ASSERT_EQ(result->presenceList.size(), 4);

    Friend friendData = result->presenceList[0].friendData;
    UserInfo userInfo = result->presenceList[0].friendData.userInfo;

    ASSERT_EQ(PresenceUserAvailability::Offline, friendData.presenceStatus.availability);
    ASSERT_EQ("twitch", userInfo.userName);
    ASSERT_EQ(12826, userInfo.userId);

    friendData = result->presenceList[1].friendData;
    userInfo = result->presenceList[1].friendData.userInfo;

    ASSERT_EQ(PresenceUserAvailability::Online, friendData.presenceStatus.availability);
    ASSERT_EQ("twitch2", userInfo.userName);
    ASSERT_EQ(12827, userInfo.userId);

    friendData = result->presenceList[2].friendData;
    userInfo = result->presenceList[2].friendData.userInfo;

    ASSERT_EQ(PresenceUserAvailability::Away, friendData.presenceStatus.availability);
    ASSERT_EQ("twitch3", userInfo.userName);
    ASSERT_EQ(12828, userInfo.userId);

    friendData = result->presenceList[3].friendData;
    userInfo = result->presenceList[3].friendData.userInfo;

    ASSERT_EQ(PresenceUserAvailability::Busy, friendData.presenceStatus.availability);
    ASSERT_EQ("twitch4", userInfo.userName);
    ASSERT_EQ(12829, userInfo.userId);
  };

  auto checkFunction = [&callbackCalled]() -> bool { return callbackCalled; };

  task = std::make_shared<SocialGetFriendsPresenceTask>(kTestUserId, kTestOAuthToken, callback);
  mTaskRunner->AddTask(task);

  ASSERT_TRUE(WaitUntilResultWithPollTask(1000, checkFunction, GetDefaultUpdateFunc()));
  response->AssertRequestsMade();
}

TEST_F(TaskTest, SocialPresenceGetSettingsTask) {
  auto response =
    mHttpRequest
      ->AddResponse(std::string("https://api.twitch.tv/v5/users/") + std::to_string(kTestUserId) + "/status/settings")
      .AddRequestHeader("Accept", "application/vnd.twitchtv.v3+json")
      .AddRequestHeader("Authorization", std::string("OAuth ") + kTestOAuthToken)
      .SetResponseBodyFromFile("social/socialpresencesettingstask.json")
      .Done();

  bool callbackCalled = false;
  std::shared_ptr<SocialPresenceSettingsTask> task;

  SocialPresenceSettingsTask::Callback callback = [&callbackCalled, &task](SocialPresenceSettingsTask* source,
                                                    TTV_ErrorCode ec,
                                                    std::shared_ptr<SocialPresenceSettingsTask::Result> result) {
    callbackCalled = true;

    ASSERT_EQ(task.get(), source);
    ASSERT_TRUE(TTV_SUCCEEDED(ec));
    ASSERT_TRUE(result != nullptr);

    ASSERT_EQ(result->settings.availabilityOverride, PresenceSettings::AvailabilityOverride::None);
    ASSERT_TRUE(result->settings.shareActivity);
  };

  auto checkFunction = [&callbackCalled]() -> bool { return callbackCalled; };

  task = std::make_shared<SocialPresenceSettingsTask>(kTestUserId, kTestOAuthToken, false, callback);
  mTaskRunner->AddTask(task);

  ASSERT_TRUE(WaitUntilResultWithPollTask(1000, checkFunction, GetDefaultUpdateFunc()));
  response->AssertRequestsMade();
}

TEST_F(TaskTest, SocialPresenceSetSettingsTask) {
  auto response =
    mHttpRequest
      ->AddResponse(std::string("https://api.twitch.tv/v5/users/") + std::to_string(kTestUserId) + "/status/settings")
      .SetType(ttv::HttpRequestType::HTTP_POST_REQUEST)
      .AddRequestHeader("Accept", "application/vnd.twitchtv.v3+json")
      .AddRequestHeader("Authorization", std::string("OAuth ") + kTestOAuthToken)
      .SetResponseHandler([](const std::string& /*url*/, ttv::HttpRequestType /*type*/,
                            const std::vector<ttv::HttpParam>& /*headerParams*/, const std::string& body) {
        json::Reader reader;
        json::Value root;

        EXPECT_TRUE(reader.parse(body, root));

        EXPECT_EQ(root["availability_override"].asString(), "away");
        EXPECT_TRUE(root["share_activity"].asBool());

        MockResponse::MockResponseContent responseContent;
        std::string responseString =
          "{\"availability_override\": \"away\", \"is_invisible\": false, \"share_activity\": true}";
        responseContent.body.assign(responseString.begin(), responseString.end());
        return responseContent;
      })
      .Done();

  bool callbackCalled = false;
  std::shared_ptr<SocialPresenceSettingsTask> task;

  SocialPresenceSettingsTask::Callback callback = [&callbackCalled, &task](SocialPresenceSettingsTask* source,
                                                    TTV_ErrorCode ec,
                                                    std::shared_ptr<SocialPresenceSettingsTask::Result> result) {
    callbackCalled = true;

    ASSERT_EQ(task.get(), source);
    ASSERT_TRUE(TTV_SUCCEEDED(ec));
    ASSERT_TRUE(result != nullptr);

    ASSERT_EQ(result->settings.availabilityOverride, PresenceSettings::AvailabilityOverride::Away);
    ASSERT_TRUE(result->settings.shareActivity);
  };

  auto checkFunction = [&callbackCalled]() -> bool { return callbackCalled; };

  PresenceSettings settings;
  settings.availabilityOverride = PresenceSettings::AvailabilityOverride::Away;
  settings.shareActivity = true;

  task = std::make_shared<SocialPresenceSettingsTask>(kTestUserId, kTestOAuthToken, true, callback);
  task->SetSettings(settings);
  mTaskRunner->AddTask(task);

  ASSERT_TRUE(WaitUntilResultWithPollTask(1000, checkFunction, GetDefaultUpdateFunc()));
  response->AssertRequestsMade();
}

TEST_F(TaskTest, SocialRecommendedFriendsTask) {
  auto response = mHttpRequest
                    ->AddResponse(std::string("https://api.twitch.tv/kraken/users/") + std::to_string(kTestUserId) +
                                  "/friends/recommendations")
                    .AddRequestHeader("Accept", "application/vnd.twitchtv.v3+json")
                    .AddRequestHeader("Authorization", std::string("OAuth ") + kTestOAuthToken)
                    .SetResponseBodyFromFile("social/socialrecommendedfriendstask.json")
                    .Done();

  bool callbackCalled = false;
  std::shared_ptr<SocialRecommendedFriendsTask> task;

  SocialRecommendedFriendsTask::Callback callback = [&callbackCalled, &task](SocialRecommendedFriendsTask* source,
                                                      TTV_ErrorCode ec,
                                                      std::shared_ptr<SocialRecommendedFriendsTask::Result> result) {
    callbackCalled = true;

    ASSERT_EQ(task.get(), source);
    ASSERT_TRUE(TTV_SUCCEEDED(ec));
    ASSERT_TRUE(result != nullptr);

    ASSERT_EQ(SocialRecommendedFriendsTask::Action::Fetch, result->action);
    ASSERT_EQ(1, result->recommendedFriends.size());
    ASSERT_EQ("twitch", result->recommendedFriends[0].userInfo.userName);
  };

  auto checkFunction = [&callbackCalled]() -> bool { return callbackCalled; };

  task = std::make_shared<SocialRecommendedFriendsTask>(kTestUserId, kTestOAuthToken, callback);
  task->Fetch();
  mTaskRunner->AddTask(task);

  ASSERT_TRUE(WaitUntilResultWithPollTask(1000, checkFunction, GetDefaultUpdateFunc()));
  response->AssertRequestsMade();
}

TEST_F(TaskTest, SocialUpdateFriendTask) {
  auto response = mHttpRequest
                    ->AddResponse(std::string("https://api.twitch.tv/kraken/users/") + std::to_string(kTestUserId) +
                                  "/friends/relationships/" + std::to_string(kTestUserId2))
                    .AddRequestHeader("Accept", "application/vnd.twitchtv.v3+json")
                    .AddRequestHeader("Authorization", std::string("OAuth ") + kTestOAuthToken)
                    .SetResponseBodyFromFile("social/socialgetfriendstatus.json")
                    .Done();

  bool callbackCalled = false;
  std::shared_ptr<SocialUpdateFriendTask> task;

  SocialUpdateFriendTask::Callback callback = [&callbackCalled, &task](SocialUpdateFriendTask* source, TTV_ErrorCode ec,
                                                std::shared_ptr<SocialUpdateFriendTask::Result> result) {
    callbackCalled = true;

    ASSERT_EQ(task.get(), source);
    ASSERT_TRUE(TTV_SUCCEEDED(ec));
    ASSERT_TRUE(result != nullptr);

    ASSERT_EQ(FriendStatus::Friends, result->status);
    ASSERT_EQ(UpdateFriendResult::RequestNotAllowed, result->result);
  };

  auto checkFunction = [&callbackCalled]() -> bool { return callbackCalled; };

  task = std::make_shared<SocialUpdateFriendTask>(
    kTestUserId, kTestOAuthToken, kTestUserId2, SocialUpdateFriendTask::Action::GetStatus, callback);
  mTaskRunner->AddTask(task);

  ASSERT_TRUE(WaitUntilResultWithPollTask(1000, checkFunction, GetDefaultUpdateFunc()));
  response->AssertRequestsMade();
}
