/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "socialtestutilities.h"

#include "twitchsdk/core/json/corejsonutil.h"
#include "twitchsdk/core/json/jsonobjectdescriptions.h"
#include "twitchsdk/core/stringutilities.h"
#include "twitchsdk/core/systemclock.h"
#include "twitchsdk/social/socialapi.h"

#include <fstream>

#include "gtest/gtest.h"

using namespace ttv;
using namespace ttv::social;

ttv::test::SampleFriendData::SampleFriendData(const std::string& path) {
  std::ifstream sampleFriendFile(path);
  EXPECT_TRUE(sampleFriendFile.is_open());

  std::string data((std::istreambuf_iterator<char>(sampleFriendFile)), std::istreambuf_iterator<char>());
  json::Reader reader;

  json::Value root;
  EXPECT_TRUE(reader.parse(data, root));

  json::Value userInfos = root["samplefriends"];

  for (const auto& friendJson : userInfos) {
    UserInfo userInfo;
    EXPECT_TRUE(json::ToObject(friendJson, userInfo));
    mSampleUserInfos.push_back(userInfo);
  }

  json::Value activities = root["sampleactivities"];
  for (const auto& activityJson : activities) {
    mSampleActivityJson.push_back(activityJson);
  }
}

const UserInfo& ttv::test::SampleFriendData::GetFriendInfo(std::size_t index) const {
  EXPECT_TRUE(index < mSampleUserInfos.size());
  return mSampleUserInfos[index];
}

const json::Value& ttv::test::SampleFriendData::GetActivityJson(std::size_t index) const {
  EXPECT_TRUE(index < mSampleActivityJson.size());
  return mSampleActivityJson[index];
}

json::Value ttv::test::SampleFriendData::GeneratePresenceJson(const UserInfo& userInfo, Timestamp lastUpdate,
  Timestamp lastHearbeat, uint64_t updateIndex, PresenceUserAvailability availability,
  const json::Value& activityJson) const {
  json::Value presenceEntry;

  std::string availabilityString;
  switch (availability) {
    case PresenceUserAvailability::Online: {
      availabilityString = "online";
      break;
    }

    case PresenceUserAvailability::Offline: {
      availabilityString = "offline";
      break;
    }

    case PresenceUserAvailability::Away: {
      availabilityString = "away";
      break;
    }

    case PresenceUserAvailability::Busy: {
      availabilityString = "busy";
      break;
    }
  }

  presenceEntry["availability"] = std::move(availabilityString);
  presenceEntry["index"] = updateIndex;
  presenceEntry["updated_at"] = lastUpdate;
  presenceEntry["timestamp"] = lastHearbeat;

  json::Value& userJson = presenceEntry["user"];
  userJson["_id"] = std::to_string(userInfo.userId);
  userJson["bio"] = userInfo.bio;
  userJson["created_at"] = UnixTimestampToRFC3339String(userInfo.createdTimestamp);
  userJson["display_name"] = userInfo.displayName;
  userJson["logo"] = userInfo.logoImageUrl;
  userJson["name"] = userInfo.userName;

  presenceEntry["user_id"] = userInfo.userId;

  if (!activityJson.isNull()) {
    presenceEntry["activity"] = activityJson;
  }

  return presenceEntry;
}

ttv::social::SocialFriendRequestsTask::FriendRequestEntry ttv::test::SampleFriendData::GenerateFriendRequestEntry(
  const UserInfo& friendInfo, Timestamp requested) const {
  ttv::social::SocialFriendRequestsTask::FriendRequestEntry friendRequest;
  friendRequest.userInfo = friendInfo;
  friendRequest.requestedTimestamp = requested;

  return friendRequest;
}

Timestamp ttv::test::SampleFriendData::GenerateRandomTimestamp() const {
  const Timestamp aboutFiveYears = 60 * 60 * 24 * 365 * 5;
  return static_cast<Timestamp>(GetSystemTimeMilliseconds() - (std::rand() % aboutFiveYears));
}
