/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/social/internal/pch.h"

#include "twitchsdk/social/java_socialutil.h"

#include "twitchsdk/social/generated/java_all.h"

#include <functional>

extern JNIEnv* ttv::binding::java::
  gActiveJavaEnvironment;  //!< The is cached on every call into native code so that it's current.  Never use from another thread.

using namespace ttv;
using namespace ttv::social;

void ttv::binding::java::LoadAllSocialJavaClassInfo(JNIEnv* jEnv) {
  GetJavaClassInfo_SocialAPI_PostPresenceCallback(jEnv);
  GetJavaClassInfo_SocialAPI_FetchPresenceSettingsCallback(jEnv);
  GetJavaClassInfo_SocialAPI_SetPresenceSettingsCallback(jEnv);
  GetJavaClassInfo_SocialAPI_FetchFriendListCallback(jEnv);
  GetJavaClassInfo_SocialAPI_UpdateFriendshipCallback(jEnv);
  GetJavaClassInfo_SocialAPI_FetchFriendRequestsCallback(jEnv);
  GetJavaClassInfo_SocialAPI_FetchUnreadFriendRequestCountCallback(jEnv);
  GetJavaClassInfo_SocialAPI_MarkAllFriendRequestsReadCallback(jEnv);
  GetJavaClassInfo_SocialAPI_FetchFriendStatusCallback(jEnv);
  GetJavaClassInfo_SocialAPI_FetchRecommendedFriendsCallback(jEnv);
  GetJavaClassInfo_SocialAPI_DismissRecommendedFriendCallback(jEnv);

  GetJavaClassInfo_ISocialAPIListener(jEnv);
  GetJavaClassInfo_SocialPresence(jEnv);
  GetJavaClassInfo_SocialPresenceActivity(jEnv);
  GetJavaClassInfo_SocialPresenceActivityBroadcasting(jEnv);
  GetJavaClassInfo_SocialPresenceActivityWatching(jEnv);
  GetJavaClassInfo_SocialPresenceActivityPlaying(jEnv);
  GetJavaClassInfo_SocialFriend(jEnv);
  GetJavaClassInfo_SocialFriendRequest(jEnv);
  GetJavaClassInfo_SocialFeatureFlags(jEnv);
  GetJavaClassInfo_SocialFriendStatus(jEnv);
  GetJavaClassInfo_SocialPresenceSettings(jEnv);
  GetJavaClassInfo_SocialPresenceSessionAvailability(jEnv);
  GetJavaClassInfo_SocialUpdateFriendAction(jEnv);
  GetJavaClassInfo_SocialUpdateFriendResult(jEnv);
  GetJavaClassInfo_SocialPresenceUserAvailability(jEnv);
  GetJavaClassInfo_SocialPresenceAvailabilityOverride(jEnv);
}

jobject ttv::binding::java::GetJavaInstance_SocialPresenceActivity(JNIEnv* jEnv, const PresenceActivity& val) {
  JavaClassInfo& activityInfo = GetJavaClassInfo_SocialPresenceActivity(jEnv);
  JavaClassInfo& broadcastingActivityInfo = GetJavaClassInfo_SocialPresenceActivityBroadcasting(jEnv);
  JavaClassInfo& watchingActivityInfo = GetJavaClassInfo_SocialPresenceActivityWatching(jEnv);
  JavaClassInfo& playingActivityInfo = GetJavaClassInfo_SocialPresenceActivityPlaying(jEnv);

  auto type = val.GetType();

  AUTO_DELETE_LOCAL_REF(jEnv, jobject, jActivity, nullptr);
  AUTO_DELETE_LOCAL_REF(
    jEnv, jobject, jType, GetJavaInstance_SimpleEnum(jEnv, GetJavaClassInfo_SocialPresenceActivityType(jEnv), type));

  switch (type) {
    case PresenceActivity::Type::Broadcasting: {
      jActivity = jEnv->NewObject(broadcastingActivityInfo.klass, broadcastingActivityInfo.methods["<init>"]);

      const auto& broadcastingActivity = static_cast<const BroadcastingActivity&>(val);

      jEnv->SetIntField(
        jActivity, broadcastingActivityInfo.fields["channelId"], static_cast<jint>(broadcastingActivity.channelId));

      AUTO_DELETE_LOCAL_REF(
        jEnv, jstring, jChannelLogin, GetJavaInstance_String(jEnv, broadcastingActivity.channelLogin));
      jEnv->SetObjectField(jActivity, broadcastingActivityInfo.fields["channelLogin"], jChannelLogin);

      AUTO_DELETE_LOCAL_REF(
        jEnv, jstring, jChannelDisplayName, GetJavaInstance_String(jEnv, broadcastingActivity.channelDisplayName));
      jEnv->SetObjectField(jActivity, broadcastingActivityInfo.fields["channelDisplayName"], jChannelDisplayName);

      jEnv->SetIntField(
        jActivity, broadcastingActivityInfo.fields["gameId"], static_cast<jint>(broadcastingActivity.gameId));

      AUTO_DELETE_LOCAL_REF(jEnv, jstring, jGameName, GetJavaInstance_String(jEnv, broadcastingActivity.gameName));
      jEnv->SetObjectField(jActivity, broadcastingActivityInfo.fields["gameName"], jGameName);

      break;
    }

    case PresenceActivity::Type::Watching: {
      jActivity = jEnv->NewObject(watchingActivityInfo.klass, watchingActivityInfo.methods["<init>"]);

      const auto& watchingActivity = static_cast<const WatchingActivity&>(val);
      jEnv->SetIntField(
        jActivity, watchingActivityInfo.fields["channelId"], static_cast<jint>(watchingActivity.channelId));

      AUTO_DELETE_LOCAL_REF(jEnv, jstring, jChannelLogin, GetJavaInstance_String(jEnv, watchingActivity.channelLogin));
      jEnv->SetObjectField(jActivity, watchingActivityInfo.fields["channelLogin"], jChannelLogin);

      AUTO_DELETE_LOCAL_REF(
        jEnv, jstring, jChannelDisplayName, GetJavaInstance_String(jEnv, watchingActivity.channelDisplayName));
      jEnv->SetObjectField(jActivity, watchingActivityInfo.fields["channelDisplayName"], jChannelDisplayName);

      jEnv->SetIntField(
        jActivity, watchingActivityInfo.fields["hostedChannelId"], static_cast<jint>(watchingActivity.hostedChannelId));

      AUTO_DELETE_LOCAL_REF(
        jEnv, jstring, jHostedChannelLogin, GetJavaInstance_String(jEnv, watchingActivity.hostedChannelLogin));
      jEnv->SetObjectField(jActivity, watchingActivityInfo.fields["hostedChannelLogin"], jHostedChannelLogin);

      AUTO_DELETE_LOCAL_REF(jEnv, jstring, jHostedChannelDisplayName,
        GetJavaInstance_String(jEnv, watchingActivity.hostedChannelDisplayName));
      jEnv->SetObjectField(
        jActivity, watchingActivityInfo.fields["hostedChannelDisplayName"], jHostedChannelDisplayName);

      jEnv->SetIntField(jActivity, watchingActivityInfo.fields["gameId"], static_cast<jint>(watchingActivity.gameId));

      AUTO_DELETE_LOCAL_REF(jEnv, jstring, jGameName, GetJavaInstance_String(jEnv, watchingActivity.gameName));
      jEnv->SetObjectField(jActivity, watchingActivityInfo.fields["gameName"], jGameName);

      break;
    }

    case PresenceActivity::Type::Playing: {
      jActivity = jEnv->NewObject(playingActivityInfo.klass, playingActivityInfo.methods["<init>"]);

      const auto& playingActivity = static_cast<const PlayingActivity&>(val);

      jEnv->SetIntField(jActivity, playingActivityInfo.fields["gameId"], static_cast<jint>(playingActivity.gameId));

      AUTO_DELETE_LOCAL_REF(jEnv, jstring, jGameName, GetJavaInstance_String(jEnv, playingActivity.gameName));
      jEnv->SetObjectField(jActivity, playingActivityInfo.fields["gameName"], jGameName);

      AUTO_DELETE_LOCAL_REF(
        jEnv, jstring, jGameDisplayContext, GetJavaInstance_String(jEnv, playingActivity.gameDisplayContext));
      jEnv->SetObjectField(jActivity, playingActivityInfo.fields["gameDisplayContext"], jGameDisplayContext);
      break;
    }
    default: { break; }
  }

  if (jActivity != nullptr) {
    jEnv->SetObjectField(jActivity, activityInfo.fields["type"], jType);
  }

  return jActivity;
}

jobject ttv::binding::java::GetJavaInstance_SocialPresence(JNIEnv* jEnv, const PresenceStatus& val) {
  JavaClassInfo& info = GetJavaClassInfo_SocialPresence(jEnv);
  JavaClassInfo& availabilityInfo = GetJavaClassInfo_SocialPresenceUserAvailability(jEnv);

  jobject jInstance = jEnv->NewObject(info.klass, info.methods["<init>"]);

  jEnv->SetIntField(jInstance, info.fields["lastPresenceUpdateTime"], val.lastUpdate);

  AUTO_DELETE_LOCAL_REF(
    jEnv, jobject, jAvailability, GetJavaInstance_SimpleEnum(jEnv, availabilityInfo, val.availability));
  jEnv->SetObjectField(jInstance, info.fields["availability"], jAvailability);

  if (val.activity != nullptr) {
    AUTO_DELETE_LOCAL_REF(jEnv, jobject, jActivity, GetJavaInstance_SocialPresenceActivity(jEnv, *val.activity));
    jEnv->SetObjectField(jInstance, info.fields["activity"], jActivity);
  }

  return jInstance;
}

jobject ttv::binding::java::GetJavaInstance_SocialPresenceSettings(JNIEnv* jEnv, const PresenceSettings& val) {
  JavaClassInfo& info = GetJavaClassInfo_SocialPresenceSettings(jEnv);

  jobject jInstance = jEnv->NewObject(info.klass, info.methods["<init>"]);

  AUTO_DELETE_LOCAL_REF(jEnv, jobject, jAvailabilityOverride,
    GetJavaInstance_SimpleEnum(
      jEnv, GetJavaClassInfo_SocialPresenceAvailabilityOverride(jEnv), val.availabilityOverride));
  jEnv->SetObjectField(jInstance, info.fields["availabilityOverride"], jAvailabilityOverride);
  jEnv->SetBooleanField(jInstance, info.fields["shareActivity"], val.shareActivity);

  return jInstance;
}

jobject ttv::binding::java::GetJavaInstance_SocialFriend(JNIEnv* jEnv, const Friend& val) {
  JavaClassInfo& info = GetJavaClassInfo_SocialFriend(jEnv);

  jobject jInstance = jEnv->NewObject(info.klass, info.methods["<init>"]);

  AUTO_DELETE_LOCAL_REF(jEnv, jobject, jUserInfo, GetJavaInstance_UserInfo(jEnv, val.userInfo));
  jEnv->SetObjectField(jInstance, info.fields["userInfo"], jUserInfo);

  jEnv->SetIntField(jInstance, info.fields["friendsSinceTime"], static_cast<jint>(val.friendsSinceTime));

  AUTO_DELETE_LOCAL_REF(jEnv, jobject, jPresence, GetJavaInstance_SocialPresence(jEnv, val.presenceStatus));
  jEnv->SetObjectField(jInstance, info.fields["presence"], jPresence);

  return jInstance;
}

jobject ttv::binding::java::GetJavaInstance_SocialFriendArray(JNIEnv* jEnv, const std::vector<Friend>& list) {
  JavaClassInfo& socialFriendInfo = GetJavaClassInfo_SocialFriend(jEnv);

  jobjectArray jArray = GetJavaInstance_Array(jEnv, socialFriendInfo, static_cast<uint32_t>(list.size()),
    [jEnv, &list](uint32_t index) { return GetJavaInstance_SocialFriend(jEnv, list[index]); });

  return jArray;
}

jobject ttv::binding::java::GetJavaInstance_SocialFriendRequest(JNIEnv* jEnv, const FriendRequest& val) {
  JavaClassInfo& info = GetJavaClassInfo_SocialFriendRequest(jEnv);

  jobject jInstance = jEnv->NewObject(info.klass, info.methods["<init>"]);

  AUTO_DELETE_LOCAL_REF(jEnv, jobject, jUserInfo, GetJavaInstance_UserInfo(jEnv, val.userInfo));
  jEnv->SetObjectField(jInstance, info.fields["userInfo"], jUserInfo);

  jEnv->SetIntField(jInstance, info.fields["requestTime"], static_cast<jint>(val.requestTime));

  return jInstance;
}

jobject ttv::binding::java::GetJavaInstance_SocialFriendRequestArray(
  JNIEnv* jEnv, const std::vector<FriendRequest>& list) {
  JavaClassInfo& socialFriendRequestInfo = GetJavaClassInfo_SocialFriendRequest(jEnv);

  jobjectArray jArray = GetJavaInstance_Array(jEnv, socialFriendRequestInfo, static_cast<uint32_t>(list.size()),
    [jEnv, &list](uint32_t index) { return GetJavaInstance_SocialFriendRequest(jEnv, list[index]); });

  return jArray;
}

void ttv::binding::java::GetNativeInstance_SocialPresenceSettings(
  JNIEnv* jEnv, jobject jSettings, PresenceSettings& settings) {
  JavaClassInfo& info = GetJavaClassInfo_SocialPresenceSettings(jEnv);

  jobject jAvailabilityOverride = jEnv->GetObjectField(jSettings, info.fields["availabilityOverride"]);
  settings.availabilityOverride =
    GetNativeFromJava_SimpleEnum(jEnv, GetJavaClassInfo_SocialPresenceAvailabilityOverride(jEnv), jAvailabilityOverride,
      PresenceSettings::AvailabilityOverride::None);
  settings.shareActivity = jEnv->GetBooleanField(jSettings, info.fields["shareActivity"]) == JNI_TRUE;
}

void ttv::binding::java::GetNativeInstance_SocialFeatureFlags(JNIEnv* jEnv, jobject jFeatures, FeatureFlags& features) {
  JavaClassInfo& info = GetJavaClassInfo_SocialFeatureFlags(jEnv);

  features.friendList = jEnv->GetBooleanField(jFeatures, info.fields["friendList"]) == JNI_TRUE;
  features.friendRequests = jEnv->GetBooleanField(jFeatures, info.fields["friendRequests"]) == JNI_TRUE;
  features.presence = jEnv->GetBooleanField(jFeatures, info.fields["presence"]) == JNI_TRUE;
}
