/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/social/internal/pch.h"

#include "twitchsdk/social/generated/jni_socialapi.h"

#include "twitchsdk/core/generated/java_all.h"
#include "twitchsdk/core/generated/jni_all.h"
#include "twitchsdk/core/stringutilities.h"
#include "twitchsdk/social/generated/java_all.h"
#include "twitchsdk/social/java_socialapilistenerproxy.h"
#include "twitchsdk/social/java_socialutil.h"
#include "twitchsdk/social/socialapi.h"
#include "twitchsdk/social/sociallistener.h"

#include <functional>

#define GET_SOCIALAPI_PTR(x) reinterpret_cast<SocialAPI*>(x)

namespace {
using namespace ttv;
using namespace ttv::social;
using namespace ttv::binding::java;

struct SocialApiContext {
  std::shared_ptr<SocialAPI> chatApi;
  std::shared_ptr<JavaSocialAPIListenerProxy> nativeListener;
};

JavaNativeProxyRegistry<SocialAPI, SocialApiContext>
  gSocialApiNativeProxies;  // The mapping of native to Java instances of SocialAPI
}  // namespace

JNIEXPORT jlong JNICALL Java_tv_twitch_social_SocialAPI_CreateNativeInstance(JNIEnv* jEnv, jobject jThis) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  // Force load all types to ensure all classes are loaded by the time we need them.
  LoadAllSocialJavaClassInfo(jEnv);

  std::shared_ptr<SocialApiContext> context = std::make_shared<SocialApiContext>();
  context->chatApi = std::make_shared<SocialAPI>();
  context->nativeListener = std::make_shared<JavaSocialAPIListenerProxy>(jThis);

  gSocialApiNativeProxies.Register(context->chatApi, context, jThis);

  context->chatApi->SetListener(context->nativeListener);

  return reinterpret_cast<jlong>(context->chatApi.get());
}

JNIEXPORT void JNICALL Java_tv_twitch_social_SocialAPI_DisposeNativeInstance(
  JNIEnv* /*jEnv*/, jobject /*jThis*/, jlong jNativePointer) {
  gSocialApiNativeProxies.Unregister(jNativePointer);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_social_SocialAPI_SetCoreApi(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jobject jCoreApi) {
  auto api = GET_SOCIALAPI_PTR(jNativePointer);
  auto coreApi = GetCoreApiInstance(jCoreApi);

  TTV_ErrorCode ec;

  if (coreApi != nullptr) {
    ec = api->SetCoreApi(coreApi);
  } else {
    ec = TTV_EC_INVALID_ARG;
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_social_SocialAPI_SetEnabledFeatures(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jobject jEnabledFeatures) {
  auto api = GET_SOCIALAPI_PTR(jNativePointer);

  FeatureFlags features;
  GetNativeInstance_SocialFeatureFlags(jEnv, jEnabledFeatures, features);

  TTV_ErrorCode ec = api->SetEnabledFeatures(features);

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_social_SocialAPI_SetListener(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jobject jListener) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);

  TTV_JNI_RETURN_ON_NULL(jEnv, jListener, TTV_EC_INVALID_ARG);

  TTV_ErrorCode ec = TTV_EC_SUCCESS;

  auto context = gSocialApiNativeProxies.LookupNativeContext(jNativePointer);
  if (context != nullptr) {
    context->nativeListener->SetListener(jListener);
  } else {
    ec = TTV_EC_NOT_INITIALIZED;
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_social_SocialAPI_GetState(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_SOCIALAPI_PTR(jNativePointer);

  IModule::State state = api->GetState();

  JavaClassInfo& info = GetJavaClassInfo_ModuleState(jEnv);
  return GetJavaInstance_SimpleEnum(jEnv, info, state);
}

JNIEXPORT jstring JNICALL Java_tv_twitch_social_SocialAPI_GetModuleName(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_SOCIALAPI_PTR(jNativePointer);

  std::string name = api->GetModuleName();

  return GetJavaInstance_String(jEnv, name.c_str());
}

JNIEXPORT jobject JNICALL Java_tv_twitch_social_SocialAPI_Initialize(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_SOCIALAPI_PTR(jNativePointer);

  auto callback =
    CreateJavaCallbackWrapper<jobject>(jEnv, jCallback, GetJavaClassInfo_IModule_InitializeCallback(jEnv));

  TTV_ErrorCode ec = api->Initialize([callback](TTV_ErrorCode ec) {
    AUTO_DELETE_LOCAL_REF(
      gActiveJavaEnvironment, jobject, jError, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, ec));
    callback(jError);
  });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_social_SocialAPI_Shutdown(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_SOCIALAPI_PTR(jNativePointer);

  auto callback = CreateJavaCallbackWrapper<jobject>(jEnv, jCallback, GetJavaClassInfo_IModule_ShutdownCallback(jEnv));

  TTV_ErrorCode ec = api->Shutdown([callback](TTV_ErrorCode ec) {
    AUTO_DELETE_LOCAL_REF(
      gActiveJavaEnvironment, jobject, jError, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, ec));
    callback(jError);
  });
  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_social_SocialAPI_Update(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_SOCIALAPI_PTR(jNativePointer);

  TTV_ErrorCode ec = TTV_EC_INVALID_INSTANCE;

  // Keep a reference to the context around
  auto context = gSocialApiNativeProxies.LookupNativeContext(jNativePointer);
  if (context != nullptr) {
    ec = api->Update();
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_social_SocialAPI_SetPresenceSessionAvailability(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jint jUserId, jobject jAvailability) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_SOCIALAPI_PTR(jNativePointer);

  TTV_JNI_RETURN_ON_NULL(jEnv, jAvailability, TTV_EC_INVALID_ARG);

  JavaClassInfo& availabilityInfo = GetJavaClassInfo_SocialPresenceSessionAvailability(jEnv);

  PresenceSessionAvailability availability =
    GetNativeFromJava_SimpleEnum(jEnv, availabilityInfo, jAvailability, PresenceSessionAvailability::Online);

  TTV_ErrorCode ec = api->SetPresenceSessionAvailability(jUserId, availability);

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_social_SocialAPI_AddWatchingActivity(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jint jUserId, jint jChannelId, jobject jTokenResultContainer) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_SOCIALAPI_PTR(jNativePointer);

  TTV_JNI_RETURN_ON_NULL(jEnv, jTokenResultContainer, TTV_EC_INVALID_ARG);

  uint32_t activityToken;

  TTV_ErrorCode ec = api->AddWatchingActivity(jUserId, jChannelId, activityToken);

  if (TTV_SUCCEEDED(ec)) {
    JavaClassInfo& resultInfo = GetJavaClassInfo_ResultContainer(jEnv);

    AUTO_DELETE_LOCAL_REF(jEnv, jobject, jToken, GetJavaInstance_Integer(jEnv, activityToken));
    jEnv->SetObjectField(jTokenResultContainer, resultInfo.fields["result"], jToken);
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_social_SocialAPI_AddPlayingActivity(JNIEnv* jEnv, jobject /*jThis*/,
  jlong jNativePointer, jint jUserId, jint jChannelId, jstring jGameDisplayContext, jobject jTokenResultContainer) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_SOCIALAPI_PTR(jNativePointer);

  TTV_JNI_RETURN_ON_NULL(jEnv, jTokenResultContainer, TTV_EC_INVALID_ARG);

  uint32_t activityToken;

  ScopedJavaUTFStringConverter gameDisplayContext(jEnv, jGameDisplayContext);

  TTV_ErrorCode ec = api->AddPlayingActivity(jUserId, jChannelId, gameDisplayContext.GetNativeString(), activityToken);

  if (TTV_SUCCEEDED(ec)) {
    JavaClassInfo& resultInfo = GetJavaClassInfo_ResultContainer(jEnv);

    AUTO_DELETE_LOCAL_REF(jEnv, jobject, jToken, GetJavaInstance_Integer(jEnv, activityToken));
    jEnv->SetObjectField(jTokenResultContainer, resultInfo.fields["result"], jToken);
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_social_SocialAPI_RemoveActivity(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jint jUserId, jint jActivityToken) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_SOCIALAPI_PTR(jNativePointer);

  TTV_ErrorCode ec = api->RemoveActivity(jUserId, jActivityToken);

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_social_SocialAPI_PostPresence(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jint jUserId, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_SOCIALAPI_PTR(jNativePointer);

  auto callback =
    CreateJavaCallbackWrapper<jobject>(jEnv, jCallback, GetJavaClassInfo_SocialAPI_PostPresenceCallback(jEnv));

  TTV_ErrorCode ec = api->PostPresence(jUserId, [callback](TTV_ErrorCode ec) {
    AUTO_DELETE_LOCAL_REF(
      gActiveJavaEnvironment, jobject, jError, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, ec));
    callback(jError);
  });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_social_SocialAPI_FetchPresenceSettings(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jint jUserId, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_SOCIALAPI_PTR(jNativePointer);

  auto callback = CreateJavaCallbackWrapper<jobject, jobject>(
    jEnv, jCallback, GetJavaClassInfo_SocialAPI_FetchPresenceSettingsCallback(jEnv));
  TTV_ErrorCode ec =
    api->FetchPresenceSettings(jUserId, [callback](TTV_ErrorCode ec, const PresenceSettings& settings) {
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jError, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, ec));
      AUTO_DELETE_LOCAL_REF(gActiveJavaEnvironment, jobject, jSettings,
        (TTV_SUCCEEDED(ec) ? GetJavaInstance_SocialPresenceSettings(gActiveJavaEnvironment, settings) : nullptr));

      callback(jError, jSettings);
    });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_social_SocialAPI_SetPresenceSettings(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jint jUserId, jobject jSettings, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_SOCIALAPI_PTR(jNativePointer);

  TTV_JNI_RETURN_ON_NULL(jEnv, jSettings, TTV_EC_INVALID_ARG);

  PresenceSettings settings;
  GetNativeInstance_SocialPresenceSettings(jEnv, jSettings, settings);

  auto callback =
    CreateJavaCallbackWrapper<jobject>(jEnv, jCallback, GetJavaClassInfo_SocialAPI_SetPresenceSettingsCallback(jEnv));

  TTV_ErrorCode ec = api->SetPresenceSettings(jUserId, settings, [callback](TTV_ErrorCode ec) {
    AUTO_DELETE_LOCAL_REF(
      gActiveJavaEnvironment, jobject, jError, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, ec));

    callback(jError);
  });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_social_SocialAPI_SetAutomaticPresencePostingEnabled(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jint jUserId, jboolean jEnabled) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_SOCIALAPI_PTR(jNativePointer);

  TTV_ErrorCode ec = api->SetAutomaticPresencePostingEnabled(jUserId, jEnabled == JNI_TRUE);

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_social_SocialAPI_GetAutomaticPresencePostingEnabled(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jint jUserId, jobject jResultContainer) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_SOCIALAPI_PTR(jNativePointer);

  TTV_JNI_RETURN_ON_NULL(jEnv, jResultContainer, TTV_EC_INVALID_ARG);

  bool enabled = false;
  TTV_ErrorCode ec = api->GetAutomaticPresencePostingEnabled(jUserId, enabled);

  if (TTV_SUCCEEDED(ec)) {
    JavaClassInfo& resultContainerInfo = GetJavaClassInfo_ResultContainer(jEnv);

    AUTO_DELETE_LOCAL_REF(gActiveJavaEnvironment, jobject, jBoolean, GetJavaInstance_Boolean(jEnv, enabled));

    gActiveJavaEnvironment->SetObjectField(jResultContainer, resultContainerInfo.fields["result"], jBoolean);
  }

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_social_SocialAPI_FetchFriendList(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jint jUserId, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_SOCIALAPI_PTR(jNativePointer);

  auto callback = CreateJavaCallbackWrapper<jobject, jobjectArray>(
    jEnv, jCallback, GetJavaClassInfo_SocialAPI_FetchFriendListCallback(jEnv));

  TTV_ErrorCode ec = api->FetchFriendList(jUserId, [callback](TTV_ErrorCode ec, const std::vector<Friend>& friendList) {
    AUTO_DELETE_LOCAL_REF(
      gActiveJavaEnvironment, jobject, jError, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, ec));
    AUTO_DELETE_LOCAL_REF(gActiveJavaEnvironment, jobjectArray, jFriendList,
      (TTV_SUCCEEDED(ec) ? GetJavaInstance_SocialFriendArray(gActiveJavaEnvironment, friendList) : nullptr));

    callback(jError, jFriendList);
  });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_social_SocialAPI_UpdateFriendship(JNIEnv* jEnv, jobject /*jThis*/,
  jlong jNativePointer, jint jUserId, jint jOtherUserId, jobject jSocialUpdateFriendAction, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_SOCIALAPI_PTR(jNativePointer);

  TTV_JNI_RETURN_ON_NULL(jEnv, jSocialUpdateFriendAction, TTV_EC_INVALID_ARG);

  FriendAction action = GetNativeFromJava_SimpleEnum<FriendAction>(
    jEnv, GetJavaClassInfo_SocialUpdateFriendAction(jEnv), jSocialUpdateFriendAction, FriendAction::SendRequest);

  auto callback = CreateJavaCallbackWrapper<jobject, jobject, jobject>(
    jEnv, jCallback, GetJavaClassInfo_SocialAPI_UpdateFriendshipCallback(jEnv));
  TTV_ErrorCode ec = api->UpdateFriendship(
    jUserId, jOtherUserId, action, [callback](TTV_ErrorCode ec, UpdateFriendResult result, FriendStatus status) {
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jError, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, ec));

      jobject jResult = nullptr;
      jobject jStatus = nullptr;
      if (TTV_SUCCEEDED(ec)) {
        jResult = GetJavaInstance_SimpleEnum(
          gActiveJavaEnvironment, GetJavaClassInfo_SocialUpdateFriendResult(gActiveJavaEnvironment), result);
        jStatus = GetJavaInstance_SimpleEnum(
          gActiveJavaEnvironment, GetJavaClassInfo_SocialFriendStatus(gActiveJavaEnvironment), status);
      }
      AUTO_DELETE_LOCAL_REF_NO_DECLARE(gActiveJavaEnvironment, jobject, jResult);
      AUTO_DELETE_LOCAL_REF_NO_DECLARE(gActiveJavaEnvironment, jobject, jStatus);

      callback(jError, jResult, jStatus);
    });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_social_SocialAPI_FetchFriendRequests(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jint jUserId, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_SOCIALAPI_PTR(jNativePointer);

  auto callback = CreateJavaCallbackWrapper<jobject, jobjectArray>(
    jEnv, jCallback, GetJavaClassInfo_SocialAPI_FetchFriendRequestsCallback(jEnv));

  TTV_ErrorCode ec =
    api->FetchFriendRequests(jUserId, [callback](TTV_ErrorCode ec, const std::vector<FriendRequest>& requests) {
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jError, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, ec));
      AUTO_DELETE_LOCAL_REF(gActiveJavaEnvironment, jobjectArray, jRequests,
        (TTV_SUCCEEDED(ec) ? GetJavaInstance_SocialFriendRequestArray(gActiveJavaEnvironment, requests) : nullptr));

      callback(jError, jRequests);
    });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_social_SocialAPI_FetchUnreadFriendRequestCount(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jint jUserId, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_SOCIALAPI_PTR(jNativePointer);

  auto callback = CreateJavaCallbackWrapper<jobject, jint>(
    jEnv, jCallback, GetJavaClassInfo_SocialAPI_FetchUnreadFriendRequestCountCallback(jEnv));

  TTV_ErrorCode ec = api->FetchUnreadFriendRequestCount(jUserId, [callback](TTV_ErrorCode ec, uint32_t count) {
    AUTO_DELETE_LOCAL_REF(
      gActiveJavaEnvironment, jobject, jError, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, ec));

    callback(jError, static_cast<jint>(count));
  });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_social_SocialAPI_MarkAllFriendRequestsRead(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jint jUserId, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_SOCIALAPI_PTR(jNativePointer);

  auto callback = CreateJavaCallbackWrapper<jobject>(
    jEnv, jCallback, GetJavaClassInfo_SocialAPI_MarkAllFriendRequestsReadCallback(jEnv));

  TTV_ErrorCode ec = api->MarkAllFriendRequestsRead(jUserId, [callback](TTV_ErrorCode ec) {
    AUTO_DELETE_LOCAL_REF(
      gActiveJavaEnvironment, jobject, jError, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, ec));

    callback(jError);
  });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_social_SocialAPI_FetchRecommendedFriends(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jint jUserId, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_SOCIALAPI_PTR(jNativePointer);

  auto callback = CreateJavaCallbackWrapper<jobject, jobjectArray>(
    jEnv, jCallback, GetJavaClassInfo_SocialAPI_FetchRecommendedFriendsCallback(jEnv));

  TTV_ErrorCode ec =
    api->FetchRecommendedFriends(jUserId, [callback](TTV_ErrorCode ec, const std::vector<UserInfo>& recommendations) {
      AUTO_DELETE_LOCAL_REF(
        gActiveJavaEnvironment, jobject, jError, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, ec));
      AUTO_DELETE_LOCAL_REF(gActiveJavaEnvironment, jobjectArray, jRecommendations,
        (TTV_SUCCEEDED(ec) ? GetJavaInstance_UserInfoArray(gActiveJavaEnvironment, recommendations) : nullptr));

      callback(jError, jRecommendations);
    });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_social_SocialAPI_DismissRecommendedFriend(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jint jUserId, jint jDismissUserId, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_SOCIALAPI_PTR(jNativePointer);

  auto callback = CreateJavaCallbackWrapper<jobject>(
    jEnv, jCallback, GetJavaClassInfo_SocialAPI_DismissRecommendedFriendCallback(jEnv));

  TTV_ErrorCode ec = api->DismissRecommendedFriend(jUserId, jDismissUserId, [callback](TTV_ErrorCode ec) {
    AUTO_DELETE_LOCAL_REF(
      gActiveJavaEnvironment, jobject, jError, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, ec));

    callback(jError);
  });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}

JNIEXPORT jobject JNICALL Java_tv_twitch_social_SocialAPI_FetchFriendStatus(
  JNIEnv* jEnv, jobject /*jThis*/, jlong jNativePointer, jint jUserId, jint jOtherUserId, jobject jCallback) {
  ScopedJavaEnvironmentCacher jEnvCacher(jEnv);
  auto api = GET_SOCIALAPI_PTR(jNativePointer);

  auto callback = CreateJavaCallbackWrapper<jobject, jobject>(
    jEnv, jCallback, GetJavaClassInfo_SocialAPI_FetchFriendStatusCallback(jEnv));

  TTV_ErrorCode ec = api->FetchFriendStatus(jUserId, jOtherUserId, [callback](TTV_ErrorCode ec, FriendStatus status) {
    AUTO_DELETE_LOCAL_REF(
      gActiveJavaEnvironment, jobject, jError, GetJavaInstance_ErrorCode(gActiveJavaEnvironment, ec));
    AUTO_DELETE_LOCAL_REF(gActiveJavaEnvironment, jobject, jStatus,
      (TTV_SUCCEEDED(ec) ? GetJavaInstance_SimpleEnum(gActiveJavaEnvironment,
                             GetJavaClassInfo_SocialFriendStatus(gActiveJavaEnvironment), status)
                         : nullptr));

    callback(jError, jStatus);
  });

  return GetJavaInstance_ErrorCode(jEnv, ec);
}
