package tv.twitch.test;

import java.util.*;

import tv.twitch.*;
import tv.twitch.social.*;

public class SocialTest extends TestBase {
    public SocialTest(Library library) {
        super(library);

        SocialErrorCode.forceClassInit();
    }

    @Override
    public void run() throws Exception {
        System.out.println("Running SocialTest tests...");

        // Interfaces
        test_ISocialAPIListener();

        // SocialAPI
        test_SocialAPI();

        System.out.println("Done running SocialTest tests...");
    }

    private native void Test_ISocialAPIListener(IModule module, ISocialAPIListener listener);

    protected void test_ISocialAPIListener() throws Exception {
        final HashSet<String> calls = new HashSet<>();

        ISocialAPIListener socialApiListener = new ISocialAPIListener() {
            @Override
            public void moduleStateChanged(IModule source, ModuleState state, ErrorCode result) {
                calls.add(getCurrentMethodName());
            }

            @Override
            public void socialUnreadFriendRequestCountChanged(int userId, int count) {
                calls.add(getCurrentMethodName());
            }

            @Override
            public void socialRealtimeFriendRequestReceived(int userId, SocialFriendRequest request) {
                calls.add(getCurrentMethodName());
            }

            @Override
            public void socialPresenceSettingsChanged(int userId, SocialPresenceSettings settings) {
                calls.add(getCurrentMethodName());
            }

            @Override
            public void socialFriendshipChanged(int userId, SocialFriend[] added, SocialFriend[] removed) {
                calls.add(getCurrentMethodName());
            }

            @Override
            public void socialFriendRequestRemoved(
                int userId, int otherUserId, SocialFriendRequestRemovedReason reason) {
                calls.add(getCurrentMethodName());
            }

            @Override
            public void socialFriendInfoChanged(int userId, SocialFriend[] changes) {
                calls.add(getCurrentMethodName());
            }
        };

        // Ask native to call each method on our interface
        Test_ISocialAPIListener(mDummyModule, socialApiListener);

        // Make sure they were all called
        checkAllMethodsCalled(ISocialAPIListener.class, calls);
    }

    protected void test_SocialAPI() throws Exception {
        CoreAPI coreApi = null;
        SocialAPI socialApi = null;
        try {
            coreApi = new CoreAPI();
            addModule(coreApi);

            ErrorCode ec = coreApi.initialize(null);
            checkError(ec);

            while (coreApi.getState() != ModuleState.Initialized) {
                updateModules();
            }

            socialApi = new SocialAPI();
            addModule(socialApi);

            ec = socialApi.setCoreApi(coreApi);

            SocialFeatureFlags flags = new SocialFeatureFlags();

            flags.friendList = true;
            flags.friendRequests = true;
            flags.presence = true;

            ec = socialApi.setEnabledFeatures(flags);

            ec = socialApi.initialize(null);
            checkError(ec);

            while (socialApi.getState() != ModuleState.Initialized) {
                Thread.sleep(250);
                updateModules();
            }

            final ResultContainer<Boolean> callbackCalled = new ResultContainer<Boolean>();

            // Log in a user
            if (m_OauthToken != null) {
                callbackCalled.result = false;

                ec = coreApi.logIn(m_OauthToken, new CoreAPI.LogInCallback() {
                    @Override
                    public void invoke(ErrorCode ec, UserInfo userInfo) {
                        callbackCalled.result = true;
                        if (ec.succeeded()) {
                            System.out.println("Logged in successfully");
                        }
                    }
                });

                if (ec.succeeded()) {
                    while (!callbackCalled.result) {
                        Thread.sleep(100);
                        updateModules();
                    }
                }
            }

            ResultContainer<SocialPresenceSettings> settingsResult = new ResultContainer<>();
            callbackCalled.result = false;
            ec = socialApi.fetchPresenceSettings(m_UserId, new SocialAPI.FetchPresenceSettingsCallback() {
                @Override
                public void invoke(ErrorCode ec, SocialPresenceSettings settings) {
                    callbackCalled.result = true;
                    settingsResult.result = settings;
                }
            });

            checkError(ec);
            while (!callbackCalled.result) {
                Thread.sleep(250);
                updateModules();
            }

            SocialPresenceSettings settings = new SocialPresenceSettings();
            settings.availabilityOverride = SocialPresenceAvailabilityOverride.Away;
            settings.shareActivity = false;

            callbackCalled.result = false;
            ec = socialApi.setPresenceSettings(m_UserId, settings, new SocialAPI.SetPresenceSettingsCallback() {
                @Override
                public void invoke(ErrorCode ec) {
                    callbackCalled.result = true;
                }
            });

            checkError(ec);
            while (!callbackCalled.result) {
                Thread.sleep(250);
                updateModules();
            }

            ResultContainer<Boolean> boolResult = new ResultContainer<>();
            ec = socialApi.getAutomaticPresencePostingEnabled(m_UserId, boolResult);
            checkError(ec);

            ec = socialApi.setAutomaticPresencePostingEnabled(m_UserId, !boolResult.result);
            checkError(ec);

            callbackCalled.result = false;
            ec = socialApi.fetchFriendList(m_UserId, new SocialAPI.FetchFriendListCallback() {
                @Override
                public void invoke(ErrorCode ec, SocialFriend[] friends) {
                    callbackCalled.result = true;
                }
            });

            checkError(ec);
            while (!callbackCalled.result) {
                Thread.sleep(250);
                updateModules();
            }

            callbackCalled.result = false;
            ec = socialApi.fetchFriendRequests(m_UserId, new SocialAPI.FetchFriendRequestsCallback() {
                @Override
                public void invoke(ErrorCode ec, SocialFriendRequest[] requests) {
                    callbackCalled.result = true;
                }
            });

            checkError(ec);
            while (!callbackCalled.result) {
                Thread.sleep(250);
                updateModules();
            }

            callbackCalled.result = false;
            ec = socialApi.postPresence(m_UserId, new SocialAPI.PostPresenceCallback() {
                @Override
                public void invoke(ErrorCode ec) {
                    callbackCalled.result = true;
                }
            });

            checkError(ec);
            while (!callbackCalled.result) {
                Thread.sleep(250);
                updateModules();
            }
        } finally {
            if (socialApi != null) {
                socialApi.shutdown(null);
                while (socialApi.getState() != ModuleState.Uninitialized) {
                    Thread.sleep(250);
                    updateModules();
                }
                socialApi.dispose();
            }
            if (coreApi != null) {
                coreApi.shutdown(null);
                while (coreApi.getState() != ModuleState.Uninitialized) {
                    Thread.sleep(250);
                    updateModules();
                }
                coreApi.dispose();
            }
            clearModules();
        }
    }
}
