/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/core/coreapi.h"
#include "twitchsdk/core/module.h"
#include "twitchsdk/tracking/trackingapilistener.h"
#include "twitchsdk/tracking/trackingerrortypes.h"

#include <chrono>
#include <sstream>

namespace ttv {
class TaskRunner;

namespace tracking {
class TrackingAPI;
}
}  // namespace ttv

/**
 * Provides the main tracking service functionality.
 *
 * The following properties must be set before calling Initialize().
 * - SetCoreAPI()
 * - SetListener()
 * - SetMaxBytesInBatch()
 * - SetFlushIntervalInMs()
 */
class ttv::tracking::TrackingAPI : public ttv::ModuleBase {
 public:
  TrackingAPI();
  ~TrackingAPI() override;

  // IModule implementation
  virtual std::string GetModuleName() const override;
  virtual TTV_ErrorCode Initialize(const InitializeCallback& callback) override;
  virtual TTV_ErrorCode Shutdown(const ShutdownCallback& callback) override;
  virtual TTV_ErrorCode Update() override;

  TTV_ErrorCode SetCoreApi(const std::shared_ptr<CoreAPI>& coreApi);
  TTV_ErrorCode SetMaxBytesInBatch(uint32_t batchSize);
  TTV_ErrorCode SetMaxPendingEvents(uint32_t max);
  TTV_ErrorCode SetFlushIntervalInMs(uint32_t flushInterval);
  TTV_ErrorCode SetListener(const std::shared_ptr<ITrackingAPIListener>& listener);

  /**
   * Retrieves the URL used to send tracking events to the backend.
   */
  std::string GetSpadeUrl() const;

  /**
   * Hold tracking events from flushing on the timer interval.
   * Events will still flush when the batch is full or if FlushEvents() is explicitly called.
   */
  void HoldEvents();
  /**
   * Release and allow tracking events to be flushed on the timer interval.
   */
  void ReleaseEvents();

  /**
   * Enqueues an event for reporting.
   */
  TTV_ErrorCode TrackEvent(const std::string& eventName, const std::map<std::string, TrackingValue>& params);
  /**
   * Explicitly requests that any batched events are flushed and sent.
   */
  TTV_ErrorCode FlushEvents();

 protected:
  // ModuleBase overrides
  virtual bool CheckShutdown() override;
  virtual void CompleteShutdown() override;

 private:
  void RescheduleFailedEvents(uint32_t& numDropped);
  void NotifyEventsDiscarded(uint32_t numDropped);

  class CoreApiClient : public ICoreApiClient {
   public:
    CoreApiClient();

    // ICoreApiClient implementation
    virtual std::string GetClientName();
  };

  std::shared_ptr<CoreAPI> mCoreApi;
  std::unique_ptr<TaskRunner> mTaskRunner;
  std::shared_ptr<IEventScheduler> mMainEventScheduler;
  std::shared_ptr<CoreApiClient> mCoreApiClient;

  std::vector<std::string> mInFlightEvents;
  /**
   * The batch that is currently being filled.
   */
  std::vector<std::string> mPendingEvents;

  /**
   * The max number of bytes allowed in an encoded batch.  This is a value dictated by backend.
   */
  size_t mMaxBytesInBatch;
  /**
   * The max number of events that can be kept around waiting to be sent.
   */
  uint32_t mMaxPendingEvents;
  std::chrono::milliseconds mFlushInterval;

  ttv::LambdaRetryTimer mFlushTimer;
  std::shared_ptr<ITrackingAPIListener> mListener;

  std::unique_ptr<IMutex> mMutex;

  bool mHoldEvents;
};
