#include "context.h"
#include "staticinit.h"
#include "stringutilities.h"
#include "twitchsdk/core/coreapi.h"
#include "twitchsdk/core/corelistener.h"
#include "twitchsdk/core/stringutilities.h"
#include "twitchsdk/tracking/trackingapi.h"

using namespace ttv;
using namespace ttv::tracking;

namespace {
const char* kTraceComponents[] = {
  // TODO: Add some
  "TrackingAPI"};

class TrackingAPIListener;

std::shared_ptr<TrackingAPI> gTrackingApi;
std::shared_ptr<TrackingAPIListener> gTrackingApiListener;
std::shared_ptr<Context> gContext;  // Cached by the static initializer

class TrackingAPIListener : public ITrackingAPIListener {
  virtual void ModuleStateChanged(IModule* /*source*/, IModule::State state, TTV_ErrorCode ec) override {
    std::cout << "ModuleStateChanged: " << ::ToString(state) << " " << ErrorToString(ec) << std::endl;
  }

  virtual void TrackingEventFlushCompleted(TTV_ErrorCode ec, uint32_t numEvents) override {
    std::cout << "TrackingEventFlushCompleted: Completed flushing " << numEvents << " events with EC "
              << ErrorToString(ec) << "\n";
  }

  virtual void TrackingEventsDiscarded(uint32_t numEvents) override {
    std::cout << "TrackingEventsDiscarded: Discarded " << numEvents << " events\n";
  }
};

std::shared_ptr<CommandCategory> RegisterTrackingCommands(std::shared_ptr<Context> /*context*/) {
  std::shared_ptr<CommandCategory> category =
    std::make_shared<CommandCategory>("Tracking", "The ttv::TrackingAPI command interface");

  category->AddCommand("TrackingInit")
    .AddFunction()
    .Description("Calls TrackingAPI::Initialize")
    .Function([](const std::vector<ParamValue> & /*params*/) -> bool {
      TTV_ErrorCode ec = gTrackingApi->Initialize([](TTV_ErrorCode ec) {
        std::cout << "TrackingInit completed with error:";
        ReportCommandResult(ec);
      });
      ReportCommandResult(ec);
      return true;
    });

  category->AddCommand("TrackEvent")
    .AddFunction()
    .Description(
      "Calls TrackingAPI::TrackEvent. Params after event-name are space-delimited, and treated as consecutive key-value pairs")
    .AddParam("event-name", ParamType::String)
    .AddParam("[{property-name} {property-value}]", ParamType::RemainderAsString)
    .Function([](const std::vector<ParamValue>& params) -> bool {
      std::map<std::string, TrackingValue> properties;
      std::string args = params[1].GetString();
      std::vector<std::string> tokens = ttv::ParseArguments(params[1].GetString());

      for (unsigned int i = 0; i < tokens.size() && i + 1 < tokens.size(); i += 2) {
        properties[tokens[i]] = tokens[i + 1];
      }

      TTV_ErrorCode ec = gTrackingApi->TrackEvent(params[0], properties);
      ReportCommandResult(ec);
      std::cout << "Track Event" << std::endl;
      return true;
    });

  return category;
}

StaticInitializer gStaticInitializer([]() {
  StaticInitializationRegistrar::GetInstance().RegisterInitializer(
    [](std::shared_ptr<Context> context) {
      context->RegisterLoggers(kTraceComponents, sizeof(kTraceComponents) / sizeof(kTraceComponents[0]));

      gContext = context;

      auto coreApi = std::static_pointer_cast<CoreAPI>(context->GetModule("ttv::CoreAPI"));
      gTrackingApi = std::make_shared<TrackingAPI>();
      gTrackingApi->SetCoreApi(coreApi);
      context->RegisterModule(gTrackingApi);

      gTrackingApiListener = std::make_shared<TrackingAPIListener>();
      gTrackingApi->SetListener(gTrackingApiListener);

      auto commands = RegisterTrackingCommands(context);
      context->RegisterCommandCategory(commands);
    },
    0);
});
}  // namespace
