/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/tracking/internal/pch.h"

#include "twitchsdk/tracking/trackingapi.h"

#include "twitchsdk/core/component.h"
#include "twitchsdk/core/module.h"
#include "twitchsdk/core/mutex.h"
#include "twitchsdk/core/settingrepository.h"
#include "twitchsdk/core/stringutilities.h"
#include "twitchsdk/core/systemclock.h"
#include "twitchsdk/core/task/task.h"
#include "twitchsdk/core/task/taskrunner.h"
#include "twitchsdk/core/thread.h"
#include "twitchsdk/core/timer.h"
#include "twitchsdk/core/types/memorytypes.h"
#include "twitchsdk/core/user/user.h"
#include "twitchsdk/core/user/userrepository.h"
#include "twitchsdk/tracking/internal/task/trackingrequesttask.h"

#include <ios>
#include <sstream>

namespace {
using namespace ttv;
using namespace ttv::tracking;

constexpr const char* const kSpadeUrl = "https://spade.twitch.tv/track/";
constexpr const char* const kModuleName = "ttv::tracking::TrackingAPI";

const size_t kDefaultMaxBytesInBatch = 1024 * 500;  // 500 KB, limit for Spade.
const uint32_t kDefaultMaxPendingEvents = 1024;
const std::chrono::seconds kDefaultFlushInterval = std::chrono::seconds(5);

size_t Base64ExpansionSize(size_t size) {
  // 4/3 for B64 expansion factor, +1 for floor
  return (size * 4 / 3) + 1;
}

size_t Base64ExpansionSizeWithArrayMarkup(size_t size) {
  // +2 for the json array []
  return Base64ExpansionSize(size + 2);
}
}  // namespace

ttv::tracking::TrackingAPI::TrackingAPI()
    : mMaxBytesInBatch(kDefaultMaxBytesInBatch),
      mMaxPendingEvents(kDefaultMaxPendingEvents),
      mFlushInterval(kDefaultFlushInterval),
      mHoldEvents(false) {
  // Register the error lookup function for the module
  ttv::RegisterErrorToStringFunction(&ttv::tracking::TrackingErrorToString);
  ttv::RegisterErrorCodeValueFunction(&ttv::tracking::GetTrackingErrorCodeValues);

  TTV_ErrorCode ec = ttv::CreateMutex(mMutex, "TrackingAPI");
  TTV_ASSERT(TTV_SUCCEEDED(ec) && mMutex != nullptr);
}

ttv::tracking::TrackingAPI::~TrackingAPI() = default;

std::string ttv::tracking::TrackingAPI::GetSpadeUrl() const {
  return kSpadeUrl;
}

std::string ttv::tracking::TrackingAPI::GetModuleName() const {
  return kModuleName;
}

TTV_ErrorCode ttv::tracking::TrackingAPI::Initialize(const InitializeCallback& callback) {
  TTV_RETURN_ON_NULL(mCoreApi, TTV_EC_NOT_INITIALIZED);

  TTV_ErrorCode ec = ModuleBase::Initialize(callback);

  if (TTV_SUCCEEDED(ec)) {
    mState = State::Initializing;

    mCoreApiClient = std::make_shared<CoreApiClient>();
    mMainEventScheduler = mCoreApi->GetMainEventScheduler();
    mTaskRunner = std::make_unique<TaskRunner>("TrackingAPI");

    mFlushTimer.SetEventScheduler(mMainEventScheduler);
    mFlushTimer.SetCallback([this]() {
      if (!mHoldEvents) {
        FlushEvents();
      }
      // NOTE: If the timer goes off while holding events we don't set it again until
      // the event hold is released.
    });
  }

  ec = mCoreApi->RegisterClient(mCoreApiClient);
  if (TTV_SUCCEEDED(ec)) {
    NotifyStateChange();
    RegisterInitializeCallback(callback);
  } else {
    CompleteShutdown();
  }

  return ec;
}

TTV_ErrorCode ttv::tracking::TrackingAPI::Shutdown(const ShutdownCallback& callback) {
  TTV_ErrorCode ec = ModuleBase::Shutdown(callback);
  if (TTV_SUCCEEDED(ec)) {
    RegisterShutdownCallback(callback);
  }
  return ec;
}

TTV_ErrorCode ttv::tracking::TrackingAPI::Update() {
  if (mState == State::Uninitialized) {
    return TTV_EC_SUCCESS;
  }

  ModuleBase::Update();

  if (mTaskRunner != nullptr) {
    mTaskRunner->PollTasks();
  }

  switch (mState) {
    case State::Initializing: {
      mState = State::Initialized;
      NotifyStateChange();

      break;
    }
    default: { break; }
  }

  return TTV_EC_SUCCESS;
}

void ttv::tracking::TrackingAPI::HoldEvents() {
  if (mHoldEvents) {
    return;
  }

  mHoldEvents = true;

  // NOTE: Leave the timer ticking since events might be released before the timer fires.
}

void ttv::tracking::TrackingAPI::ReleaseEvents() {
  if (!mHoldEvents) {
    return;
  }

  mHoldEvents = false;

  // Either there are no events pending or the timer had gone off while holding events so flush now.
  // It is safe to flush events in either case.
  if (!mFlushTimer.IsSet()) {
    FlushEvents();
  } else {
    // The timer is still ticking so wait until it fires to flush events.
  }
}

TTV_ErrorCode ttv::tracking::TrackingAPI::TrackEvent(
  const std::string& eventName, const std::map<std::string, TrackingValue>& properties) {
  AutoMutex am(mMutex.get());

  switch (mState) {
    case State::ShuttingDown:
      return TTV_EC_SHUTTING_DOWN;
    case State::Initializing:
    case State::Uninitialized:
      return TTV_EC_NOT_INITIALIZED;
    case State::Initialized:
      break;
  }

  // Transform into json
  json::Value trackingEvent;
  json::Value trackingEventProperties;

  trackingEvent["event"] = eventName;

  if (properties.find("time") == properties.end()) {
    trackingEventProperties["time"] = GetCurrentTimeAsUnixTimestamp();
  }

  for (const auto& entry : properties) {
    json::Value trackingValue;
    switch (entry.second.GetType()) {
      case TrackingValue::Type::Null:
        // It's already null.
        break;
      case TrackingValue::Type::Boolean:
        trackingValue = entry.second.GetBooleanValue();
        break;
      case TrackingValue::Type::Integer:
        trackingValue = entry.second.GetIntegerValue();
        break;
      case TrackingValue::Type::Double:
        trackingValue = entry.second.GetDoubleValue();
        break;
      case TrackingValue::Type::String:
        trackingValue = entry.second.GetStringValue();
        break;
    }
    trackingEventProperties[entry.first] = trackingValue;
  }

  trackingEvent["properties"] = trackingEventProperties;

  // Serialize
  json::FastWriter writer;
  std::string serializedEvent = writer.write(trackingEvent);

  // Make sure the single event is not too big - should never happen in practice
  size_t encodedEventSize = Base64ExpansionSizeWithArrayMarkup(serializedEvent.size());
  TTV_ASSERT(encodedEventSize <= mMaxBytesInBatch);
  if (encodedEventSize > mMaxBytesInBatch) {
    // Just discard the event
    return TTV_EC_INVALID_JSON;
  }

  // NOTE: We let the pending event list grow larger than the max number configured but we'll throw them out if they
  // fail to send
  mPendingEvents.emplace_back(std::move(serializedEvent));

  // Start the timer if needed
  if (!mFlushTimer.IsSet() && mInFlightEvents.empty()) {
    mFlushTimer.Start(static_cast<uint64_t>(mFlushInterval.count()));
  }

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::tracking::TrackingAPI::FlushEvents() {
  std::shared_ptr<TrackingRequestTask> spadeTask;
  uint32_t batchSize = 0;

  {
    AutoMutex am(mMutex.get());

    switch (mState) {
      case State::ShuttingDown:
        return TTV_EC_SHUTTING_DOWN;
      case State::Initializing:
      case State::Uninitialized:
        return TTV_EC_NOT_INITIALIZED;
      case State::Initialized:
        break;
    }

    auto completeCallback = [this](TTV_ErrorCode ec) {
      // Start the timer if needed
      if (!mFlushTimer.IsSet() && mInFlightEvents.empty()) {
        if (TTV_SUCCEEDED(ec)) {
          mFlushTimer.Start(static_cast<uint64_t>(mFlushInterval.count()));
        } else {
          mFlushTimer.StartBackoff();
        }
      }
    };

    // We only send out one batch at a time
    if (!mInFlightEvents.empty()) {
      completeCallback(TTV_EC_SUCCESS);

      return TTV_EC_REQUEST_PENDING;
    }

    // Nothing to send
    if (mPendingEvents.empty()) {
      completeCallback(TTV_EC_SUCCESS);

      return TTV_EC_SUCCESS;
    }

    // Serialize and encode until we reach the max that can be sent at once
    std::stringstream stream;
    stream << "[";

    size_t totalSerializedSize = 0;

    // Accumulate the encoded data until our max batch size is reached
    auto iter = mPendingEvents.begin();
    for (; iter != mPendingEvents.end(); ++iter) {
      const auto& serializedEvent = *iter;

      // Number of bytes if we were to immediately flush the event queue after adding serializedEvent.
      size_t bytesToFlush =
        Base64ExpansionSizeWithArrayMarkup(serializedEvent.size() + totalSerializedSize + 1);  // +1 for the ,

      // Batch full
      if (bytesToFlush > mMaxBytesInBatch) {
        break;
      }

      totalSerializedSize += serializedEvent.size();

      // Append to the buffer
      if (iter != mPendingEvents.begin()) {
        stream << ",";
      }

      stream << serializedEvent;
    }

    // Terminate the serialized stream
    stream << "]";

    // Add the events to the in-flight list and remove from pending
    std::move(mPendingEvents.begin(), iter, std::back_inserter(mInFlightEvents));
    mPendingEvents.erase(mPendingEvents.begin(), iter);

    batchSize = static_cast<uint32_t>(mInFlightEvents.size());

    std::string finalJson = stream.str();

// Make sure we have serialized correctly
#if defined(_DEBUG)
    {
      json::Reader jsonReader;
      json::Value jsonVal;
      bool parsed = jsonReader.parse(finalJson.data(), finalJson.data() + finalJson.size(), jsonVal);
      TTV_ASSERT(parsed);
    }
#endif

    TrackingRequestTask::Callback callback = [this, completeCallback, batchSize](TrackingRequestTask* /*source*/,
                                               const std::string& /*serverUrl*/, TTV_ErrorCode ec) {
      uint32_t numDropped = 0;

      {
        AutoMutex scope_am(mMutex.get());

        // Send successful
        if (TTV_SUCCEEDED(ec)) {
          mFlushTimer.ResetBackoff();
        }
        // Handle errors
        else {
          // Failed to reach the server, possibly offline so save events for later
          if (ec == TTV_EC_REQUEST_TIMEDOUT) {
            RescheduleFailedEvents(numDropped);
          }
          // We got an HTTP status that is 3XX or 4XX so discard the events now
          else {
            numDropped = batchSize;
          }
        }

        mInFlightEvents.clear();
      }

      // Pass along the success/failure of the request, mainly for logging purposes
      Invoke<ITrackingAPIListener>([batchSize, ec](const std::shared_ptr<ITrackingAPIListener>& listener) {
        listener->TrackingEventFlushCompleted(ec, batchSize);
      });

      // Notify that some events were discarded
      if (numDropped > 0) {
        NotifyEventsDiscarded(numDropped);
      }

      {
        AutoMutex scope_am(mMutex.get());

        completeCallback(ec);
      }
    };

    spadeTask = std::make_shared<TrackingRequestTask>(kSpadeUrl, finalJson, std::move(callback));
  }

  if (mTaskRunner->AddTask(spadeTask)) {
    return TTV_EC_SUCCESS;
  } else {
    // The only way this should happen is if the module is shutting down
    NotifyEventsDiscarded(batchSize);

    return TTV_EC_UNKNOWN_ERROR;
  }
}

void ttv::tracking::TrackingAPI::RescheduleFailedEvents(uint32_t& numDropped) {
  numDropped = 0;

  // Move the failed events back into the pending list
  mPendingEvents.insert(mPendingEvents.begin(), mInFlightEvents.begin(), mInFlightEvents.end());
  mInFlightEvents.clear();

  // Too many events are queued up, drop some and notify
  if (static_cast<uint32_t>(mPendingEvents.size()) > mMaxPendingEvents) {
    // Drop the oldest events
    numDropped = static_cast<uint32_t>(mPendingEvents.size()) - mMaxPendingEvents;
    mPendingEvents.erase(mPendingEvents.begin(), mPendingEvents.begin() + static_cast<int>(numDropped));
  }
}

void ttv::tracking::TrackingAPI::NotifyEventsDiscarded(uint32_t numDropped) {
  Invoke<ITrackingAPIListener>([numDropped](const std::shared_ptr<ITrackingAPIListener>& listener) {
    listener->TrackingEventsDiscarded(numDropped);
  });
}

TTV_ErrorCode ttv::tracking::TrackingAPI::SetCoreApi(const std::shared_ptr<CoreAPI>& coreApi) {
  if (mState == IModule::State::Uninitialized) {
    mCoreApi = coreApi;
    return TTV_EC_SUCCESS;
  } else {
    return TTV_EC_ALREADY_INITIALIZED;
  }
}

TTV_ErrorCode ttv::tracking::TrackingAPI::SetMaxBytesInBatch(uint32_t batchSize) {
  mMaxBytesInBatch = static_cast<size_t>(batchSize);
  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::tracking::TrackingAPI::SetMaxPendingEvents(uint32_t max) {
  mMaxPendingEvents = max;
  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::tracking::TrackingAPI::SetFlushIntervalInMs(uint32_t flushInterval) {
  mFlushInterval = std::chrono::milliseconds(flushInterval);
  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::tracking::TrackingAPI::SetListener(const std::shared_ptr<ITrackingAPIListener>& listener) {
  if (mState != State::Uninitialized) {
    return TTV_EC_ALREADY_INITIALIZED;
  }

  mListeners.ClearListeners();

  if (listener != nullptr) {
    mListeners.AddListener(listener);
  }

  return TTV_EC_SUCCESS;
}

bool ttv::tracking::TrackingAPI::CheckShutdown() {
  if (!ModuleBase::CheckShutdown()) {
    return false;
  }

  if (!mTaskRunner->IsShutdown()) {
    mTaskRunner->Shutdown();
    return false;
  }

  return true;
}

void ttv::tracking::TrackingAPI::CompleteShutdown() {
  if (!mPendingEvents.empty()) {
    uint32_t numDropped = static_cast<uint32_t>(mPendingEvents.size());
    mPendingEvents.clear();

    Invoke<ITrackingAPIListener>([numDropped](const std::shared_ptr<ITrackingAPIListener>& listener) {
      listener->TrackingEventsDiscarded(numDropped);
    });
  }

  if (mCoreApiClient != nullptr) {
    if (mCoreApi != nullptr) {
      mCoreApi->UnregisterClient(mCoreApiClient);
    }

    mCoreApiClient.reset();
  }

  if (mTaskRunner != nullptr) {
    mTaskRunner->CompleteShutdown();
    mTaskRunner.reset();
  }

  TTV_ASSERT(mInFlightEvents.empty());

  mCoreApi.reset();

  ModuleBase::CompleteShutdown();
}

ttv::tracking::TrackingAPI::CoreApiClient::CoreApiClient() {}

std::string ttv::tracking::TrackingAPI::CoreApiClient::GetClientName() {
  return kModuleName;
}
