/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "fixtures/trackingapitest.h"

#include "testhttprequest.h"
#include "testutilities.h"
#include "trackingapitestmanager.h"
#include "twitchsdk/core/coreapi.h"
#include "twitchsdk/core/httprequest.h"
#include "twitchsdk/core/module.h"
#include "twitchsdk/core/pubsub/pubsubclient.h"
#include "twitchsdk/core/task/taskrunner.h"
#include "twitchsdk/core/thread.h"
#include "twitchsdk/core/user/user.h"
#include "twitchsdk/core/user/userrepository.h"
#include "twitchsdk/tracking/trackingapi.h"

#include <thread>

#include "gtest/gtest.h"

namespace {
using namespace ttv;
using namespace ttv::tracking;
}  // namespace

TrackingAPITest::TrackingAPITest()
    : mCoreApi(std::make_shared<CoreAPI>()), mTrackingApi(std::make_shared<TrackingAPI>()) {
  mTrackingApi->SetFlushIntervalInMs(kTestingFlushIntervalInMs);
  mTrackingApi->SetMaxBytesInBatch(kTestingMaxBytesInBatch);

  mTrackingApiTestManager = std::make_shared<TrackingAPITestManager>();
  mTrackingApi->SetListener(mTrackingApiTestManager);
}

TrackingAPITest::~TrackingAPITest() {
  std::vector<std::shared_ptr<IModule>> modules;

  if (mTrackingApi != nullptr) {
    mTrackingApi->SetListener(nullptr);
    modules.push_back(mTrackingApi);
  }

  if (mCoreApi != nullptr) {
    modules.push_back(mCoreApi);
  }

  ttv::ShutdownModulesSync(modules);
}

void TrackingAPITest::SetUpStubs() {
  SdkBaseTest::SetUpStubs();
}

void TrackingAPITest::SetUpComponents() {
  SdkBaseTest::SetUpComponents();

  ttv::test::InitializeModule(mCoreApi);

  mTrackingApi->SetCoreApi(mCoreApi);

  ttv::test::InitializeModule(mTrackingApi);

  AddModule(mCoreApi);
  AddModule(mTrackingApi);
}

void TrackingAPITest::TearDownComponents() {
  ShutdownModulesSync({mTrackingApi, mCoreApi});

  SdkBaseTest::TearDownComponents();
}

void TrackingAPITest::UpdateUntilEventFlushed(uint32_t maxWaitInMs, uint32_t eventId) {
  uint32_t batchStartId = 0;

  auto searchPredicate = [eventId, &batchStartId](const TrackingBatch& batch) {
    bool inBatchRange = eventId >= batchStartId && eventId < batchStartId + batch.numEvents;

    batchStartId += batch.numEvents;

    return inBatchRange;
  };

  auto checkFunction = [this, &searchPredicate, &batchStartId]() {
    batchStartId = 0;

    return std::find_if(mTrackingApiTestManager->mCompletedBatches.begin(),
             mTrackingApiTestManager->mCompletedBatches.end(),
             searchPredicate) != mTrackingApiTestManager->mCompletedBatches.end();
  };

  auto pollFunction = [this]() { Update(); };

  ttv::test::WaitUntilResultWithPollTask(maxWaitInMs, checkFunction, pollFunction);
}

bool TrackingAPITest::CheckEventHasStatus(uint32_t eventId, TrackingBatchStatus status) {
  uint32_t batchStartId = 0;

  auto searchPredicate = [eventId, status, &batchStartId](const TrackingBatch& batch) {
    bool inBatchRange = eventId >= batchStartId && eventId < batchStartId + batch.numEvents;
    if (batch.status != TTV_TRACKING_BATCH_FAILED) {
      batchStartId += batch.numEvents;
    }
    return inBatchRange && batch.status == status;
  };

  return std::find_if(mTrackingApiTestManager->mCompletedBatches.begin(),
           mTrackingApiTestManager->mCompletedBatches.end(),
           searchPredicate) != mTrackingApiTestManager->mCompletedBatches.end();
}

void TrackingAPITest::AssertEventSucceeded(uint32_t eventId) {
  ASSERT_EQ(CheckEventHasStatus(eventId, TTV_TRACKING_BATCH_SUCCEEDED), true);
}

void TrackingAPITest::AssertEventNotSucceeded(uint32_t eventId) {
  ASSERT_EQ(CheckEventHasStatus(eventId, TTV_TRACKING_BATCH_SUCCEEDED), false);
}

void TrackingAPITest::AssertEventFailed(uint32_t eventId) {
  ASSERT_EQ(CheckEventHasStatus(eventId, TTV_TRACKING_BATCH_FAILED), true);
}

void TrackingAPITest::AssertEventNotFailed(uint32_t eventId) {
  ASSERT_EQ(CheckEventHasStatus(eventId, TTV_TRACKING_BATCH_FAILED), false);
}

void TrackingAPITest::AssertEventDiscarded(uint32_t eventId) {
  ASSERT_EQ(CheckEventHasStatus(eventId, TTV_TRACKING_BATCH_DISCARDED), true);
}

void TrackingAPITest::AssertEventNotDiscarded(uint32_t eventId) {
  ASSERT_EQ(CheckEventHasStatus(eventId, TTV_TRACKING_BATCH_DISCARDED), false);
}

void TrackingAPITest::SetResponsesAsSucceeding() {
  // TODO: Fix so that we don't have to turn off strict header params
  mSpadeResponse = mHttpRequest->AddResponse("https://spade.twitch.tv/track/")
                     .SetResponseBody("")
                     .SetStatusCode(204)
                     .SetType(ttv::HTTP_POST_REQUEST)
                     .StrictBody(false)
                     .StrictHeaderParams(false)
                     .StrictRequestParams(false)
                     .shared_from_this();
}

void TrackingAPITest::SetResponsesAsFailing(uint32_t statusCode) {
  mSpadeResponse = mHttpRequest->AddResponse("https://spade.twitch.tv/track/")
                     .SetResponseBody("")
                     .SetStatusCode(statusCode)
                     .SetType(ttv::HTTP_POST_REQUEST)
                     .StrictBody(false)
                     .StrictHeaderParams(false)
                     .StrictRequestParams(false)
                     .shared_from_this();
}
