#include "fixtures/trackingapitest.h"
#include "testhttprequest.h"
#include "testutilities.h"
#include "twitchsdk/tracking/trackingapi.h"

#include "gtest/gtest.h"

using namespace ttv;
using namespace ttv::tracking;

TEST_F(TrackingAPITest, TestShutdownWithUnflushedEvents) {
  SetResponsesAsSucceeding();

  std::map<std::string, TrackingValue> params;
  params["username"] = "TrackingTestUser";

  mTrackingApi->TrackEvent("TrackingTestEvent", params);

  ttv::test::ShutdownModules({mTrackingApi});

  AssertEventDiscarded(0);
  ASSERT_EQ(mSpadeResponse->GetRequestCount(), 0);
}

TEST_F(TrackingAPITest, TestTrackEventThenFlushFromTimeout) {
  SetResponsesAsSucceeding();

  std::map<std::string, TrackingValue> params;
  params["username"] = "TrackingTestUser";

  mTrackingApi->TrackEvent("TrackingTestEvent1", params);
  mTrackingApi->TrackEvent("TrackingTestEvent2", params);

  UpdateUntilEventFlushed(500, 1);

  AssertEventSucceeded(0);
  AssertEventSucceeded(1);
  ASSERT_EQ(mSpadeResponse->GetRequestCount(), 1);
}

TEST_F(TrackingAPITest, TestTrackEventThenHttpFailure) {
  SetResponsesAsFailing(400);

  std::map<std::string, TrackingValue> params;
  params["username"] = "TrackingTestUser";

  mTrackingApi->TrackEvent("TrackingTestEvent", params);

  UpdateUntilEventFlushed(500, 0);

  AssertEventFailed(0);
  AssertEventDiscarded(0);
  ASSERT_EQ(mSpadeResponse->GetRequestCount(), 1);
}

TEST_F(TrackingAPITest, TestTrackEventNetworkFailureThenSuccess) {
  SetResponsesAsFailing(0);

  std::map<std::string, TrackingValue> params;
  params["username"] = "TrackingTestUser";

  mTrackingApi->TrackEvent("TrackingTestEvent", params);

  UpdateUntilEventFlushed(500, 0);

  AssertEventFailed(0);
  AssertEventNotDiscarded(0);
  ASSERT_EQ(mSpadeResponse->GetRequestCount(), 1);

  mHttpRequest->RemoveResponse(mSpadeResponse);
  mTrackingApiTestManager->mCompletedBatches.clear();

  SetResponsesAsSucceeding();

  mTrackingApi->TrackEvent("TrackingTestEvent", params);

  UpdateUntilEventFlushed(2000, 1);

  AssertEventSucceeded(0);
  AssertEventSucceeded(1);
  ASSERT_EQ(mSpadeResponse->GetRequestCount(), 1);

  ttv::test::ShutdownModules({mTrackingApi});
}

TEST_F(TrackingAPITest, TestMaxPendingListSizeReachedAfterFailure) {
  mTrackingApi->SetMaxPendingEvents(2);

  SetResponsesAsFailing(0);

  std::map<std::string, TrackingValue> params;
  params["username"] = "TrackingTestUser";

  mTrackingApi->TrackEvent("TrackingTestEvent", params);
  mTrackingApi->TrackEvent("TrackingTestEvent", params);
  mTrackingApi->TrackEvent("TrackingTestEvent", params);
  mTrackingApi->TrackEvent("TrackingTestEvent", params);
  mTrackingApi->TrackEvent("TrackingTestEvent", params);
  mTrackingApi->TrackEvent("TrackingTestEvent", params);

  UpdateUntilEventFlushed(500, 0);

  AssertEventFailed(0);
  AssertEventFailed(1);
  AssertEventFailed(2);
  AssertEventFailed(3);
  AssertEventFailed(4);
  AssertEventFailed(5);
  AssertEventDiscarded(0);
  AssertEventDiscarded(1);
  AssertEventDiscarded(2);
  AssertEventDiscarded(3);
  AssertEventNotDiscarded(4);
  AssertEventNotDiscarded(5);
  ASSERT_EQ(mSpadeResponse->GetRequestCount(), 1);

  mHttpRequest->RemoveResponse(mSpadeResponse);
  mTrackingApiTestManager->mCompletedBatches.clear();

  SetResponsesAsSucceeding();

  UpdateUntilEventFlushed(2000, 0);

  AssertEventSucceeded(0);
  AssertEventSucceeded(1);
  ASSERT_EQ(mSpadeResponse->GetRequestCount(), 1);
}

TEST_F(TrackingAPITest, TestSingleEventTooBig) {
  mTrackingApi->SetMaxBytesInBatch(20);

  TTV_ErrorCode ec = mTrackingApi->TrackEvent("111111111122222222223333333333", {});
  ASSERT_EQ(ec, TTV_EC_INVALID_JSON);
}

TEST_F(TrackingAPITest, TestMaxBatchBytes) {
  mTrackingApi->SetMaxBytesInBatch(100);

  SetResponsesAsSucceeding();

  mTrackingApi->TrackEvent("1111111111", {});
  mTrackingApi->TrackEvent("111111111122", {});

  UpdateUntilEventFlushed(500, 0);

  AssertEventSucceeded(0);
  AssertEventNotSucceeded(1);
  AssertEventNotFailed(1);
  AssertEventNotDiscarded(1);
  ASSERT_EQ(mSpadeResponse->GetRequestCount(), 1);

  UpdateUntilEventFlushed(500, 1);

  AssertEventSucceeded(1);
  ASSERT_EQ(mSpadeResponse->GetRequestCount(), 2);
}
