import __future__

import argparse
import os
import subprocess
import sys


def is_python_3():
    return sys.version_info >= (3, 0)


def run_command(input):
    if is_python_3():
        process = subprocess.Popen(input,
                                   stdout=subprocess.PIPE,
                                   encoding='utf8')
    else:
        process = subprocess.Popen(input, stdout=subprocess.PIPE)
    stdout, stderr = process.communicate()
    if process.returncode != 0:
        sys.exit(process.returncode)
    return stdout.splitlines()


def get_filter_files(file_paths):
    included_filter = ['modules']
    excluded_filter = ['/generated/', '/dependencies/']

    return [
        x for x in file_paths
        if x.startswith(tuple(included_filter)) and not any(
            f in x for f in excluded_filter)
    ]


extensions = ['*.h', '*.hpp', '*.inl', '*.c', '*.cpp', '*.java']


def get_modified_files():
    return get_filter_files(
        # include files that have been modified + new files
        run_command(['git', 'diff', '--name-only', 'HEAD'] + extensions) +
        run_command(['git', 'ls-files', '--other', '--exclude-standard'] +
                    extensions))


def get_all_files():
    return get_filter_files(run_command(['git', 'ls-files'] + extensions))


def wait_for_tasks(tasks, max_size):
    completed = [task for task in tasks if task.poll()]
    for c in completed:
        c.wait()
        tasks.remove(c)

    while len(tasks) > max_size:
        tasks[0].wait()
        tasks.pop(0)


def format(all_files):
    # set path to current file
    os.chdir(os.path.dirname(os.path.realpath(__file__)))

    # we can now run git to get root path and set it as our new working path
    os.chdir(run_command(['git', 'rev-parse', '--show-toplevel'])[0])
    print(os.getcwd())

    # run the local clang-format to ensure consistency
    if (all_files):
        git_files = get_all_files()
    else:
        git_files = get_modified_files()

    tasks = []
    for filtered_file in git_files:
        abs_filtered_file = os.path.abspath(filtered_file)
        print(abs_filtered_file)

        # temp workaround bc jenkins not updating path
        if 'Path' in os.environ and 'LLVM' not in os.environ['Path']:
            clang_command = 'C:\\Program Files\\LLVM\\bin\\clang-format-7.exe'
        else:
            clang_command = 'clang-format-7'

        tasks.append(
            subprocess.Popen([clang_command, '-i', abs_filtered_file],
                             stdout=subprocess.PIPE))
        wait_for_tasks(tasks, 4)

    wait_for_tasks(tasks, 0)


def main():
    p = argparse.ArgumentParser(
        description='runs clang-format on project files')

    p.add_argument(
        '-a',
        '--all',
        action='store_true',
        help=
        'format all files valid files in git, default is to run on only the modified files'
    )

    opts = p.parse_args()
    format(opts.all)


if __name__ == '__main__':
    main()
