/********************************************************************************************
* Twitch Broadcasting SDK
*
* This software is supplied under the terms of a license agreement with Twitch Interactive, Inc. and
* may not be copied or used except in accordance with the terms of that agreement
* Copyright (c) 2012-2016 Twitch Interactive, Inc.
*********************************************************************************************/

#pragma once

#if TTV_PLATFORM_WINDOWS
#	define TTV_SUPPORT_D3D11 1
#endif


#if TTV_SUPPORT_D3D11

#include "bindingframecapturer.h"
#include "internal/bindings/graphics_d3d11.h"

namespace ttv
{
	class BindingFrameCapturer_D3D11;
}

using namespace ttv::graphics::d3d11;

class ttv::BindingFrameCapturer_D3D11 : public ttv::BindingFrameCapturer
{
public:
	BindingFrameCapturer_D3D11();
	virtual ~BindingFrameCapturer_D3D11();

	virtual void SetGraphicsDevice(void* device, GfxDeviceEventType::Enum eventType);
	virtual TTV_ErrorCode Start(const TTV_VideoParams* videoParams, const TTV_AudioParams* audioParams, const TTV_IngestServer* ingestServer, uint32_t flags);
	virtual TTV_ErrorCode SubmitTexture(void* p, int width, int height);

protected:

	struct Snapshot
	{
		ID3D11DeviceContext* context;
		StagingTexture stagingTexture;
		Query query;
		
		Snapshot(ID3D11DeviceContext* context);
		~Snapshot();

		bool IsLocked() const { return isTextureLocked; }
		uint8_t* GetCopiedBuffer() { return copied; }
		uint8_t* GetLockedBuffer() { return static_cast<uint8_t*>(lockedInfo.pData); }
		uint GetLockedRowPitch() { return lockedInfo.RowPitch; }
		uint8_t* GetActiveBuffer() { return isTextureLocked ? static_cast<uint8_t*>(lockedInfo.pData) : copied; }

		void CreateCopyOfLocked();
		bool Lock();
		void Unlock();

	private:
		D3D11_MAPPED_SUBRESOURCE lockedInfo;
		uint8_t* copied; //!< The copy of the locked buffer.
		bool isTextureLocked; //!< Whether or not to use the mapped buffer.
	};

	typedef std::vector< std::shared_ptr<Snapshot> > SnapshotList;

	ID3D11Device* mGraphicsDevice;
	ID3D11DeviceContext* mDeviceContext;
	ID3D11RasterizerState* mRasterizerState;
	ID3D11ShaderResourceView* mSceneTextureResourceView; //!< The resource view for the currently active external scene texture.
	ID3D11Texture2D* mSceneTexture; //!< The currently passed in scene texture.
	D3D11_VIEWPORT mViewport; //!< The viewport mapping the scene texture into the broadcast texture at the correct aspect ratio.

	virtual void CleanupObjects();
	virtual bool ShouldCreateFrameBuffers() const;
	virtual void HandleBufferUnlock(const uint8_t* buffer);
	std::shared_ptr<Snapshot> UnlockTexture(const void* buffer);
	TTV_ErrorCode AllocateTextures(uint broadcastWidth, uint broadcastHeight);

	Quad mBroadcastQuad;
	QuadShader mQuadShader;
	RenderTexture mRenderTexture;
	
	SnapshotList mFreeTextures; //!< Textures which are free to render to.
	SnapshotList mPendingTextures; //!< Textures which are being rendered to.
	SnapshotList mLockedTextures; //!< Textures which have been locked and have been submitted to the sdk.
	
	int mRenderTextureCount; //!< The number of textures to use for delay.

	int mSceneWidth;
	int mSceneHeight;
	bool mAllocatedTextures;
};

#endif
