/********************************************************************************************
* Twitch Broadcasting SDK
*
* This software is supplied under the terms of a license agreement with Twitch Interactive, Inc. and
* may not be copied or used except in accordance with the terms of that agreement
* Copyright (c) 2012-2016 Twitch Interactive, Inc.
*********************************************************************************************/

#pragma once

#if TTV_PLATFORM_WINDOWS
#	define TTV_SUPPORT_D3D11 1
#endif


#if TTV_SUPPORT_D3D11

#include <d3d11.h>

#define SAFE_RELEASE(x) if (x) { (void)x->Release(); x = nullptr; } 

namespace ttv
{
	namespace graphics
	{
		namespace d3d11
		{
			/**
			 * A texture that the scene is rendered to.
			 */
			struct RenderTexture
			{
			public:
				RenderTexture();
				~RenderTexture();
				
				bool Create(ID3D11Device* device, uint width, uint height);
				void Destroy();
				
				void Bind(ID3D11DeviceContext* context);
				void Clear(ID3D11DeviceContext* context, float r, float g, float b, float a);
				
				ID3D11Texture2D* GetTexture() { return mTexture; }
				uint GetWidth() const { return mWidth; }
				uint GetHeight() const { return mHeight; }

			private:
				ID3D11Texture2D* mTexture;
				ID3D11RenderTargetView* mRenderTargetView;
				uint mWidth;
				uint mHeight;
			};

			/**
			 * A CPU-accessible texture.
			 */
			struct StagingTexture
			{
			public:
				StagingTexture();
				~StagingTexture();
				
				bool Create(ID3D11Device* device, uint width, uint height);
				void Destroy();
				
				ID3D11Texture2D* GetTexture() { return mTexture; }
				uint GetWidth() const { return mWidth; }
				uint GetHeight() const { return mHeight; }

			private:
				ID3D11Texture2D* mTexture;
				uint mWidth;
				uint mHeight;
			};

			struct QuadShader
			{
				QuadShader();
				
				bool Create(ID3D11Device* device);
				void Destroy();
				
				void SetUniforms(ID3D11DeviceContext* context, float width, float height, ID3D11ShaderResourceView* texture);
				void Bind(ID3D11DeviceContext* context);

			private:
				// this must match the cbuffer in the vertex shader
				struct VertexShaderConstantsBuffer
				{
					float mvpMatrix[16];
				};
	
				ID3D11VertexShader* mVertexShader;
				ID3D11PixelShader* mPixelShader;
				ID3D11InputLayout* mVertexLayout;
				ID3D11Buffer* mVertexConstantsBuffer;
				ID3D11SamplerState* mSamplerState;
			};
			
			struct Quad
			{
			public:
				static const int kNumQuadVertices = 6;
				
				Quad();
				~Quad();
				
				void Create(ID3D11Device* device, float width, float height);
				void Destroy();
				void Draw(ID3D11DeviceContext* context, QuadShader& shader);
				
				float GetWidth() const { return mWidth; }
				float GetHeight() const { return mHeight; }
				
			private:
				struct VertexType
				{
					float x, y, tx, ty;
				};

				ID3D11Buffer* mVertexBuffer;
				float mWidth;
				float mHeight;
			};

			struct Query
			{
			public:
				Query();
				~Query();
				
				bool Create(ID3D11Device* device);
				void Destroy();
				
				void Mark(ID3D11DeviceContext* context);
				bool IsReady(ID3D11DeviceContext* context);

			private:
				ID3D11Query* mQuery;
			};
		}
	}
}

#endif
