/********************************************************************************************
* Twitch Broadcasting SDK
*
* This software is supplied under the terms of a license agreement with Twitch Interactive, Inc. and
* may not be copied or used except in accordance with the terms of that agreement
* Copyright (c) 2012-2016 Twitch Interactive, Inc.
*
* This file was automatically generated.  Do not edit manually.
*********************************************************************************************/

#include "internal/pch.h"
#include "internal/bindings/bindingframecapturer_d3d11.h"


#if TTV_SUPPORT_D3D11

#include "twitchsdk.h"



ttv::BindingFrameCapturer_D3D11::Snapshot::Snapshot(ID3D11DeviceContext* context)
:	copied(nullptr)
,	isTextureLocked(false)
{
	Snapshot::context = context;
	memset(&lockedInfo, 0, sizeof(lockedInfo));
}


void ttv::BindingFrameCapturer_D3D11::Snapshot::CreateCopyOfLocked()
{
	assert(isTextureLocked);

	int rowSize = stagingTexture.GetWidth() * 4;
	int bufferSize = rowSize * stagingTexture.GetHeight();

	if (copied == nullptr)
	{
		copied  = new uint8_t[bufferSize];
	}

	uint expectedPitch = stagingTexture.GetWidth() * 4;

	if (lockedInfo.RowPitch == expectedPitch)
	{
		memcpy(copied, lockedInfo.pData, bufferSize);
	}
	else
	{
		uint8_t* src = static_cast<uint8_t*>(lockedInfo.pData);
		uint8_t* dest = copied;

		for (uint32_t y = 0; y < stagingTexture.GetHeight(); ++y)
		{
			memcpy(dest, src, rowSize);

			dest += rowSize;
			src += lockedInfo.RowPitch;
		}
	}
}


bool ttv::BindingFrameCapturer_D3D11::Snapshot::Lock()
{
	assert(lockedInfo.pData == nullptr && !isTextureLocked);
	HRESULT hr = context->Map(stagingTexture.GetTexture(), 0, D3D11_MAP_READ, 0, &lockedInfo);

	if ( SUCCEEDED(hr) )
	{
		isTextureLocked = true;
		return true;
	}
	else
	{
		memset(&lockedInfo, 0, sizeof(lockedInfo));
		return false;
	}
}


void ttv::BindingFrameCapturer_D3D11::Snapshot::Unlock()
{
	if (isTextureLocked && lockedInfo.pData != nullptr)
	{
		context->Unmap(stagingTexture.GetTexture(), 0);

		memset(&lockedInfo, 0, sizeof(lockedInfo));
		isTextureLocked = false;
	}
}


ttv::BindingFrameCapturer_D3D11::Snapshot::~Snapshot()
{
	Unlock();
			
	delete [] copied;
	copied = nullptr;

	stagingTexture.Destroy();
}


ttv::BindingFrameCapturer_D3D11::BindingFrameCapturer_D3D11()
:	mGraphicsDevice(nullptr)
,	mDeviceContext(nullptr)
,	mRasterizerState(nullptr)
,	mSceneTextureResourceView(nullptr)
,	mSceneTexture(nullptr)
,	mSceneWidth(0)
,	mSceneHeight(0)
,	mRenderTextureCount(3)
,	mAllocatedTextures(false)
{
	ttv::trace::Message("BindingFrameCapturer", TTV_ML_INFO, "BindingFrameCapturer_D3D11 created");
}


ttv::BindingFrameCapturer_D3D11::~BindingFrameCapturer_D3D11()
{
}


void ttv::BindingFrameCapturer_D3D11::CleanupObjects()
{
	assert(mLockedTextures.empty());
	mFreeTextures.clear();
	mPendingTextures.clear();
	
	mRenderTexture.Destroy();
	mQuadShader.Destroy();
	mBroadcastQuad.Destroy();

	SAFE_RELEASE(mSceneTexture);
	SAFE_RELEASE(mSceneTextureResourceView);
	SAFE_RELEASE(mRasterizerState);

	mSceneWidth = 0;
	mSceneHeight = 0;
	mAllocatedTextures = false;
}


void ttv::BindingFrameCapturer_D3D11::SetGraphicsDevice(void* device, GfxDeviceEventType::Enum eventType)
{
	ID3D11Device* d3dDevice = reinterpret_cast<ID3D11Device*>(device);

	// clean up previous objects
	CleanupObjects();

	SAFE_RELEASE(mDeviceContext);
	SAFE_RELEASE(mGraphicsDevice);

	switch (eventType) 
	{
		case GfxDeviceEventType::kGfxDeviceEventInitialize:
		case GfxDeviceEventType::kGfxDeviceEventAfterReset:
		{
			mGraphicsDevice = d3dDevice;
			(void)d3dDevice->AddRef();

			mGraphicsDevice->GetImmediateContext(&mDeviceContext);

			break;
		}
		case GfxDeviceEventType::kGfxDeviceEventBeforeReset:
		case GfxDeviceEventType::kGfxDeviceEventShutdown:
		{
			break;
		}
	}
}


TTV_ErrorCode ttv::BindingFrameCapturer_D3D11::Start(const TTV_VideoParams* videoParams, const TTV_AudioParams* audioParams, const TTV_IngestServer* ingestServer, uint32_t flags)
{
	assert(mDeviceContext != nullptr);
	assert(mGraphicsDevice != nullptr);

	TTV_ErrorCode ec = BindingFrameCapturer::Start(videoParams, audioParams, ingestServer, flags);
	
	if (TTV_SUCCEEDED(ec))
	{
		// TODO: we might need to move creation of all objects into SubmitTexture due to multithreaded rendering
		mBroadcastQuad.Create(mGraphicsDevice, static_cast<float>(videoParams->outputWidth), static_cast<float>(videoParams->outputHeight));
		mQuadShader.Create(mGraphicsDevice);
	}
	
	return ec;
}


bool ttv::BindingFrameCapturer_D3D11::ShouldCreateFrameBuffers() const
{
	// we don't want to allocate buffers, we'll pass the locked texture data directly to the sdk
	return false;
}


std::shared_ptr<ttv::BindingFrameCapturer_D3D11::Snapshot> ttv::BindingFrameCapturer_D3D11::UnlockTexture(const void* buffer)
{
	auto iter = std::find_if(mLockedTextures.begin(), mLockedTextures.end(),
							[buffer](std::shared_ptr<Snapshot>& rt)->bool { return rt->GetActiveBuffer() == buffer; });
	
	assert(iter != mLockedTextures.end());
	
	std::shared_ptr<Snapshot> snapshot;
	
	if (iter != mLockedTextures.end())
	{
		snapshot = *iter;
		snapshot->Unlock();

		mLockedTextures.erase(iter);
	}
	
	return snapshot;
}


void ttv::BindingFrameCapturer_D3D11::HandleBufferUnlock(const uint8_t* buffer)
{
	auto rt = UnlockTexture(buffer);
	
	if (rt != nullptr)
	{
		mFreeTextures.push_back(rt);
	}
}


namespace
{
	void CalculateViewport(int screenWidth, int screenHeight, int broadcastWidth, int broadcastHeight, D3D11_VIEWPORT& viewport)
	{
		float screenAspectRatio = static_cast<float>(screenWidth) / static_cast<float>(screenHeight);
		float broadcastApectRatio = static_cast<float>(broadcastWidth) / static_cast<float>(broadcastHeight);
	
		if (broadcastApectRatio >= screenAspectRatio)
		{
			viewport.TopLeftY = 0;
			viewport.Height = static_cast<float>(broadcastHeight);
		
			viewport.Width = static_cast<float>(broadcastHeight) * screenAspectRatio;
			viewport.TopLeftX = static_cast<float>((broadcastWidth - viewport.Width) / 2);
		}
		else
		{
			viewport.TopLeftX = 0;
			viewport.Width = static_cast<float>(broadcastWidth);
		
			viewport.Height = static_cast<float>(broadcastWidth) / screenAspectRatio;
			viewport.TopLeftY = static_cast<float>((broadcastHeight - viewport.Height) / 2);
		}

		viewport.MinDepth = 0;
		viewport.MaxDepth = 1;
	}
}


TTV_ErrorCode ttv::BindingFrameCapturer_D3D11::AllocateTextures(uint broadcastWidth, uint broadcastHeight)
{
	D3D11_RASTERIZER_DESC rasterDesc;

	rasterDesc.AntialiasedLineEnable = false;
	rasterDesc.CullMode = D3D11_CULL_NONE;
	rasterDesc.DepthBias = 0;
	rasterDesc.DepthBiasClamp = 0.0f;
	rasterDesc.DepthClipEnable = true;
	rasterDesc.FillMode = D3D11_FILL_SOLID;
	rasterDesc.FrontCounterClockwise = false;
	rasterDesc.MultisampleEnable = false;
	rasterDesc.ScissorEnable = false;
	rasterDesc.SlopeScaledDepthBias = 0.0f;
	
	TTV_ErrorCode ec = TTV_EC_SUCCESS;

	// Create the rasterizer state from the description we just filled out.
	if ( FAILED(mGraphicsDevice->CreateRasterizerState(&rasterDesc, &mRasterizerState)) )
	{
		ec = TTV_EC_GRAPHICS_API_ERROR;
	}

	bool created = mRenderTexture.Create(mGraphicsDevice, broadcastWidth, broadcastHeight);
	assert(created);

	for (int i = 0; i < mRenderTextureCount; ++i)
	{
		std::shared_ptr<Snapshot> snapshot = std::make_shared<Snapshot>(mDeviceContext);

		created = snapshot->stagingTexture.Create(mGraphicsDevice, broadcastWidth, broadcastHeight);
		assert(created);

		created = snapshot->query.Create(mGraphicsDevice);
		assert(created);

		mFreeTextures.push_back(snapshot);
	}

	mAllocatedTextures = true;

	return ec;
}


TTVSDK_API TTV_ErrorCode ttv::BindingFrameCapturer_D3D11::SubmitTexture(void* p, int width, int height)
{
	if (p == nullptr)
	{
		return TTV_EC_INVALID_ARG;
	}
	else if (mGraphicsDevice == nullptr || mDeviceContext == nullptr)
	{
		return TTV_EC_NOT_INITIALIZED;
	}

	TTV_ErrorCode err = TTV_EC_SUCCESS;

	ID3D11Texture2D* sceneTexture = reinterpret_cast<ID3D11Texture2D*>(p);
	bool requiresNewResourceView = false;

	uint32_t broadcastWidth = mVideoParams.outputWidth;
	uint32_t broadcastHeight = mVideoParams.outputHeight;

	// the incoming texture has changed
	if (mSceneTexture != sceneTexture)
	{
		SAFE_RELEASE(mSceneTexture);

		mSceneTexture = sceneTexture;
		mSceneTexture->AddRef();

		requiresNewResourceView = true;
	}

	// get information about the source surface
	D3D11_TEXTURE2D_DESC srcDesc;
	sceneTexture->GetDesc(&srcDesc);
	
	if (!requiresNewResourceView)
	{
		requiresNewResourceView = (width != mSceneWidth) || (height != mSceneHeight);
	}

	// update the texture binding since it's changed
	if (requiresNewResourceView)
	{
		SAFE_RELEASE(mSceneTextureResourceView);

		D3D11_SHADER_RESOURCE_VIEW_DESC resourceViewDesc;
		resourceViewDesc.Format = DXGI_FORMAT_R8G8B8A8_UNORM;
		resourceViewDesc.ViewDimension = D3D_SRV_DIMENSION_TEXTURE2D;
		resourceViewDesc.Texture2D.MostDetailedMip = 0;
		resourceViewDesc.Texture2D.MipLevels = srcDesc.MipLevels;
		
		if ( FAILED(mGraphicsDevice->CreateShaderResourceView(sceneTexture, &resourceViewDesc, &mSceneTextureResourceView)) )
		{
			err = TTV_EC_GRAPHICS_API_ERROR;
		}

		if (TTV_SUCCEEDED(err))
		{
			mSceneWidth = width;
			mSceneHeight = height;

			mRenderTexture.Clear(mDeviceContext, 0, 0, 0, 0);

			CalculateViewport(width, height, broadcastWidth, broadcastHeight, mViewport);
		}
	}

	// allocate the broadcast textures if not already allocated
	if (TTV_SUCCEEDED(err) && !mAllocatedTextures)
	{
		err = AllocateTextures(broadcastWidth, broadcastHeight);
	}

	// lock and submit the oldest frame
	if (TTV_SUCCEEDED(err) && mPendingTextures.size() > 0)
	{
		std::shared_ptr<Snapshot> snapshot = mPendingTextures.front();
		
		// make sure the render is complete
		if (snapshot->query.IsReady(mDeviceContext))
		{
			mPendingTextures.erase(mPendingTextures.begin());
			
			// try and lock the pixels
			if ( !snapshot->Lock() )
			{
 				err = TTV_EC_GRAPHICS_API_ERROR;
			}

			// the lock succeeded
			if (TTV_SUCCEEDED(err))
			{
				uint32_t rowSize = broadcastWidth * 4;
				uint8_t* dest = nullptr;

				// we can pass the locked pixels directy to the SDK if the pitch has no padding and the pointer is 16-byte aligned
				if ((snapshot->GetLockedRowPitch() == rowSize) &&
					((reinterpret_cast<uintptr_t>(snapshot->GetLockedBuffer()) & 15) == 0))
				{
					dest = snapshot->GetLockedBuffer();
				}
				// we need to copy the locked pixels
				else
				{
					snapshot->CreateCopyOfLocked();
					snapshot->Unlock();
					dest = snapshot->GetCopiedBuffer();
				}

				err = TTV_SubmitVideoFrame(dest, BufferUnlockCallback, static_cast<BindingFrameCapturer*>(this));

				// submission failed so abort the texture lock
				if (TTV_FAILED(err))
				{
					if (snapshot->IsLocked())
					{
						snapshot->Unlock();
					}
				}
			}

			if (TTV_SUCCEEDED(err))
			{
				mLockedTextures.push_back(snapshot);
			}
			else
			{
				mFreeTextures.push_back(snapshot);
			}
		}
	}

	// capture the current frame
	if (TTV_SUCCEEDED(err) && mFreeTextures.size() > 0)
	{
		std::shared_ptr<Snapshot> snapshot = mFreeTextures.front();
		mFreeTextures.erase(mFreeTextures.begin());

		// save state
		UINT prevNumViewports = 4;
		D3D11_VIEWPORT prevViewports[4];
		mDeviceContext->RSGetViewports(&prevNumViewports, prevViewports);

		ID3D11RasterizerState* prevRasterizerState = nullptr;
		mDeviceContext->RSGetState(&prevRasterizerState);

		// render the scene into the broadcast texture
		mDeviceContext->RSSetState(mRasterizerState);
		mDeviceContext->RSSetViewports(1, &mViewport);
		mQuadShader.SetUniforms(mDeviceContext, static_cast<float>(broadcastWidth), static_cast<float>(broadcastHeight), mSceneTextureResourceView);
		mRenderTexture.Bind(mDeviceContext);
		mBroadcastQuad.Draw(mDeviceContext, mQuadShader);

		// restore state
		mDeviceContext->RSSetViewports(prevNumViewports, prevViewports);
		mDeviceContext->RSSetState(prevRasterizerState);

		// copy the render target to the staging texture which is readable by the CPU
		mDeviceContext->CopyResource(snapshot->stagingTexture.GetTexture(), mRenderTexture.GetTexture());

		// schedule a query so we know when it's done
		snapshot->query.Mark(mDeviceContext);

		mPendingTextures.push_back(snapshot);
	}

	return err;
}

#endif
