/********************************************************************************************
* Twitch Broadcasting SDK
*
* This software is supplied under the terms of a license agreement with Twitch Interactive, Inc. and
* may not be copied or used except in accordance with the terms of that agreement
* Copyright (c) 2012-2016 Twitch Interactive, Inc.
*
* This file was automatically generated.  Do not edit manually.
*********************************************************************************************/

#include "internal/pch.h"
#include "internal/bindings/bindingframecapturer_d3d9.h"


#if TTV_SUPPORT_D3D9

#include "twitchsdk.h"

#define SAFE_RELEASE(x) if (x) { (void)x->Release(); x = nullptr; } 


ttv::BindingFrameCapturer_D3D9::BindingFrameCapturer_D3D9()
:	mGraphicsDevice(nullptr)
,	mCaptureTexture(nullptr)
,	mCaptureSurface(nullptr)
,	mMainRenderTexture(nullptr)
,	mMainRenderTargetSurface(nullptr)
,	mCapturePut(0)
,	mCaptureGet(0)
,	mSceneWidth(0)
,	mSceneHeight(0)
,	mPreviousDeviceStatus(S_OK)
{
	ttv::trace::Message("BindingFrameCapturer", TTV_ML_INFO, "BindingFrameCapturer_D3D9 created");

	memset(mCaptureQuery, 0, sizeof(mCaptureQuery));
	memset(mResizeSurface, 0, sizeof(mResizeSurface));
}


ttv::BindingFrameCapturer_D3D9::~BindingFrameCapturer_D3D9()
{
}


void ttv::BindingFrameCapturer_D3D9::CleanupObjects()
{
	SAFE_RELEASE(mMainRenderTexture);
	SAFE_RELEASE(mMainRenderTargetSurface);

	// Cleanup the previous texture
	SAFE_RELEASE(mCaptureTexture);
	SAFE_RELEASE(mCaptureSurface);

	// Release previous queries and surfaces
	for (int i = 0; i < kNumCaptureSurfaces; ++i)
	{
		SAFE_RELEASE(mCaptureQuery[i]);
		SAFE_RELEASE(mResizeSurface[i]);
	}

	mSceneWidth = 0;
	mSceneHeight = 0;
}


bool ttv::BindingFrameCapturer_D3D9::RequiresVerticalFlip()
{
	return true;
}


void ttv::BindingFrameCapturer_D3D9::SetGraphicsDevice(void* device, GfxDeviceEventType::Enum eventType)
{
	IDirect3DDevice9* d3dDevice = reinterpret_cast<IDirect3DDevice9*>(device);

	// no change
	if (d3dDevice == mGraphicsDevice)
	{
		return;
	}

	// clean up previous objects
	CleanupObjects();

	SAFE_RELEASE(mGraphicsDevice);

	switch (eventType) 
	{
		case GfxDeviceEventType::kGfxDeviceEventInitialize:
		case GfxDeviceEventType::kGfxDeviceEventAfterReset:
		{
			mGraphicsDevice = d3dDevice;
			mGraphicsDevice->AddRef();
			break;
		}
		case GfxDeviceEventType::kGfxDeviceEventBeforeReset:
		case GfxDeviceEventType::kGfxDeviceEventShutdown:
		{
			break;
		}
	}
}


TTV_ErrorCode ttv::BindingFrameCapturer_D3D9::Stop()
{
	TTV_ErrorCode ec = BindingFrameCapturer::Stop();

	if (TTV_SUCCEEDED(ec))
	{
		mPreviousDeviceStatus = S_OK;
	}

	return ec;
}


namespace
{
	void CalculateViewport(int screenWidth, int screenHeight, int broadcastWidth, int broadcastHeight, RECT& dest)
	{
		float screenAspectRatio = static_cast<float>(screenWidth) / static_cast<float>(screenHeight);
		float broadcastApectRatio = static_cast<float>(broadcastWidth) / static_cast<float>(broadcastHeight);
	
		if (broadcastApectRatio >= screenAspectRatio)
		{
			dest.top = 0;
			dest.bottom = broadcastHeight;
		
			long width = static_cast<int>( static_cast<float>(broadcastHeight) * screenAspectRatio );
			dest.left = (broadcastWidth - width) / 2;
			dest.right = (broadcastWidth + width) / 2;
		}
		else
		{
			dest.left = 0;
			dest.right = broadcastWidth;
		
			long height = static_cast<int>( static_cast<float>(broadcastWidth) / screenAspectRatio );
			dest.top = (broadcastHeight - height) / 2;
			dest.bottom = (broadcastHeight + height) / 2;
		}
	}
}


TTVSDK_API TTV_ErrorCode ttv::BindingFrameCapturer_D3D9::SubmitTexture(void* p, int width, int height)
{
	if (p == nullptr)
	{
		return TTV_EC_INVALID_ARG;
	}
	else if (mGraphicsDevice == nullptr)
	{
		return TTV_EC_NOT_INITIALIZED;
	}

	assert(!mFreeFrameBuffers.empty());
	if (mFreeFrameBuffers.empty())
	{
		return TTV_EC_NOT_INITIALIZED;
	}

	// make sure the device hasn't been lost
	HRESULT hr = mGraphicsDevice->TestCooperativeLevel();
	switch (hr)
	{
		case S_OK:
		{
			if ( FAILED(mPreviousDeviceStatus) )
			{
				// the device has been lost so pause the video until it comes back
				ttv::trace::Message("BindingFrameCapturer", TTV_ML_INFO, "Graphics device lost");
				mPreviousDeviceStatus = hr;
			}
			break;
		}
		case D3DERR_DEVICELOST:
		case D3DERR_DEVICENOTRESET:
		{
			if ( SUCCEEDED(mPreviousDeviceStatus) )
			{
				// the device has been restored so continue submitting frames
				ttv::trace::Message("BindingFrameCapturer", TTV_ML_INFO, "Graphics device restored");
				TTV_PauseVideo();
				CleanupObjects();
				mPreviousDeviceStatus = hr;
			}

			// feign success
			return TTV_EC_SUCCESS;
		}
		case D3DERR_DRIVERINTERNALERROR:
		{
			// a fatal error
			ttv::trace::Message("BindingFrameCapturer", TTV_ML_ERROR, "D3DERR_DRIVERINTERNALERROR");
			mPreviousDeviceStatus = hr;
			return TTV_EC_GRAPHICS_API_ERROR;
		}
		default:
		{
			assert(false);
			break;
		}
	}

	IDirect3DTexture9* renderTexture = reinterpret_cast<IDirect3DTexture9*>(p);

	// Get a valid surface from the texture
	if (mMainRenderTargetSurface == nullptr || renderTexture != mMainRenderTexture)
	{
		SAFE_RELEASE(mMainRenderTargetSurface);
		SAFE_RELEASE(mMainRenderTexture);

		mMainRenderTexture = renderTexture;
		(void)renderTexture->AddRef();

		// Create the surface to render to
		if ( FAILED(mMainRenderTexture->GetSurfaceLevel(0, &mMainRenderTargetSurface)) )
		{
			SAFE_RELEASE(mMainRenderTexture);
			return TTV_EC_GRAPHICS_API_ERROR;
		}
	}

	// Get information about the source surface
	D3DSURFACE_DESC srcDesc;
	if ( FAILED( mMainRenderTargetSurface->GetDesc( &srcDesc ) ) )
	{
		SAFE_RELEASE(mMainRenderTexture);
		return TTV_EC_GRAPHICS_API_ERROR;
	}

	uint32_t captureWidth = mVideoParams.outputWidth;
	uint32_t captureHeight = mVideoParams.outputHeight;

	// Allocate the capture texture and surfaces
	if (mCaptureTexture == nullptr)
	{
		// Cleanup the previous texture
		SAFE_RELEASE(mCaptureTexture);
		SAFE_RELEASE(mCaptureSurface);

		// Release previous queries and surfaces
		for (int i = 0; i < kNumCaptureSurfaces; ++i)
		{
			SAFE_RELEASE(mCaptureQuery[i]);
			SAFE_RELEASE(mResizeSurface[i]);
		}

		// Allocate another texture of the correct size
		if ( FAILED(mGraphicsDevice->CreateTexture(captureWidth, captureHeight, 1, D3DUSAGE_RENDERTARGET, srcDesc.Format, D3DPOOL_DEFAULT, &mCaptureTexture, nullptr)) )
		{
			CleanupObjects();
			return TTV_EC_GRAPHICS_API_ERROR;
		}

		// Get the main surface from this texture
		if ( FAILED(mCaptureTexture->GetSurfaceLevel(0, &mCaptureSurface)) )
		{
			CleanupObjects();
			return TTV_EC_GRAPHICS_API_ERROR;
		}

		// Allocate surfaces
		for (int i = 0; i < kNumCaptureSurfaces; ++i)
		{
			// Create offscreen render targets for the results of the captures
			if ( FAILED(mGraphicsDevice->CreateOffscreenPlainSurface(captureWidth, captureHeight, srcDesc.Format, D3DPOOL_SYSTEMMEM, &mResizeSurface[i], nullptr)) )
			{
				CleanupObjects();
				return TTV_EC_GRAPHICS_API_ERROR;
			}
		}

		mCaptureGet = 0;
		mCapturePut = 0;
	}

	// We haven't queued too many requests
	if ( (mCapturePut - mCaptureGet) < kNumCaptureSurfaces )
	{
		int idx = mCapturePut % kNumCaptureSurfaces;

		// clear the background to black if the aspect ratio of the scene has changed
		if (width != mSceneWidth || height != mSceneHeight)
		{
			if ( FAILED(mGraphicsDevice->ColorFill(mCaptureSurface, nullptr, D3DCOLOR_ARGB(255, 0, 0, 0))) )
			{
				return TTV_EC_GRAPHICS_API_ERROR;
			}

			mSceneWidth = width;
			mSceneHeight = height;
		}

		// Copy the main render target to our capture render target
		RECT destRect;
		CalculateViewport(width, height, captureWidth, captureHeight, destRect);

		if ( FAILED(mGraphicsDevice->StretchRect(mMainRenderTargetSurface, nullptr, mCaptureSurface, &destRect, D3DTEXF_LINEAR)) )
		{
			return TTV_EC_GRAPHICS_API_ERROR;
		}

		// Copy data from the rendertarget to the memory surface
		if ( FAILED(mGraphicsDevice->GetRenderTargetData(mCaptureSurface, mResizeSurface[idx])) )
		{
			return TTV_EC_GRAPHICS_API_ERROR;
		}

		// Create a query that will indicate when the GetRenderTargetData call has finished
		if (mCaptureQuery[idx] == nullptr)
		{
			if ( FAILED(mGraphicsDevice->CreateQuery(D3DQUERYTYPE_EVENT, &mCaptureQuery[idx])) )
			{
				return TTV_EC_GRAPHICS_API_ERROR;
			}
		}

		// Schedule the query
		if ( FAILED(mCaptureQuery[idx]->Issue( D3DISSUE_END )) )
		{
			return TTV_EC_GRAPHICS_API_ERROR;
		}

		++mCapturePut;
	}

	TTV_ErrorCode err = TTV_EC_SUCCESS;

	// Get the latest capture
	int idx = mCaptureGet % kNumCaptureSurfaces;
	if (mCaptureGet != mCapturePut && 
		mCaptureQuery[idx] && 
		mCaptureQuery[idx]->GetData(nullptr, 0, 0) == S_OK)
	{
		D3DLOCKED_RECT locked;
		memset( &locked, 0, sizeof(locked) );

		// Attempt to lock the surface to obtain the pixel data
		HRESULT ret = mResizeSurface[idx]->LockRect(&locked, nullptr, D3DLOCK_READONLY);
		if ( SUCCEEDED(ret) )
		{
			// get a free buffer
			uint8_t* dest = mFreeFrameBuffers.back();
			mFreeFrameBuffers.pop_back();

			// Fill the output buffer
			uint8_t* src = static_cast<uint8_t*>(locked.pBits);
			uint32_t rowSize = captureWidth*4;
			uint32_t bufferSize = rowSize*captureHeight;

			if (locked.Pitch == static_cast<INT>(rowSize))
			{
				memcpy(dest, src, bufferSize);
			}
			else
			{
				for (uint32_t y=0; y<captureHeight; ++y)
				{
					memcpy(dest, src, rowSize);

					dest += rowSize;
					src += locked.Pitch;
				}			
			}

			// Unlock the surface
			ret = mResizeSurface[idx]->UnlockRect();
			assert( SUCCEEDED(ret) );

			++mCaptureGet;

			err = TTV_SubmitVideoFrame(dest, BufferUnlockCallback, static_cast<BindingFrameCapturer*>(this));
		}
	}

	return err;
}

#endif
