/********************************************************************************************
* Twitch Broadcasting SDK
*
* This software is supplied under the terms of a license agreement with Twitch Interactive, Inc. and
* may not be copied or used except in accordance with the terms of that agreement
* Copyright (c) 2012-2016 Twitch Interactive, Inc.
*********************************************************************************************/

#include "internal/pch.h"
#include "internal/bindings/bindingframecapturer_ios_rendertexture_gles2.h"
#include "internal/bindings/graphics_common_gl.h"

#if TTV_SUPPORT_GLES2

using namespace ttv::graphics::ios;
using namespace ttv::graphics::gl;


ttv::BindingFrameCapturer_iOS_RenderTexture_GLES2::BindingFrameCapturer_iOS_RenderTexture_GLES2()
:	mRenderTextureCount(3)
,	mSceneWidth(0)
,	mSceneHeight(0)
{
	ttv::trace::Message("BindingFrameCapturer", TTV_ML_INFO, "BindingFrameCapturer_iOS_RenderTexture_GLES2 created");
}


ttv::BindingFrameCapturer_iOS_RenderTexture_GLES2::~BindingFrameCapturer_iOS_RenderTexture_GLES2()
{
}


void ttv::BindingFrameCapturer_iOS_RenderTexture_GLES2::CleanupObjects()
{
	assert(mLockedTextures.empty());
	mFreeTextures.clear();
	mPendingTextures.clear();
	
	mResizeFrameBuffer.Destroy();
	mQuadShader.Destroy();
	mBroadcastQuad.Destroy();
	mTextureCache.Destroy();
	
	mSceneHeight = 0;
	mSceneWidth = 0;
}


TTV_ErrorCode ttv::BindingFrameCapturer_iOS_RenderTexture_GLES2::Start(const TTV_VideoParams* videoParams, const TTV_AudioParams* audioParams, const TTV_IngestServer* ingestServer, uint32_t flags)
{
	EAGLContext* context = [EAGLContext currentContext];
	assert(context != nullptr);
	if (context == nullptr)
	{
		return TTV_EC_GRAPHICS_API_ERROR;
	}

	TTV_ErrorCode ec = BindingFrameCapturer::Start(videoParams, audioParams, ingestServer, flags);
	
	if (TTV_SUCCEEDED(ec))
	{
		mResizeFrameBuffer.Create();
		mQuadShader.Create();
		mTextureCache.Create(context);
		
		for (int i = 0; i < mRenderTextureCount; ++i)
		{
			std::shared_ptr<RenderTexture> rt = std::make_shared<RenderTexture>();
			
			mTextureCache.CreateTexture(*rt, videoParams->outputWidth, videoParams->outputHeight, true);
			
			mFreeTextures.push_back(rt);
			
			// cause the first few frames to have some delay
			mPendingTextures.push_back(nullptr);
		}
	}
	
	return ec;
}


TTV_ErrorCode ttv::BindingFrameCapturer_iOS_RenderTexture_GLES2::Stop()
{
	TTV_ErrorCode ec = BindingFrameCapturer::Stop();
	
	if (TTV_SUCCEEDED(ec))
	{
		CleanupObjects();
	}
	
	return ec;
}


bool ttv::BindingFrameCapturer_iOS_RenderTexture_GLES2::CheckError()
{
	return GLCHECK();
}


bool ttv::BindingFrameCapturer_iOS_RenderTexture_GLES2::ShouldCreateFrameBuffers() const
{
	// we don't want to allocate buffers, we'll pass the locked texture data directly to the sdk
	return false;
}


std::shared_ptr<RenderTexture> ttv::BindingFrameCapturer_iOS_RenderTexture_GLES2::UnlockTexture(const void* buffer)
{
	TextureList::iterator iter = std::find_if(mLockedTextures.begin(), mLockedTextures.end(),
											  [buffer](std::shared_ptr<RenderTexture>& rt)->bool { return rt->GetLocked() == buffer; });
	
	assert(iter != mLockedTextures.end());
	
	std::shared_ptr<RenderTexture> rt;
	
	if (iter != mLockedTextures.end())
	{
		rt = *iter;
		rt->Unlock();
		
		mLockedTextures.erase(iter);
	}
	
	return rt;
}

				   
void ttv::BindingFrameCapturer_iOS_RenderTexture_GLES2::HandleBufferUnlock(const uint8_t* buffer)
{
	auto rt = UnlockTexture(buffer);
	
	if (rt != nullptr)
	{
		mFreeTextures.push_back(rt);
	}
}


TTV_ErrorCode ttv::BindingFrameCapturer_iOS_RenderTexture_GLES2::SubmitTexture(void* p, int width, int height)
{
	if (p == nullptr)
	{
		return  TTV_EC_INVALID_ARG;
	}
	else if (mResizeFrameBuffer.GetId() == 0)
	{
		return TTV_EC_NOT_INITIALIZED;
	}
	
	TTV_ErrorCode err = TTV_EC_SUCCESS;
	
	// lock and submit the oldest frame
	if (mPendingTextures.size() > 0)
	{
		auto lockTexture = mPendingTextures.front();
		mPendingTextures.erase(mPendingTextures.begin());
		
		// the first entries will be null to cause some delay
		if (lockTexture != nullptr)
		{
			// get the pixels of the broadcast texture
			uint8_t* locked = nullptr;
			if (TTV_SUCCEEDED(err))
			{
				mTextureCache.Flush();
				
				locked = reinterpret_cast<uint8_t*>( lockTexture->Lock() );
				if (locked == nullptr)
				{
					err = TTV_EC_GRAPHICS_API_ERROR;
				}
			}
			
			// submit the frame
			if (TTV_SUCCEEDED(err))
			{
				mLockedTextures.push_back(lockTexture);
				
				err = ttv::BindingFrameCapturer::SubmitVideoFrame(locked, BufferUnlockCallback, static_cast<BindingFrameCapturer*>(this));
				
				if (TTV_FAILED(err))
				{
					UnlockTexture(locked);
				}
			}
			
			if (TTV_FAILED(err))
			{
				mFreeTextures.push_back(lockTexture);
			}
		}
	}
	
	// capture the current frame
	if (TTV_SUCCEEDED(err) && mFreeTextures.size() > 0)
	{
		auto broadcastTexture = mFreeTextures.front();
		mFreeTextures.erase(mFreeTextures.begin());
		
		GLuint sceneTextureId = static_cast<GLuint>( reinterpret_cast<uint64_t>(p) );
		
		// cache state
		GLint prevFrameBufferId = 0;
		glGetIntegerv(GL_FRAMEBUFFER_BINDING, &prevFrameBufferId);
		if (!CheckError())
		{
			err = TTV_EC_GRAPHICS_API_ERROR;
		}
		
		// ensure the shader and quad are setup for the given scene texture size
		if (TTV_SUCCEEDED(err))
		{
			if (width != mSceneWidth || height != mSceneHeight)
			{
				mQuadShader.SetUniforms(width, height);	
				mBroadcastQuad.Create(width, height, false);
				
				mSceneWidth = width;
				mSceneHeight = height;
			}
		}

		// bind the resize frame buffer
		if (TTV_SUCCEEDED(err))
		{
			// bind the resize frame buffer
			glBindFramebuffer(GL_FRAMEBUFFER, mResizeFrameBuffer.GetId());
			if (!CheckError())
			{
				err = TTV_EC_GRAPHICS_API_ERROR;
			}
		}

		// hook the broadcast texture up to the resize frame buffer
		if (TTV_SUCCEEDED(err))
		{
			glFramebufferTexture2D(GL_FRAMEBUFFER, GL_COLOR_ATTACHMENT0, GL_TEXTURE_2D, broadcastTexture->GetId(), 0);
			if (!CheckError())
			{
				err = TTV_EC_GRAPHICS_API_ERROR;
			}
		}
		
		// make sure the frame buffer object is correctly configured
		if (TTV_SUCCEEDED(err))
		{
			if (glCheckFramebufferStatus(GL_FRAMEBUFFER) != GL_FRAMEBUFFER_COMPLETE)
			{
				err = TTV_EC_GRAPHICS_API_ERROR;
			}
		}

		// render the scene texture into the broadcast texture
		if (TTV_SUCCEEDED(err))
		{
			// cache the viewport
			GLint previousViewport[4];
			glGetIntegerv(GL_VIEWPORT, previousViewport);

			// clear to black
			broadcastTexture->Clear(0, 0, 0, 0);
			
			// setup the viewport
			GLint viewport[4];
			CalculateViewport(width, height, mVideoParams.outputWidth, mVideoParams.outputHeight, viewport[0], viewport[1], viewport[2], viewport[3]);
			glViewport(viewport[0], viewport[1], viewport[2], viewport[3]);
			
			// render the scene
			mBroadcastQuad.Draw(mQuadShader, sceneTextureId);
			if (!CheckError())
			{
				err = TTV_EC_GRAPHICS_API_ERROR;
			}
			
			// restore the viewport
			glViewport(previousViewport[0], previousViewport[1], previousViewport[2], previousViewport[3]);
		}
		
		// mark the texture as pending
		if (TTV_SUCCEEDED(err))
		{
			mPendingTextures.push_back(broadcastTexture);
		}
		else
		{
			mFreeTextures.push_back(broadcastTexture);
		}
		
		// restore state
		glBindFramebuffer(GL_FRAMEBUFFER, prevFrameBufferId);
	}
	
	return err;
}

#endif
