/********************************************************************************************
 * Twitch Broadcasting SDK
 *
 * This software is supplied under the terms of a license agreement with Twitch Interactive, Inc. and
 * may not be copied or used except in accordance with the terms of that agreement
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 *********************************************************************************************/

#include "internal/pch.h"
#include "internal/bindings/bindingframecapturer_ios_swizzle_gles2.h"


#if TTV_SUPPORT_GLES2

#include "twitchsdk.h"
#import <objc/runtime.h>
#import <QuartzCore/CAEAGLLayer.h>

namespace
{
	GLint gGameRenderBufferId = 0; // Stored globally as it will be received prior to initialization
}

std::shared_ptr<ttv::BindingFrameCapturer_iOS_Swizzle_GLES2> ttv::BindingFrameCapturer_iOS_Swizzle_GLES2::sInstance;

@interface EAGLContext(TwitchSwizzleRenderbufferCapture)

+ (void)dummyForceClassLoad;
+ (void)load;
- (BOOL)twitch_presentRenderbuffer:(NSUInteger)target;
- (BOOL)twitch_renderbufferStorage:(NSUInteger)target fromDrawable:(id<EAGLDrawable>)drawable;

@end


@implementation EAGLContext(TwitchSwizzleRenderbufferCapture)

+ (void)dummyForceClassLoad
{
	// don't do anything
}


+ (void)load
{
	Method original, swizzle;
	
	// swizzle presentRenderbuffer
	original = class_getInstanceMethod(self, @selector(presentRenderbuffer:));
	swizzle = class_getInstanceMethod(self, @selector(twitch_presentRenderbuffer:));
	method_exchangeImplementations(original, swizzle);
	
	// swizzle renderbufferStorage
	original = class_getInstanceMethod(self, @selector(renderbufferStorage:fromDrawable:));
	swizzle = class_getInstanceMethod(self, @selector(twitch_renderbufferStorage:fromDrawable:));
	method_exchangeImplementations(original, swizzle);
}


- (BOOL)twitch_presentRenderbuffer:(NSUInteger)target
{
	std::shared_ptr<ttv::BindingFrameCapturer_iOS_Swizzle_GLES2> capturer = ttv::BindingFrameCapturer_iOS_Swizzle_GLES2::GetInstance();
	
	if (capturer != nullptr) {
		
		if (capturer->GetIsStarted()) {
		
			assert(self == [EAGLContext currentContext]);
			assert(target == GL_RENDERBUFFER);
		
			BOOL ret = NO;
		
			if (capturer->IsSetUpForCapture())
			{
				TTV_ErrorCode ec = capturer->CaptureBoundRenderbuffer();
				if (TTV_FAILED(ec))
				{
					ttv::trace::Message("BindingFrameCapturer", TTV_ML_ERROR, "BindingFrameCapturer_iOS_Swizzle_GLES2 failed to submit frame from FBO");
				}
				
				ec = capturer->DrawScreenQuad();
				if (TTV_FAILED(ec))
				{
					ttv::trace::Message("BindingFrameCapturer", TTV_ML_ERROR, "BindingFrameCapturer_iOS_Swizzle_GLES2 failed to draw screen quad");
				}
				
				// call the original and present the result
				ret = [self twitch_presentRenderbuffer:target];
				
				capturer->BindGameFramebuffer();
			}
			else
			{
				ret = [self twitch_presentRenderbuffer:target];
			
				if (!capturer->SetupCapture())
				{
					ttv::trace::Message("BindingFrameCapturer", TTV_ML_ERROR, "BindingFrameCapturer_iOS_Swizzle_GLES2 could not set up for capture");
				}
			}
		
			GLenum status = glCheckFramebufferStatus(GL_FRAMEBUFFER);
			assert(status == GL_FRAMEBUFFER_COMPLETE);
		
			return ret;
		}
		else
		{
			if (capturer->IsSetUpForCapture())
			{
				if (!capturer->RevertSetup())
				{
					ttv::trace::Message("BindingFrameCapturer", TTV_ML_ERROR, "BindingFrameCapturer_iOS_Swizzle_GLES2 could not revert capture setup");
				}
			}
		}
	}
	
	return [self twitch_presentRenderbuffer: target];
}


- (BOOL)twitch_renderbufferStorage:(NSUInteger)target fromDrawable:(id<EAGLDrawable>)drawable
{
	assert(target == GL_RENDERBUFFER);
	assert([EAGLContext currentContext] == self);
	
	std::shared_ptr<ttv::BindingFrameCapturer_iOS_Swizzle_GLES2> capturer = ttv::BindingFrameCapturer_iOS_Swizzle_GLES2::GetInstance();
	
	// cleanup
	if (drawable == nullptr)
	{
		// cleanup GLES objects
		if (capturer != nullptr && capturer->GetIsStarted()) {
			
			capturer->Stop();
		}
	}
	// bind to the given layer
	else
	{
		// get the id of the game's framebuffer's renderbuffer
		glGetIntegerv(GL_RENDERBUFFER_BINDING, &gGameRenderBufferId);
		assert(gGameRenderBufferId != 0);
	}
	
	// call the original method
	return [self twitch_renderbufferStorage:target fromDrawable:drawable];
}

@end

using namespace ttv::graphics::ios;
using namespace ttv::graphics::gl;

ttv::BindingFrameCapturer_iOS_Swizzle_GLES2::BindingFrameCapturer_iOS_Swizzle_GLES2()
:	mGameFrameBufferId(0)
,	mGameRenderBufferId(0)
,	mIsSetupForCapture(false)
,	mIsFrameSubmissionPaused(false)
{
	ttv::trace::Message("BindingFrameCapturer", TTV_ML_INFO, "BindingFrameCapturer_iOS_Swizzle_GLES2 created");
	
	[EAGLContext dummyForceClassLoad];
}


ttv::BindingFrameCapturer_iOS_Swizzle_GLES2::~BindingFrameCapturer_iOS_Swizzle_GLES2()
{
}


void ttv::BindingFrameCapturer_iOS_Swizzle_GLES2::CleanupObjects()
{
	BindingFrameCapturer_iOS_RenderTexture_GLES2::CleanupObjects();
	
	mScreenQuad.Destroy();
	mScreenQuadShader.Destroy();
	mScreenTexture.Destroy();
	mScreenFrameBuffer.Destroy();
}


TTV_ErrorCode ttv::BindingFrameCapturer_iOS_Swizzle_GLES2::Start(const TTV_VideoParams* videoParams, const TTV_AudioParams* audioParams, const TTV_IngestServer* ingestServer, uint32_t flags)
{
	TTV_ErrorCode ec = BindingFrameCapturer_iOS_RenderTexture_GLES2::Start(videoParams, audioParams, ingestServer, flags);
	
	if (TTV_SUCCEEDED(ec))
	{
		mIsFrameSubmissionPaused = false;
	}
	
	return ec;
}


TTV_ErrorCode ttv::BindingFrameCapturer_iOS_Swizzle_GLES2::Stop()
{
	TTV_ErrorCode ec = BindingFrameCapturer_iOS_RenderTexture_GLES2::Stop();
	
	if (TTV_SUCCEEDED(ec))
	{
		// Currently, all of this is handled by the superclass
	}
	
	return ec;
}


TTV_ErrorCode ttv::BindingFrameCapturer_iOS_Swizzle_GLES2::PauseFrameSubmission()
{
	if (!GetIsStarted() || mIsFrameSubmissionPaused)
	{
		return TTV_EC_STREAM_NOT_STARTED;
	}
	
	mIsFrameSubmissionPaused = true;
	
	return TTV_EC_SUCCESS;
}


TTV_ErrorCode ttv::BindingFrameCapturer_iOS_Swizzle_GLES2::ResumeFrameSubmission()
{
	if (!GetIsStarted())
	{
		return TTV_EC_STREAM_NOT_STARTED;
	}
	else if (!mIsFrameSubmissionPaused)
	{
		return TTV_EC_STREAM_ALREADY_STARTED;
	}
	
	mIsFrameSubmissionPaused = false;
	
	return TTV_EC_SUCCESS;
}


bool ttv::BindingFrameCapturer_iOS_Swizzle_GLES2::CheckError()
{
	GLenum ret = glGetError();
	
	switch (ret)
	{
		case GL_NO_ERROR:
		{
			break;
		}
		default:
		{
			ttv::trace::Message("OpenGL", TTV_ML_ERROR, "OpenGL error: %d", ret);
			break;
		}
	}
	
	return ret == GL_NO_ERROR;
}


TTV_ErrorCode ttv::BindingFrameCapturer_iOS_Swizzle_GLES2::CaptureBoundRenderbuffer()
{
	TTV_ErrorCode ec = TTV_EC_SUCCESS;
	
	// save state
	GLint previousFrameBufferId = 0;
	glGetIntegerv(GL_FRAMEBUFFER_BINDING, &previousFrameBufferId);
	assert(previousFrameBufferId == mGameFrameBufferId);
	
	// verify state
	GLint attachmentType = GL_NONE;
	glGetFramebufferAttachmentParameteriv(GL_FRAMEBUFFER, GL_COLOR_ATTACHMENT0, GL_FRAMEBUFFER_ATTACHMENT_OBJECT_TYPE, &attachmentType);
	assert(attachmentType == GL_TEXTURE);
	
	GLint previousRenderTextureBinding = 0;
	glGetFramebufferAttachmentParameteriv(GL_FRAMEBUFFER, GL_COLOR_ATTACHMENT0, GL_FRAMEBUFFER_ATTACHMENT_OBJECT_NAME, &previousRenderTextureBinding);
	assert(previousRenderTextureBinding == GLint (mScreenTexture.GetId()));
	
	// detach the screen texture from the game frame buffer
	glFramebufferTexture2D(GL_FRAMEBUFFER, GL_COLOR_ATTACHMENT0, GL_TEXTURE_2D, 0, 0);
	
	// send the screen texture to the SDK
	if (!mIsFrameSubmissionPaused)
	{
		ec = BindingFrameCapturer_iOS_RenderTexture_GLES2::SubmitTexture(reinterpret_cast<void*>(mScreenTexture.GetId()), mVideoParams.outputWidth, mVideoParams.outputHeight);
	}
	
	return ec;
}

TTV_ErrorCode ttv::BindingFrameCapturer_iOS_Swizzle_GLES2::DrawScreenQuad()
{
	TTV_ErrorCode ec = TTV_EC_SUCCESS;
	
	// bind the screen frame buffer
	glBindFramebuffer(GL_FRAMEBUFFER, mScreenFrameBuffer.GetId());
	
	// make sure the game render buffer is bound to the screen frame buffer
	glBindRenderbuffer(GL_RENDERBUFFER, mGameRenderBufferId);
	glFramebufferRenderbuffer(GL_FRAMEBUFFER, GL_COLOR_ATTACHMENT0, GL_RENDERBUFFER, mGameRenderBufferId);
	
	// TODO: We should test to see if there's a difference between capturing for the SDK before vs. after calling Apple's presentRenderbuffer method.
	
	// draw the screen texture to the game renderbuffer
	mScreenQuad.Draw(mScreenQuadShader, mScreenTexture.GetId());
	
	if (!CheckError())
	{
		ec = TTV_EC_GRAPHICS_API_ERROR;
	}
	
	return ec;
}

void ttv::BindingFrameCapturer_iOS_Swizzle_GLES2::BindGameFramebuffer()
{
	// restore state
	glBindFramebuffer(GL_FRAMEBUFFER, mGameFrameBufferId);
	
	// re-attach the screen texture to the Unity frame buffer
	glBindTexture(GL_TEXTURE_2D, mScreenTexture.GetId());
	glFramebufferTexture2D(GL_FRAMEBUFFER, GL_COLOR_ATTACHMENT0, GL_TEXTURE_2D, mScreenTexture.GetId(), 0);
	
	CheckError();
}

bool ttv::BindingFrameCapturer_iOS_Swizzle_GLES2::SetupCapture()
{
	if (mIsSetupForCapture)
	{
		return false;
	}
	
	assert(mGameFrameBufferId == 0);
	
	GLint gameRenderBufferWidth = 0;
	GLint gameRenderBufferHeight = 0;
	glGetRenderbufferParameteriv(GL_RENDERBUFFER,GL_RENDERBUFFER_WIDTH,&gameRenderBufferWidth);
	glGetRenderbufferParameteriv(GL_RENDERBUFFER,GL_RENDERBUFFER_HEIGHT,&gameRenderBufferHeight);
	
	// save state
	GLint previousTextureBinding = 0;
	glGetIntegerv(GL_TEXTURE_BINDING_2D, &previousTextureBinding);
	glGetIntegerv(GL_FRAMEBUFFER_BINDING, &mGameFrameBufferId);
	
	// make sure the game has bound the expected render buffer to its frame buffer
	glGetFramebufferAttachmentParameteriv(GL_FRAMEBUFFER, GL_COLOR_ATTACHMENT0, GL_FRAMEBUFFER_ATTACHMENT_OBJECT_NAME, &mGameRenderBufferId);
	assert(mGameRenderBufferId == gGameRenderBufferId);
	
	// create a quad and a separate shader to draw it to the screen
	mScreenQuadShader.Create();
	mScreenQuadShader.SetUniforms(gameRenderBufferWidth, gameRenderBufferHeight);
	mScreenQuad.Create(gameRenderBufferWidth, gameRenderBufferHeight,true);
	
	CheckError();
	
	// create the screen texture
	mTextureCache.CreateTexture(mScreenTexture, gameRenderBufferWidth, gameRenderBufferHeight, true);
	
	// set the game frame buffer to render to our screen texture
	glBindTexture(GL_TEXTURE_2D, mScreenTexture.GetId());
	glFramebufferTexture2D(GL_FRAMEBUFFER, GL_COLOR_ATTACHMENT0, GL_TEXTURE_2D, mScreenTexture.GetId(), 0);
	
	CheckError();
	
	// set the game render buffer as the target of rendering from our screen frame buffer
	mScreenFrameBuffer.Create();
	glBindFramebuffer(GL_FRAMEBUFFER, mScreenFrameBuffer.GetId());
	glFramebufferRenderbuffer(GL_FRAMEBUFFER, GL_COLOR_ATTACHMENT0, GL_RENDERBUFFER, gGameRenderBufferId);
	
	CheckError();
	
	// restore state
	glBindFramebuffer(GL_FRAMEBUFFER, mGameFrameBufferId);
	glBindTexture(GL_TEXTURE_2D, previousTextureBinding);
	
	CheckError();
	
	mIsSetupForCapture = true;
	
	ttv::trace::Message("BindingFrameCapturer", TTV_ML_INFO, "BindingFrameCapturer_iOS_Swizzle_GLES2 created screen framebuffer with ID: %d",mScreenFrameBuffer.GetId());
	ttv::trace::Message("BindingFrameCapturer", TTV_ML_INFO, "BindingFrameCapturer_iOS_Swizzle_GLES2 created screen texture with ID: %d",mScreenTexture.GetId());
	
	return true;
}

bool ttv::BindingFrameCapturer_iOS_Swizzle_GLES2::RevertSetup()
{
	if (!mIsSetupForCapture)
	{
		return false;
	}
	
	assert(mGameFrameBufferId != 0);
	assert(mGameRenderBufferId != 0);
	
	// revert game framebuffer renderbuffer attachment
	glBindFramebuffer(GL_FRAMEBUFFER, mGameFrameBufferId);
	glFramebufferRenderbuffer(GL_FRAMEBUFFER, GL_COLOR_ATTACHMENT0, GL_RENDERBUFFER, mGameFrameBufferId);
	
	CheckError();
	
	GLenum status = glCheckFramebufferStatus(GL_FRAMEBUFFER);
	assert(status == GL_FRAMEBUFFER_COMPLETE);
	
	mGameFrameBufferId = 0;
	mGameRenderBufferId = 0;
	mIsSetupForCapture = false;
	
	return true;
}

TTV_ErrorCode ttv::BindingFrameCapturer_iOS_Swizzle_GLES2::SubmitTexture(void* p, int width, int height)
{
	// this capture method doesn't use a render texture submitted by the game
	return TTV_EC_SUCCESS;
}

#endif
