/********************************************************************************************
* Twitch Broadcasting SDK
*
* This software is supplied under the terms of a license agreement with Twitch Interactive, Inc. and
* may not be copied or used except in accordance with the terms of that agreement
* Copyright (c) 2012-2016 Twitch Interactive, Inc.
*********************************************************************************************/

#include "internal/pch.h"
#include "internal/bindings/bindingframecapturer_opengl2.h"
#include "internal/bindings/graphics_common_gl.h"


#if TTV_SUPPORT_OPENGL2

#include "twitchsdk.h"
#include <thread>

using namespace ttv::graphics::opengl2;
using namespace ttv::graphics::gl;

#if TTV_PLATFORM_WINDOWS
bool ttv::BindingFrameCapturer_OpenGL2::sGlewInitialized = false;
#endif

// Seems like Unity only supports OpenGL 2.1 after doing some Googling so I'll make that assumption here.
// http://www.opengl.org/sdk/docs/man2/


ttv::BindingFrameCapturer_OpenGL2::Snapshot::Snapshot()
:	copied(nullptr)
,	size(0)
{
}


ttv::BindingFrameCapturer_OpenGL2::Snapshot::~Snapshot()
{
	ttv::gFreeCallback(copied);
	copied = nullptr;

	Destroy();
}


void ttv::BindingFrameCapturer_OpenGL2::Snapshot::CacheBuffer(void* locked, int bufferSize)
{
	if (copied == nullptr || size != bufferSize)
	{
		ttv::gFreeCallback(copied);
		copied = reinterpret_cast<uint8_t*>( ttv::AlignedAlloc(bufferSize, 16) );
	}

	memcpy(copied, locked, bufferSize);
}


void ttv::BindingFrameCapturer_OpenGL2::Snapshot::Destroy()
{
	rt.Destroy();
	buffer.Destroy();
	query.Destroy();
}


void ttv::BindingFrameCapturer_OpenGL2::Snapshot::Forget()
{
	rt.Forget();
	buffer.Forget();
	query.Forget();
}




ttv::BindingFrameCapturer_OpenGL2::BindingFrameCapturer_OpenGL2()
:	mRenderTextureCount(3)
,	mSceneWidth(0)
,	mSceneHeight(0)
,	mOpenGL2Supported(false)
{
	ttv::trace::Message("BindingFrameCapturer", TTV_ML_INFO, "BindingFrameCapturer_OpenGL2 created");

#if TTV_PLATFORM_WINDOWS

	// initialize glew if not yet initialized
	if (!sGlewInitialized)
	{
		GLenum err = glewInit();
		if (GLEW_OK == err)
		{
			sGlewInitialized = glewIsSupported("GL_VERSION_2_1") == GL_TRUE;
			sGlewInitialized = sGlewInitialized && (glewIsSupported("GL_ARB_framebuffer_object") == GL_TRUE);

			if (!sGlewInitialized)
			{
				ttv::trace::Message("BindingFrameCapturer", TTV_ML_ERROR, "OpenGL 2 not supported");
			}
		}
		else
		{
			sGlewInitialized = false;
			ttv::trace::Message("BindingFrameCapturer", TTV_ML_ERROR, "BindingFrameCapturer_OpenGL2 failed to initialize GLEW");
		}
	}

	mOpenGL2Supported = sGlewInitialized;

#else

	mOpenGL2Supported = true;

#endif
}


ttv::BindingFrameCapturer_OpenGL2::~BindingFrameCapturer_OpenGL2()
{
}


void ttv::BindingFrameCapturer_OpenGL2::CleanupObjects()
{
	// NOTE: Depending on the engine we're integrated into, we may not get a notification that the OpenGL context we were using
	// was destroyed (as is the case with Unity).  Unfortunately, we can only detect this by failing to make the context current.
	// If this happens, the assumption is that all the objects that were allocated in the context by us were destroyed along with
	// the context, which seems to be what documentation indicates.  The "forget" calls below are to handle the case when the 
	// context is no longer accessible.

	// set the context we expect in order to cleanup
	DeviceContext currentContext;
	currentContext.CreateFromCurrent();

	bool destroy = true;
	if (!mDeviceContext.IsCurrent())
	{
		destroy = mDeviceContext.Bind();
	}

	// destroy objects since the context is still valid
	if (destroy)
	{
		mResizeFrameBuffer.Destroy();
		mQuadShader.Destroy();
		mBroadcastQuad.Destroy();
	}
	// the context was destroyed and all objects released for us
	else
	{
		mResizeFrameBuffer.Forget();
		mQuadShader.Forget();
		mBroadcastQuad.Forget();

		for (size_t i = 0; i < mPendingTextures.size(); ++i)
		{
			if (mPendingTextures[i] != nullptr)
			{
				mPendingTextures[i]->Forget();
			}
		}

		for (size_t i = 0; i < mFreeTextures.size(); ++i)
		{
			mFreeTextures[i]->Forget();
		}

		for (size_t i = 0; i < mLockedTextures.size(); ++i)
		{
			mLockedTextures[i]->Forget();
		}
	}

	assert(mLockedTextures.empty());
	mFreeTextures.clear();
	mPendingTextures.clear();
	
	// restore the previous context
	currentContext.Bind();

	mDeviceContext.Clear();

	mSceneHeight = 0;
	mSceneWidth = 0;
}


void ttv::BindingFrameCapturer_OpenGL2::SetGraphicsDevice(void* /*device*/, GfxDeviceEventType::Enum /*eventType*/)
{
	// NOTE: there is no device or context passed in here
}


TTV_ErrorCode ttv::BindingFrameCapturer_OpenGL2::Start(const TTV_VideoParams* videoParams, const TTV_AudioParams* audioParams, const TTV_IngestServer* ingestServer, uint32_t flags)
{
	TTV_ErrorCode ec = BindingFrameCapturer::Start(videoParams, audioParams, ingestServer, flags);
	
	if (TTV_SUCCEEDED(ec))
	{
		CreateObjects();
	}
	
	return ec;
}


bool ttv::BindingFrameCapturer_OpenGL2::CreateObjects()
{
	CleanupObjects();

	mDeviceContext.CreateFromCurrent();
	
	mResizeFrameBuffer.Create();
	mQuadShader.Create();
		
	for (int i = 0; i < mRenderTextureCount; ++i)
	{
		std::shared_ptr<Snapshot> snapshot = std::make_shared<Snapshot>();
			
		snapshot->rt.Create(mVideoParams.outputWidth, mVideoParams.outputHeight);
		snapshot->buffer.Create();
		snapshot->query.Create();

		mFreeTextures.push_back(snapshot);
	}

	return true;
}


bool ttv::BindingFrameCapturer_OpenGL2::CheckError()
{
	return GLCHECK();
}


bool ttv::BindingFrameCapturer_OpenGL2::ShouldCreateFrameBuffers() const
{
	// we don't want to allocate buffers, we'll pass the locked texture data directly to the sdk
	return false;
}


std::shared_ptr<ttv::BindingFrameCapturer_OpenGL2::Snapshot> ttv::BindingFrameCapturer_OpenGL2::UnlockTexture(const void* buffer)
{
	auto iter = std::find_if(mLockedTextures.begin(), mLockedTextures.end(),
											  [buffer](std::shared_ptr<Snapshot>& rt)->bool { return rt->copied == buffer; });
	
	assert(iter != mLockedTextures.end());
	
	std::shared_ptr<Snapshot> snapshot;
	
	if (iter != mLockedTextures.end())
	{
		snapshot = *iter;
		mLockedTextures.erase(iter);
	}
	
	return snapshot;
}


void ttv::BindingFrameCapturer_OpenGL2::HandleBufferUnlock(const uint8_t* buffer)
{
	auto snapshot = UnlockTexture(buffer);
	
	if (snapshot != nullptr)
	{
		// handle the case where the context changes while the buffer was in the SDK
		if (snapshot->rt.GetId() != 0)
		{
			mFreeTextures.push_back(snapshot);
		}
	}
}


TTV_ErrorCode ttv::BindingFrameCapturer_OpenGL2::SubmitTexture(void* p, int width, int height)
{
	//std::thread::id tid = std::this_thread::get_id();
	//trace::Message("BindingFrameCapturer_OpenGL2", TTV_ML_ERROR, "SubmitTexture: %d", (int)tid.hash());

	if (!mOpenGL2Supported)
	{
		return TTV_EC_NOT_INITIALIZED;
	}
	else if (p == nullptr)
	{
		return  TTV_EC_INVALID_ARG;
	}
	else if (mResizeFrameBuffer.GetId() == 0)
	{
		return TTV_EC_NOT_INITIALIZED;
	}
	
	// cache the current context
	DeviceContext currentContext;
	currentContext.CreateFromCurrent();

	// make sure the expected context is current
	if (!mDeviceContext.IsCurrent())
	{
//		trace::Message("BindingFrameCapturer_OpenGL2", TTV_ML_ERROR, "current gl: %d, active gl: %d", (int)previousGLContext, (int)mActiveHGLRC);
//		trace::Message("BindingFrameCapturer_OpenGL2", TTV_ML_ERROR, "current device: %d, active device: %d", (int)previousDeviceContext, (int)mActiveHDC);

		// recreate objects in the new gl context
		CreateObjects();
	}
	
	TTV_ErrorCode err = TTV_EC_SUCCESS;

	GLuint sceneTextureId = static_cast<GLuint>( reinterpret_cast<uint64_t>(p) );

	// lock and submit the oldest frame
	if (mPendingTextures.size() > 0)
	{
		auto lockTexture = mPendingTextures.front();

		if (lockTexture->query.IsReady())
		{
			mPendingTextures.erase(mPendingTextures.begin());
		
			// the first entries will be null to cause some delay
			if (lockTexture != nullptr)
			{
				// get the pixels of the broadcast texture
				if (TTV_SUCCEEDED(err))
				{
					// save state
					GLint prevPixelPackBuffer = 0;
					glGetIntegerv(GL_PIXEL_PACK_BUFFER_BINDING, &prevPixelPackBuffer);

					glBindBuffer(GL_PIXEL_PACK_BUFFER_ARB, lockTexture->buffer.GetId());
					uint8_t* locked = reinterpret_cast<uint8_t*>( glMapBuffer(GL_PIXEL_PACK_BUFFER_ARB, GL_READ_ONLY) );
					assert(locked != nullptr);
				
					// NOTE: I would have loved to just pass the pointer to the locked buffer into the SDK directly but
					// we can't control the lifetime of the opengl context and it may be destroyed while the SDK 
					// is using the buffer.  Thus, we need to copy the buffer.

					if (locked != nullptr)
					{
						lockTexture->CacheBuffer(locked, mVideoParams.outputWidth * mVideoParams.outputHeight * 4);

						// release the lock
						glBindBuffer(GL_PIXEL_PACK_BUFFER_ARB, lockTexture->buffer.GetId());
						glUnmapBuffer(GL_PIXEL_PACK_BUFFER_ARB);
					}
					else
					{
						err = TTV_EC_GRAPHICS_API_ERROR;
					}
				
					// restore state
					glBindBuffer(GL_PIXEL_PACK_BUFFER, prevPixelPackBuffer);
				}
			
				// submit the frame
				if (TTV_SUCCEEDED(err))
				{
					mLockedTextures.push_back(lockTexture);
				
					err = TTV_SubmitVideoFrame(reinterpret_cast<uint8_t*>(lockTexture->copied), BufferUnlockCallback, static_cast<BindingFrameCapturer*>(this));
				
					if (TTV_FAILED(err))
					{
						UnlockTexture(lockTexture->copied);
					}
				}
			
				if (TTV_FAILED(err))
				{
					mFreeTextures.push_back(lockTexture);
				}
			}
		}
	}

	// capture the current frame
	if (TTV_SUCCEEDED(err) && mFreeTextures.size() > 0)
	{
		auto broadcastTexture = mFreeTextures.front();
		mFreeTextures.erase(mFreeTextures.begin());
		
		// cache state
		GLint prevFrameBufferId = 0;
		glGetIntegerv(GL_FRAMEBUFFER_BINDING, &prevFrameBufferId);
		if (!CheckError())
		{
			err = TTV_EC_GRAPHICS_API_ERROR;
		}
		
		// ensure the shader and quad are setup for the given scene texture size
		if (TTV_SUCCEEDED(err))
		{
			if (width != mSceneWidth || height != mSceneHeight)
			{
				float fWidth = static_cast<float>(width);
				float fHeight = static_cast<float>(height);

				mQuadShader.SetUniforms(fWidth, fHeight);
				if (!CheckError())
				{
					err = TTV_EC_GRAPHICS_API_ERROR;
				}

				if (TTV_SUCCEEDED(err))
				{
					mBroadcastQuad.Create(fWidth, fHeight, false);
					if (!CheckError())
					{
						err = TTV_EC_GRAPHICS_API_ERROR;
					}
				}

				if (TTV_SUCCEEDED(err))
				{
					mSceneWidth = width;
					mSceneHeight = height;
				}
			}
		}
		
		// bind the resize frame buffer
		if (TTV_SUCCEEDED(err))
		{
			// bind the resize frame buffer
			glBindFramebuffer(GL_FRAMEBUFFER, mResizeFrameBuffer.GetId());
			if (!CheckError())
			{
				err = TTV_EC_GRAPHICS_API_ERROR;
			}
		}
		
		// hook the broadcast texture up to the resize frame buffer
		if (TTV_SUCCEEDED(err))
		{
			glFramebufferTexture2D(GL_FRAMEBUFFER, GL_COLOR_ATTACHMENT0, GL_TEXTURE_2D, broadcastTexture->rt.GetId(), 0);
			if (!CheckError())
			{
				err = TTV_EC_GRAPHICS_API_ERROR;
			}
		}
		
		// make sure the frame buffer object is correctly configured
		if (TTV_SUCCEEDED(err))
		{
			if (glCheckFramebufferStatus(GL_FRAMEBUFFER) != GL_FRAMEBUFFER_COMPLETE)
			{
				err = TTV_EC_GRAPHICS_API_ERROR;
			}
		}
		
		// render the scene texture into the broadcast texture
		if (TTV_SUCCEEDED(err))
		{
			// cache the viewport
			GLint previousViewport[4];
			glGetIntegerv(GL_VIEWPORT, previousViewport);
			
			broadcastTexture->query.Begin();
			
			// clear to black
			broadcastTexture->rt.Clear(0, 0, 0, 0);
			
			// setup the viewport
			GLint viewport[4];
			CalculateViewport(width, height, mVideoParams.outputWidth, mVideoParams.outputHeight, viewport[0], viewport[1], viewport[2], viewport[3]);
			glViewport(viewport[0], viewport[1], viewport[2], viewport[3]);
			
			// render the scene
			mBroadcastQuad.Draw(mQuadShader, sceneTextureId);
			if (!CheckError())
			{
				err = TTV_EC_GRAPHICS_API_ERROR;
				broadcastTexture->query.End();
			}
			
			// restore the viewport
			glViewport(previousViewport[0], previousViewport[1], previousViewport[2], previousViewport[3]);
		}
		
		// trigger a DMA of texture data to main memory
		if (TTV_SUCCEEDED(err))
		{
			// save state
			GLint prevPixelPackBuffer = 0;
			glGetIntegerv(GL_PIXEL_PACK_BUFFER_BINDING, &prevPixelPackBuffer);
			
			glBindBuffer(GL_PIXEL_PACK_BUFFER, broadcastTexture->buffer.GetId());
			glBufferData(GL_PIXEL_PACK_BUFFER, mVideoParams.outputWidth * mVideoParams.outputHeight * 4 * sizeof(uint8_t), nullptr, GL_STREAM_READ);
			glReadPixels(0, 0, mVideoParams.outputWidth, mVideoParams.outputHeight, GL_RGBA, GL_UNSIGNED_BYTE, nullptr);
			
			// restore state
			glBindBuffer(GL_PIXEL_PACK_BUFFER, prevPixelPackBuffer);

			broadcastTexture->query.End();
		}
		
		// mark the texture as pending
		if (TTV_SUCCEEDED(err))
		{
			mPendingTextures.push_back(broadcastTexture);
		}
		else
		{
			mFreeTextures.push_back(broadcastTexture);
		}
		
		// restore state
		glBindFramebuffer(GL_FRAMEBUFFER, prevFrameBufferId);
	}

	return err;
}

#endif
