/********************************************************************************************
* Twitch Broadcasting SDK
*
* This software is supplied under the terms of a license agreement with Twitch Interactive, Inc. and
* may not be copied or used except in accordance with the terms of that agreement
* Copyright (c) 2012-2016 Twitch Interactive, Inc.
*********************************************************************************************/

#include "internal/pch.h"
#include "internal/bindings/bindings_stream.h"
#include <functional>

// TODO: the contents of this file should all be merged into the Java C++ code files since it's not shared with the C# implementation anymore

struct BindingRequestData
{
	void* result;
	TTV_BindingGenericCallback bindingCallback;
	std::function<TTV_ErrorCode()>* freeFunc;

	BindingRequestData()
		: result(nullptr)
		, bindingCallback(nullptr)
		, freeFunc(nullptr)
	{
	}
};


template<typename T> static void CleanupBindingRequestData(T* data)
{
	delete reinterpret_cast<T*>(data->freeFunc);
	data->result = nullptr;
	data->freeFunc = nullptr;
	data->bindingCallback = nullptr;
}


TTV_ErrorCode TTV_Binding_AllocateFrameBuffer(unsigned int size, void** buffer)
{
	*buffer = ttv::gAllocCallback(size, 16);

	if ((*buffer) == nullptr)
	{
		return TTV_EC_MEMORY;
	}

#if 0
	unsigned char* pDest = static_cast<unsigned char*>(*buffer);

	for (size_t i=0; i<size; ++i)
	{
		pDest[i] = (uint8_t)rand();
	}
#endif

	return TTV_EC_SUCCESS;
}


TTV_ErrorCode TTV_Binding_FreeFrameBuffer(void* buffer)
{
	ttv::gFreeCallback(buffer);
	return TTV_EC_SUCCESS;
}


TTV_ErrorCode TTV_Binding_RandomizeFrameBuffer(uint8_t* buffer, unsigned int size)
{
	if (buffer == nullptr)
	{
		return TTV_EC_INVALID_ARG;
	}

	for (unsigned int i=0; i<size; ++i)
	{
		buffer[i] = static_cast<uint8_t>(rand());
	}

	return TTV_EC_SUCCESS;
}


namespace
{
	void HandleBindingCallback(TTV_ErrorCode result, void* userData)
	{
		BindingRequestData* data = reinterpret_cast<BindingRequestData*>(userData);

		// pass the result to the binding
		if (data->bindingCallback != nullptr)
		{
			data->bindingCallback(result, data->result);
		}

		// call the free function if necessary
		if (data->freeFunc != nullptr)
		{
			(void)(*data->freeFunc)();
		}
	
		CleanupBindingRequestData(data);
	}	

	BindingRequestData gTTV_RequestAuthTokenRequest;
	BindingRequestData gTTV_LoginRequest;
	BindingRequestData gTTV_GetIngestServersRequest;
	BindingRequestData gTTV_GetUserInfoRequest;
	BindingRequestData gTTV_GetStreamInfoRequest;
	BindingRequestData gTTV_GetArchivingStateRequest;
	BindingRequestData gTTV_GetGameNameListRequest;
}


TTV_ErrorCode TTV_Binding_RequestAuthToken(const TTV_AuthParams* authParams, uint32_t flags, TTV_BindingGenericCallback callback)
{
	if (gTTV_RequestAuthTokenRequest.bindingCallback != nullptr) 
	{
		return TTV_EC_REQUEST_PENDING;
	}
	else if (callback == nullptr)
	{
		return TTV_EC_INVALID_ARG;
	}

	TTV_AuthToken* result = new TTV_AuthToken();
			
	gTTV_RequestAuthTokenRequest.result = reinterpret_cast<void*>(result);
	gTTV_RequestAuthTokenRequest.bindingCallback = callback;

	TTV_ErrorCode ret = TTV_RequestAuthToken(authParams, flags, HandleBindingCallback, &gTTV_RequestAuthTokenRequest, result);

	// call failed so abort
	if (TTV_FAILED(ret))
	{
		CleanupBindingRequestData(&gTTV_RequestAuthTokenRequest);
	}

	return ret;
}


TTV_ErrorCode TTV_Binding_Login(const TTV_AuthToken* authToken, TTV_BindingGenericCallback callback)
{
	if (gTTV_LoginRequest.bindingCallback != nullptr) 
	{
		return TTV_EC_REQUEST_PENDING;
	}
	else if (callback == nullptr)
	{
		return TTV_EC_INVALID_ARG;
	}

	TTV_ChannelInfo* result = new TTV_ChannelInfo();
	result->size = sizeof(TTV_ChannelInfo);

	gTTV_LoginRequest.result = reinterpret_cast<void*>(result);
	gTTV_LoginRequest.bindingCallback = callback;

	TTV_ErrorCode ret = TTV_Login(authToken, HandleBindingCallback, &gTTV_LoginRequest, result);

	// call failed so abort
	if (TTV_FAILED(ret))
	{
		CleanupBindingRequestData(&gTTV_LoginRequest);
	}

	return ret;
}


TTV_ErrorCode TTV_Binding_GetIngestServers(const TTV_AuthToken* authToken, TTV_BindingGenericCallback callback)
{
	if (gTTV_GetIngestServersRequest.bindingCallback != nullptr) 
	{
		return TTV_EC_REQUEST_PENDING;
	}
	else if (callback == nullptr)
	{
		return TTV_EC_INVALID_ARG;
	}

	TTV_IngestList* result = new TTV_IngestList();
			
	gTTV_GetIngestServersRequest.result = reinterpret_cast<void*>(result);
	gTTV_GetIngestServersRequest.bindingCallback = callback;
	gTTV_GetIngestServersRequest.freeFunc = new std::function<TTV_ErrorCode()>( std::bind(TTV_FreeIngestList, result) );

	TTV_ErrorCode ret = TTV_GetIngestServers(authToken, HandleBindingCallback, &gTTV_GetIngestServersRequest, result);

	// call failed so abort
	if (TTV_FAILED(ret))
	{
		CleanupBindingRequestData(&gTTV_GetIngestServersRequest);
	}

	return ret;
}


TTV_ErrorCode TTV_Binding_GetUserInfo(const TTV_AuthToken* authToken, TTV_BindingGenericCallback callback)
{
	if (gTTV_GetUserInfoRequest.bindingCallback != nullptr) 
	{
		return TTV_EC_REQUEST_PENDING;
	}
	else if (callback == nullptr)
	{
		return TTV_EC_INVALID_ARG;
	}

	TTV_UserInfo___* result = new TTV_UserInfo___();
	result->size = sizeof(*result);
			
	gTTV_GetUserInfoRequest.result = reinterpret_cast<void*>(result);
	gTTV_GetUserInfoRequest.bindingCallback = callback;

	TTV_ErrorCode ret = TTV_GetUserInfo(authToken, HandleBindingCallback, &gTTV_GetUserInfoRequest, result);

	// call failed so abort
	if (TTV_FAILED(ret))
	{
		CleanupBindingRequestData(&gTTV_GetUserInfoRequest);
	}

	return ret;
}


TTV_ErrorCode TTV_Binding_GetStreamInfo(const TTV_AuthToken* authToken, const char* channel, TTV_BindingGenericCallback callback)
{
	if (gTTV_GetStreamInfoRequest.bindingCallback != nullptr) 
	{
		return TTV_EC_REQUEST_PENDING;
	}
	else if (callback == nullptr)
	{
		return TTV_EC_INVALID_ARG;
	}

	TTV_StreamInfo* result = new TTV_StreamInfo();
	result->size = sizeof(TTV_StreamInfo);
			
	gTTV_GetStreamInfoRequest.result = reinterpret_cast<void*>(result);
	gTTV_GetStreamInfoRequest.bindingCallback = callback;

	TTV_ErrorCode ret = TTV_GetStreamInfo(authToken, HandleBindingCallback, &gTTV_GetStreamInfoRequest, channel, result);

	// call failed so abort
	if (TTV_FAILED(ret))
	{
		CleanupBindingRequestData(&gTTV_GetStreamInfoRequest);
	}

	return ret;
}


TTV_ErrorCode TTV_Binding_GetArchivingState(const TTV_AuthToken* authToken, TTV_BindingGenericCallback callback)
{
	if (gTTV_GetArchivingStateRequest.bindingCallback != nullptr) 
	{
		return TTV_EC_REQUEST_PENDING;
	}
	else if (callback == nullptr)
	{
		return TTV_EC_INVALID_ARG;
	}

	TTV_ArchivingState* result = new TTV_ArchivingState();
	result->size = sizeof(TTV_ArchivingState);
			
	gTTV_GetArchivingStateRequest.result = reinterpret_cast<void*>(result);
	gTTV_GetArchivingStateRequest.bindingCallback = callback;

	TTV_ErrorCode ret = TTV_GetArchivingState(authToken, HandleBindingCallback, &gTTV_GetArchivingStateRequest, result);

	// call failed so abort
	if (TTV_FAILED(ret))
	{
		CleanupBindingRequestData(&gTTV_GetArchivingStateRequest);
	}

	return ret;
}


TTV_ErrorCode TTV_Binding_GetGameNameList(const char* str, TTV_BindingGenericCallback callback)
{
	if (gTTV_GetGameNameListRequest.bindingCallback != nullptr) 
	{
		return TTV_EC_REQUEST_PENDING;
	}
	else if (callback == nullptr)
	{
		return TTV_EC_INVALID_ARG;
	}

	TTV_GameInfoList* result = new TTV_GameInfoList();
			
	gTTV_GetGameNameListRequest.result = reinterpret_cast<void*>(result);
	gTTV_GetGameNameListRequest.bindingCallback = callback;
	gTTV_GetGameNameListRequest.freeFunc = new std::function<TTV_ErrorCode()>( std::bind(TTV_FreeGameNameList, result) );

	TTV_ErrorCode ret = TTV_GetGameNameList(str, HandleBindingCallback, result, &gTTV_GetGameNameListRequest);

	// call failed so abort
	if (TTV_FAILED(ret))
	{
		CleanupBindingRequestData(&gTTV_GetGameNameListRequest);
	}

	return ret;
}
