/********************************************************************************************
* Twitch Broadcasting SDK
*
* This software is supplied under the terms of a license agreement with Twitch Interactive, Inc. and
* may not be copied or used except in accordance with the terms of that agreement
* Copyright (c) 2012-2016 Twitch Interactive, Inc.
*********************************************************************************************/

#include "internal/pch.h"
#include "internal/bindings/graphics_d3d11.h"

#if TTV_SUPPORT_D3D11

#include "twitchsdk.h"


ttv::graphics::d3d11::RenderTexture::RenderTexture()
:	mWidth(0)
,	mHeight(0)
,	mTexture(nullptr)
,	mRenderTargetView(nullptr)
{
}


ttv::graphics::d3d11::RenderTexture::~RenderTexture()
{
	Destroy();
}


bool ttv::graphics::d3d11::RenderTexture::Create(ID3D11Device* device, uint width, uint height)
{
	Destroy();
	
	// create the render texture
	D3D11_TEXTURE2D_DESC textureDesc;
	memset(&textureDesc, 0, sizeof(textureDesc));
	textureDesc.Width = width;
	textureDesc.Height = height;
	textureDesc.MipLevels = 1;
	textureDesc.ArraySize = 1;
	textureDesc.Format = DXGI_FORMAT_R8G8B8A8_UNORM;
	textureDesc.SampleDesc.Count = 1;
	textureDesc.Usage = D3D11_USAGE_DEFAULT;
	textureDesc.BindFlags = D3D11_BIND_RENDER_TARGET;
	textureDesc.CPUAccessFlags = 0;
	textureDesc.MiscFlags = 0;

	if ( FAILED(device->CreateTexture2D(&textureDesc, NULL, &mTexture)) )
	{
		return false;
	}

	// create the render target view
	D3D11_RENDER_TARGET_VIEW_DESC renderTargetViewDesc;
	memset(&renderTargetViewDesc, 0, sizeof(renderTargetViewDesc));
	renderTargetViewDesc.Format = textureDesc.Format;
	renderTargetViewDesc.ViewDimension = D3D11_RTV_DIMENSION_TEXTURE2D;
	renderTargetViewDesc.Texture2D.MipSlice = 0;

	if ( FAILED(device->CreateRenderTargetView(mTexture, &renderTargetViewDesc, &mRenderTargetView)) )
	{
		return false;
	}

	mWidth = width;
	mHeight = height;

	return true;
}


void ttv::graphics::d3d11::RenderTexture::Destroy()
{
	SAFE_RELEASE(mRenderTargetView);
	SAFE_RELEASE(mTexture);
	
	mWidth = 0;
	mHeight = 0;
}


void ttv::graphics::d3d11::RenderTexture::Bind(ID3D11DeviceContext* context)
{
	context->OMSetRenderTargets(1, &mRenderTargetView, nullptr);
}


void ttv::graphics::d3d11::RenderTexture::Clear(ID3D11DeviceContext* context, float r, float g, float b, float a)
{
	FLOAT color[4] = { r, g, b, a };
	context->ClearRenderTargetView(mRenderTargetView, color);
}


ttv::graphics::d3d11::StagingTexture::StagingTexture()
:	mWidth(0)
,	mHeight(0)
,	mTexture(nullptr)
{
}


ttv::graphics::d3d11::StagingTexture::~StagingTexture()
{
	Destroy();
}


bool ttv::graphics::d3d11::StagingTexture::Create(ID3D11Device* device, uint width, uint height)
{
	Destroy();
	
	// create the render texture
	D3D11_TEXTURE2D_DESC textureDesc;
	memset(&textureDesc, 0, sizeof(textureDesc));
	textureDesc.Width = width;
	textureDesc.Height = height;
	textureDesc.MipLevels = 1;
	textureDesc.ArraySize = 1;
	textureDesc.Format = DXGI_FORMAT_R8G8B8A8_UNORM;
	textureDesc.SampleDesc.Count = 1;
	textureDesc.Usage = D3D11_USAGE_STAGING;
	textureDesc.BindFlags = 0;
	textureDesc.CPUAccessFlags = D3D11_CPU_ACCESS_READ;
	textureDesc.MiscFlags = 0;

	if ( FAILED(device->CreateTexture2D(&textureDesc, NULL, &mTexture)) )
	{
		return false;
	}

	mWidth = width;
	mHeight = height;

	return true;
}


void ttv::graphics::d3d11::StagingTexture::Destroy()
{
	SAFE_RELEASE(mTexture);
	
	mWidth = 0;
	mHeight = 0;
}


ttv::graphics::d3d11::QuadShader::QuadShader()
:	mVertexShader(nullptr)
,	mPixelShader(nullptr)
,	mVertexConstantsBuffer(nullptr)
,	mVertexLayout(nullptr)
,	mSamplerState(nullptr)
{
}


bool ttv::graphics::d3d11::QuadShader::Create(ID3D11Device* device)
{
	// NOTE: these include files were auto-generated by an offline shader compiler - see the files under twitchsdk\bindings\directx
	
	{
		// create the vertex shader
		#include "internal/bindings/quadshader_vs_d3d11.h"
		const void* vertexBinary = reinterpret_cast<const void*>(g_main);
		if ( FAILED(device->CreateVertexShader(vertexBinary, sizeof(g_main), nullptr, &mVertexShader)) )
		{
			Destroy();
			return false;
		}

		// create the vertex layout
		D3D11_INPUT_ELEMENT_DESC vertexLayout[1];
		vertexLayout[0].SemanticName = "POSITION";
		vertexLayout[0].SemanticIndex = 0;
		vertexLayout[0].Format = DXGI_FORMAT_R32G32B32A32_FLOAT;
		vertexLayout[0].InputSlot = 0;
		vertexLayout[0].AlignedByteOffset = 0;
		vertexLayout[0].InputSlotClass = D3D11_INPUT_PER_VERTEX_DATA;
		vertexLayout[0].InstanceDataStepRate = 0;

		if ( FAILED(device->CreateInputLayout(vertexLayout, 1, vertexBinary, sizeof(g_main), &mVertexLayout)) )
		{
			return false;
		}

		// create the vertex shader contants buffer
		D3D11_BUFFER_DESC matrixBufferDesc;
		matrixBufferDesc.Usage = D3D11_USAGE_DYNAMIC;
		matrixBufferDesc.ByteWidth = sizeof(VertexShaderConstantsBuffer);
		matrixBufferDesc.BindFlags = D3D11_BIND_CONSTANT_BUFFER;
		matrixBufferDesc.CPUAccessFlags = D3D11_CPU_ACCESS_WRITE;
		matrixBufferDesc.MiscFlags = 0;
		matrixBufferDesc.StructureByteStride = 0;

		if ( FAILED(device->CreateBuffer(&matrixBufferDesc, NULL, &mVertexConstantsBuffer)) )
		{
			Destroy();
			return false;
		}
	}

	{
		// create the pixel shader
		#include "internal/bindings/quadshader_ps_d3d11.h"
		const void* pixelBinary = reinterpret_cast<const void*>(g_main);
		if ( FAILED(device->CreatePixelShader(pixelBinary, sizeof(g_main), nullptr, &mPixelShader)) )
		{
			Destroy();
			return false;
		}

		// setup the texture sampler
		D3D11_SAMPLER_DESC samplerDesc;
		samplerDesc.Filter = D3D11_FILTER_MIN_MAG_MIP_LINEAR;
		samplerDesc.AddressU = D3D11_TEXTURE_ADDRESS_CLAMP;
		samplerDesc.AddressV = D3D11_TEXTURE_ADDRESS_CLAMP;
		samplerDesc.AddressW = D3D11_TEXTURE_ADDRESS_CLAMP;
		samplerDesc.MipLODBias = 0.0f;
		samplerDesc.MaxAnisotropy = 1;
		samplerDesc.ComparisonFunc = D3D11_COMPARISON_ALWAYS;
		samplerDesc.BorderColor[0] = 0;
		samplerDesc.BorderColor[1] = 0;
		samplerDesc.BorderColor[2] = 0;
		samplerDesc.BorderColor[3] = 0;
		samplerDesc.MinLOD = 0;
		samplerDesc.MaxLOD = D3D11_FLOAT32_MAX;

		if ( FAILED(device->CreateSamplerState(&samplerDesc, &mSamplerState)) )
		{
			return false;
		}
	}

	return true;
}


void ttv::graphics::d3d11::QuadShader::Destroy()
{
	SAFE_RELEASE(mVertexConstantsBuffer);
	SAFE_RELEASE(mVertexShader);
	SAFE_RELEASE(mPixelShader);
	SAFE_RELEASE(mVertexLayout);
	SAFE_RELEASE(mSamplerState);
}


void ttv::graphics::d3d11::QuadShader::SetUniforms(ID3D11DeviceContext* context, float width, float height, ID3D11ShaderResourceView* texture)
{
	// set the ortho projection uniform
	float nearPlane = 1;
	float farPlane = 3;

	float mvp[4][4] =
	{
		{ 2.0f / width,	0.0f,			0.0f,											0 },
		{ 0.0f,			2 / -height,	0,												0 },
		{ 0.0f,			0.0f,			-2.0f / (farPlane-nearPlane),					0 },
		{ -1.0f,		1.0f,			-(farPlane+nearPlane) / (farPlane-nearPlane),	1 }
	};

	// transpose
	for (int y = 0; y < 4; ++y)
	{
		for (int x = y+1; x < 4; ++x)
		{
			float temp = mvp[x][y];
			mvp[x][y] = mvp[y][x];
			mvp[y][x] = temp;
		}
	}

	D3D11_MAPPED_SUBRESOURCE mappedResource;

	// set the constants in the vertex shader
	if ( FAILED(context->Map(mVertexConstantsBuffer, 0, D3D11_MAP_WRITE_DISCARD, 0, &mappedResource)) )
	{
		return;
	}

	VertexShaderConstantsBuffer* constants = static_cast<VertexShaderConstantsBuffer*>(mappedResource.pData);
	memcpy(constants->mvpMatrix, mvp, sizeof(constants->mvpMatrix));

	context->Unmap(mVertexConstantsBuffer, 0);
	context->VSSetConstantBuffers(0, 1, &mVertexConstantsBuffer);

	// set the texture in the pixel shader
	context->PSSetShaderResources(0, 1, &texture);
}


void ttv::graphics::d3d11::QuadShader::Bind(ID3D11DeviceContext* context)
{
	context->IASetInputLayout(mVertexLayout);

	context->VSSetShader(mVertexShader, nullptr, 0);
	context->PSSetShader(mPixelShader, nullptr, 0);

	context->PSSetSamplers(0, 1, &mSamplerState);
}


ttv::graphics::d3d11::Quad::Quad()
:	mWidth(0)
,	mHeight(0)
,	mVertexBuffer(nullptr)
{
}


ttv::graphics::d3d11::Quad::~Quad()
{
	Destroy();
}


void ttv::graphics::d3d11::Quad::Create(ID3D11Device* device, float width, float height)
{
	Destroy();
	
	mWidth = width;
	mHeight = height;
	
	// x, y, tx, ty
	const VertexType quad[] =
	{
		{ 0, 0,				0, 1 },
		{ 0, height,		0, 0 },
		{ width, height,	1, 0 },
		
		{ 0, 0,				0, 1 },
		{ width, height,	1, 0 },
		{ width, 0,			1, 1 }
	};

	// create the vertex buffer
	D3D11_SUBRESOURCE_DATA vertexData;
	vertexData.pSysMem = quad;
	vertexData.SysMemPitch = 0;
	vertexData.SysMemSlicePitch = 0;

	D3D11_BUFFER_DESC desc;
	desc.Usage = D3D11_USAGE_DEFAULT;
	desc.ByteWidth = sizeof(VertexType) * kNumQuadVertices;
	desc.BindFlags = D3D11_BIND_VERTEX_BUFFER;
	desc.CPUAccessFlags = 0;
	desc.MiscFlags = 0;
	desc.StructureByteStride = 0;

	if ( FAILED(device->CreateBuffer(&desc, &vertexData, &mVertexBuffer)) )
	{
		Destroy();
	}
}


void ttv::graphics::d3d11::Quad::Destroy()
{
	SAFE_RELEASE(mVertexBuffer);
}


void ttv::graphics::d3d11::Quad::Draw(ID3D11DeviceContext* context, QuadShader& shader)
{
	shader.Bind(context);

	UINT stride = sizeof(VertexType);
	UINT offset = 0;
	
	context->IASetVertexBuffers(0, 1, &mVertexBuffer, &stride, &offset);
	context->IASetPrimitiveTopology(D3D11_PRIMITIVE_TOPOLOGY_TRIANGLELIST);

	context->Draw(kNumQuadVertices, 0);
}


ttv::graphics::d3d11::Query::Query()
:	mQuery(nullptr)
{
}


ttv::graphics::d3d11::Query::~Query()
{
	Destroy();
}

				
bool ttv::graphics::d3d11::Query::Create(ID3D11Device* device)
{
	D3D11_QUERY_DESC desc;
	desc.MiscFlags = 0;
	desc.Query = D3D11_QUERY_EVENT;

	if ( FAILED(device->CreateQuery(&desc, &mQuery)) )
	{
		return false;
	}

	return true;
}


void ttv::graphics::d3d11::Query::Destroy()
{
	SAFE_RELEASE(mQuery);
}


void ttv::graphics::d3d11::Query::Mark(ID3D11DeviceContext* context)
{
	context->End(mQuery);
}


bool ttv::graphics::d3d11::Query::IsReady(ID3D11DeviceContext* context)
{
	return context->GetData(mQuery, nullptr, 0, 0) == S_OK;
}


#endif
