/********************************************************************************************
* Twitch Broadcasting SDK
*
* This software is supplied under the terms of a license agreement with Twitch Interactive, Inc. and
* may not be copied or used except in accordance with the terms of that agreement
* Copyright (c) 2012-2016 Twitch Interactive, Inc.
*********************************************************************************************/

#include "internal/pch.h"
#include "internal/bindings/graphics_ios_gles2.h"


#if TTV_SUPPORT_GLES2

#include "twitchsdk.h"
#import <objc/runtime.h>
#import <QuartzCore/QuartzCore.h>
#import <CoreVideo/CoreVideo.h>

ttv::graphics::ios::TextureCache::TextureCache()
:	mTextureCache(nullptr)
{

}


ttv::graphics::ios::TextureCache::~TextureCache()
{
	Destroy();
}


bool ttv::graphics::ios::TextureCache::Create(EAGLContext* context)
{
	assert(mTextureCache == nullptr);

	if (mTextureCache == nullptr)
	{
		CVReturn err = CVOpenGLESTextureCacheCreate(kCFAllocatorDefault, nullptr, context, nullptr, &mTextureCache);

		assert(err == kCVReturnSuccess);
		return err != kCVReturnSuccess;
	}
	else
	{
		return false;
	}
}


void ttv::graphics::ios::TextureCache::Destroy()
{
	if (mTextureCache != nullptr)
	{
		CFRelease(mTextureCache);

		mTextureCache = nullptr;
	}
}


bool ttv::graphics::ios::TextureCache::CreateTexture(RenderTexture& rt, uint width, uint height, bool fromCache)
{
	rt.Destroy();

	// save state
	GLint previousTextureBinding = 0;
	glGetIntegerv(GL_TEXTURE_BINDING_2D, &previousTextureBinding);

	GLuint textureId = 0;

	if (fromCache)
	{
		CVPixelBufferRef pb = CreatePixelBuffer(width, height);
		assert(pb != nullptr);

		if (pb != nullptr)
		{
			CVOpenGLESTextureRef tx = AllocateTextureFromCache(width, height, pb);
			assert(tx != nullptr);

			if (tx == nullptr)
			{
				CFRelease(pb);
				return false;
			}

			rt.Create(width, height, pb, tx);

			textureId = CVOpenGLESTextureGetName(tx);
			assert(textureId != 0);
		}
	}
	else
	{
		glGenTextures(1, &textureId);
		glBindTexture(GL_TEXTURE_2D, textureId);

		GLCHECK();

		rt.Create(width, height, textureId);
	}

	// set texture properties
	glBindTexture(GL_TEXTURE_2D, textureId);

	if (!fromCache)
	{
		glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, width, height, 0, GL_RGBA, GL_UNSIGNED_BYTE, nullptr);
	}

	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);

	// a requirement for non-power of 2 textures on iOS
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);

	// restore state
	glBindTexture(GL_TEXTURE_2D, previousTextureBinding);

	return true;
}


CVPixelBufferRef ttv::graphics::ios::TextureCache::CreatePixelBuffer(uint width, uint height)
{
	CVPixelBufferRef pixelBuffer = nullptr;

	CFDictionaryRef empty = CFDictionaryCreate(kCFAllocatorDefault, nullptr, nullptr, 0, &kCFTypeDictionaryKeyCallBacks, &kCFTypeDictionaryValueCallBacks);
	CFMutableDictionaryRef attrs = CFDictionaryCreateMutable(kCFAllocatorDefault, 1, &kCFTypeDictionaryKeyCallBacks, &kCFTypeDictionaryValueCallBacks);
	CFDictionarySetValue(attrs, kCVPixelBufferIOSurfacePropertiesKey, empty);

	CVReturn err = CVPixelBufferCreate(kCFAllocatorDefault, width, height, kCVPixelFormatType_32BGRA, attrs, &pixelBuffer);
	if (err != kCVReturnSuccess)
	{
		NSLog(@"Failed to create CVPixelBuffer");
		assert(false);
	}

	CFRelease(attrs);
	CFRelease(empty);

	return pixelBuffer;
}


CVOpenGLESTextureRef ttv::graphics::ios::TextureCache::AllocateTextureFromCache(uint width, uint height, CVPixelBufferRef pixelBuffer)
{
	CVOpenGLESTextureRef texture = nullptr;
	CVReturn err = CVOpenGLESTextureCacheCreateTextureFromImage(kCFAllocatorDefault,
		mTextureCache,
		pixelBuffer,
		nullptr, // texture attributes
		GL_TEXTURE_2D,
		GL_RGBA, // opengl format
		width,
		height,
		GL_BGRA, // native iOS format
		GL_UNSIGNED_BYTE,
		0,
		&texture);

	if (err != kCVReturnSuccess)
	{
		NSLog(@"Could not create render texture");
		assert(false);
		return nullptr;
	}

	return texture;
}


void ttv::graphics::ios::TextureCache::Flush()
{
	CVOpenGLESTextureCacheFlush(mTextureCache, 0);
}



ttv::graphics::ios::RenderTexture::RenderTexture()
:	mPixelBuffer(nullptr)
	,	mTexture(nullptr)
	,	mTextureId(0)
	,	mWidth(0)
	,	mHeight(0)
	,	mLocked(nullptr)
	,	mFromCache(false)
{
}


ttv::graphics::ios::RenderTexture::~RenderTexture()
{
	Destroy();
}


void ttv::graphics::ios::RenderTexture::Create(uint width, uint height, GLuint textureId)
{
	Destroy();

	mFromCache = false;
	mWidth = width;
	mHeight = height;
	mTextureId = textureId;
}


void ttv::graphics::ios::RenderTexture::Create(uint width, uint height, CVPixelBufferRef pb, CVOpenGLESTextureRef tx)
{
	Destroy();

	mFromCache = true;
	mWidth = width;
	mHeight = height;
	mPixelBuffer = pb;
	mTexture = tx;
}


void ttv::graphics::ios::RenderTexture::Destroy()
{
	assert(mLocked == nullptr);

	if (mFromCache)
	{
		if (mPixelBuffer != nullptr)
		{
			CFRelease(mPixelBuffer);

			mPixelBuffer = nullptr;
		}

		if (mTexture != nullptr)
		{
			CFRelease(mTexture);

			mTexture = nullptr;
		}
	}
	else
	{
		if (mTextureId != 0)
		{
			glDeleteTextures(1, &mTextureId);

			mTextureId = 0;
		}
	}

	mWidth = 0;
	mHeight = 0;
}


void* ttv::graphics::ios::RenderTexture::Lock()
{
	assert(mLocked == nullptr);
	assert(mFromCache);
	assert(mTexture != nullptr);

	if (mLocked == nullptr && mFromCache)
	{
		GLuint textureType = CVOpenGLESTextureGetTarget(mTexture);
		GLuint textureName = CVOpenGLESTextureGetName(mTexture);
		glBindTexture(textureType, textureName);

		CVReturn ret = CVPixelBufferLockBaseAddress(mPixelBuffer, kCVPixelBufferLock_ReadOnly);
		assert(kCVReturnSuccess == ret);

		if (kCVReturnSuccess == ret)
		{
			mLocked = CVPixelBufferGetBaseAddress(mPixelBuffer);
		}
	}

	return mLocked;
}


void ttv::graphics::ios::RenderTexture::Unlock()
{
	assert(mLocked != nullptr);
	assert(mFromCache);

	if (mLocked != nullptr && mFromCache)
	{
		CVPixelBufferUnlockBaseAddress(mPixelBuffer, kCVPixelBufferLock_ReadOnly);

		mLocked = nullptr;
	}
}


void ttv::graphics::ios::RenderTexture::Clear(float r, float g, float b, float a)
{
	// save state
	GLint previousTextureBinding = 0;
	glGetIntegerv(GL_TEXTURE_BINDING_2D, &previousTextureBinding);

	glBindTexture(GL_TEXTURE_2D, GetId());

	glClearColor(r, g, b, a);
	glClear(GL_COLOR_BUFFER_BIT);

	// restore state
	glBindTexture(GL_TEXTURE_2D, previousTextureBinding);
}


void ttv::graphics::ios::RenderTexture::ManuallyClearTexture()
{
	static int n = 0;

	// save state
	GLint previousTextureBinding = 0;
	glGetIntegerv(GL_TEXTURE_BINDING_2D, &previousTextureBinding);

	glBindTexture(GL_TEXTURE_2D, GetId());

	char* pixels = new char[4 * mWidth * mHeight];
	int index = 0;

	float fHeight = static_cast<float>(mHeight);
	for (uint y = 0; y < mHeight; ++y)
	{
		float debugValue = 255.0f * static_cast<float>(y) / static_cast<float>(mHeight-1);

		for (uint x = 0; x < mWidth; ++x)
		{
			pixels[index+0] = 0;
			pixels[index+1] = 0;
			pixels[index+2] = 0;
			pixels[index+3] = 255;

			pixels[index+n] = static_cast<char>(debugValue);

			index += 4;
		}
	}

	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, mWidth, mHeight, 0, GL_RGBA, GL_UNSIGNED_BYTE, pixels);
	delete [] pixels;

	n = (n+1) % 3;

	// restore state
	glBindTexture(GL_TEXTURE_2D, previousTextureBinding);
}


GLuint ttv::graphics::ios::RenderTexture::GetId()
{
	if (mFromCache)
	{
		return mTexture ? CVOpenGLESTextureGetName(mTexture) : 0;
	}
	else
	{
		return mTextureId;
	}
}


bool ttv::graphics::ios::GLCHECK()
{
	GLenum ret = glGetError();

	switch (ret)
	{
	case GL_NO_ERROR:
		{
			break;
		}
	default:
		{
			ttv::trace::Message("OpenGL", TTV_ML_ERROR, "OpenGL error: %d", ret);
			break;
		}
	}

	assert(ret == GL_NO_ERROR);
	return ret == GL_NO_ERROR;
}



ttv::graphics::ios::QuadShader::QuadShader()
:	mMvpLocation(0)
	,	mTextureLocation(0)
	,	mPosTexLocation(0)
	,	mProgramId(0)
{
}


bool ttv::graphics::ios::QuadShader::Create()
{
	const char* vertexSource =
		"uniform mediump mat4 u_MVP;"
		""
		"attribute mediump vec4 a_PosTex;"
		""
		"varying mediump vec2 v_TexCoord;"
		""
		"void main()"
		"{"
		"	gl_Position = u_MVP * vec4(a_PosTex.x, a_PosTex.y, -2, 1);"
		"	v_TexCoord = a_PosTex.zw;"
		"}";

	const char* fragmentSource =
		"uniform sampler2D u_Texture;"
		""
		"varying mediump vec2 v_TexCoord;"
		""
		"void main()"
		"{"
		"	gl_FragColor = texture2D(u_Texture, v_TexCoord.xy);"
		""
		"	gl_FragColor.a = 1.0;"
		"}";

	GLint status = GL_TRUE;

	GLuint vertexShaderId = 0;
	GLuint fragmentShaderId = 0;

	// compile the vertex shader
	if (status == GL_TRUE)
	{
		vertexShaderId = glCreateShader(GL_VERTEX_SHADER);
		glShaderSource(vertexShaderId, 1, &vertexSource, NULL);
		glCompileShader(vertexShaderId);

		glGetShaderiv(vertexShaderId, GL_COMPILE_STATUS, &status);
	}

	// compile the fragment shader
	if (status == GL_TRUE)
	{
		fragmentShaderId = glCreateShader(GL_FRAGMENT_SHADER);
		glShaderSource(fragmentShaderId, 1, &fragmentSource, NULL);
		glCompileShader(fragmentShaderId);

		glGetShaderiv(fragmentShaderId, GL_COMPILE_STATUS, &status);
	}

	// create and link the program
	if (status == GL_TRUE)
	{
		mProgramId = glCreateProgram();

		glAttachShader(mProgramId, vertexShaderId);
		glAttachShader(mProgramId, fragmentShaderId);
		glLinkProgram(mProgramId);

		glGetProgramiv(mProgramId, GL_LINK_STATUS, &status);
	}

	// free the shaders, the program will retain the last reference to these
	if (vertexShaderId != 0)
	{
		glDeleteShader(vertexShaderId);
	}
	if (fragmentShaderId != 0)
	{
		glDeleteShader(fragmentShaderId);
	}

	if (status == GL_TRUE)
	{
		// extract uniform and attribute locations
		mMvpLocation = glGetUniformLocation(mProgramId, "u_MVP");
		mTextureLocation = glGetUniformLocation(mProgramId, "u_Texture");
		mPosTexLocation = glGetAttribLocation(mProgramId, "a_PosTex");
	}
	else
	{
		//		char log[256];
		//		GLsizei len = 0;
		//		glGetShaderInfoLog(vertexShaderId, sizeof(log), &len, log);
		//		
		//		glDeleteShader(vertexShaderId);

		Destroy();

		assert(false);
		return false;
	}

	GLCHECK();

	return true;
}


void ttv::graphics::ios::QuadShader::Destroy()
{
	if (mProgramId != 0)
	{
		glDeleteProgram(mProgramId);

		mProgramId = 0;
		mMvpLocation = 0;
		mTextureLocation = 0;
		mPosTexLocation = 0;
	}
}


void ttv::graphics::ios::QuadShader::SetUniforms(float width, float height)
{
	assert(mProgramId != 0);

	// set the ortho projection uniform
	float nearPlane = 1;
	float farPlane = 3;

	float mvp[] =
	{
		2.0f / width,	0.0f,			0.0f,											0.0f,
		0.0f,			2 / -height,	0,												0,
		0.0f,			0.0f,			-2.0f / (farPlane-nearPlane),					0.0f,
		-1.0f,			1.0f,			-(farPlane+nearPlane) / (farPlane-nearPlane),	1.0f
	};

	// save state
	GLint previousProgramId = 0;
	glGetIntegerv(GL_CURRENT_PROGRAM, &previousProgramId);

	// set the mvp value
	glUseProgram(mProgramId);
	glUniformMatrix4fv(mMvpLocation, 1, GL_FALSE, mvp);

	// set the texture location
	glUniform1i(mTextureLocation, 0);

	// restore state
	glUseProgram(previousProgramId);

	GLCHECK();
}



ttv::graphics::ios::Quad::Quad()
:	mQuadAttributeArray(0)
,	mWidth(0)
,	mHeight(0)
{
}


ttv::graphics::ios::Quad::~Quad()
{
	Destroy();
}


void ttv::graphics::ios::Quad::Create(float width, float height, bool verticalFlip)
{
	Destroy();

	mWidth = width;
	mHeight = height;

	// x, y, tx, ty
	float quad[] =
	{
		0, 0,				0, 0,
		0, height,			0, 1,
		width, height,		1, 1,

		0, 0,				0, 0,
		width, height,		1, 1,
		width, 0,			1, 0
	};

	if (verticalFlip)
	{
		for (int i = 0; i < kNumQuadVertices * 4; i += 4)
		{
			quad[i+3] = 1 - quad[i+3];
		}
	}
	
	GLCHECK();

	// cache state
	GLint previousBufferBinding = 0;
	glGetIntegerv(GL_ARRAY_BUFFER_BINDING, &previousBufferBinding);
	GLCHECK();

	glGenBuffers(1, &mQuadAttributeArray);
	GLCHECK();

	glBindBuffer(GL_ARRAY_BUFFER, mQuadAttributeArray);
	GLCHECK();

	glBufferData(GL_ARRAY_BUFFER, sizeof(quad), quad, GL_STATIC_DRAW);
	GLCHECK();

	// restore state
	glBindBuffer(GL_ARRAY_BUFFER, previousBufferBinding);
	GLCHECK();
}


void ttv::graphics::ios::Quad::Destroy()
{
	if (mQuadAttributeArray != 0)
	{
		glDeleteBuffers(1, &mQuadAttributeArray);

		mQuadAttributeArray = 0;
	}
}


void ttv::graphics::ios::Quad::Draw(const QuadShader& shader, GLint inputTextureId)
{
	assert(glCheckFramebufferStatus(GL_FRAMEBUFFER) == GL_FRAMEBUFFER_COMPLETE);
	assert(shader.mProgramId != 0);
	assert(inputTextureId != 0);

	GLCHECK();

	// cache state
	GLint previousProgramId = 0;
	glGetIntegerv(GL_CURRENT_PROGRAM, &previousProgramId);

	GLint previousBufferBinding = 0;
	glGetIntegerv(GL_ARRAY_BUFFER_BINDING, &previousBufferBinding);

	GLint previousDepthTest = 0;
	glGetIntegerv(GL_DEPTH_TEST, &previousDepthTest);

	GLint previousCullFace = 0;
	glGetIntegerv(GL_CULL_FACE, &previousCullFace);

	GLint previousActiveTextureUnit = 0;
	glGetIntegerv(GL_ACTIVE_TEXTURE, &previousActiveTextureUnit);

	GLint previousTextureBinding = 0;
	glGetIntegerv(GL_TEXTURE_BINDING_2D, &previousTextureBinding);

	const int kNumVertexAttributes = 8;
	GLint previousAttribEnabled[kNumVertexAttributes];
	for (int i = 0; i < kNumVertexAttributes; ++i)
	{
		glGetVertexAttribiv(i, GL_VERTEX_ATTRIB_ARRAY_ENABLED, &previousAttribEnabled[i]);
	}

	GLint previousAttribBuffer;
	GLint previousAttribSize;
	GLint previousAttribStride;
	GLint previousAttribType;
	GLint previousAttribNormalized;
	GLvoid *previousAttribPointer;
	glGetVertexAttribPointerv(shader.mPosTexLocation, GL_VERTEX_ATTRIB_ARRAY_POINTER, &previousAttribPointer);
	glGetVertexAttribiv(shader.mPosTexLocation, GL_VERTEX_ATTRIB_ARRAY_BUFFER_BINDING, &previousAttribBuffer);
	glGetVertexAttribiv(shader.mPosTexLocation, GL_VERTEX_ATTRIB_ARRAY_SIZE, &previousAttribSize);
	glGetVertexAttribiv(shader.mPosTexLocation, GL_VERTEX_ATTRIB_ARRAY_STRIDE, &previousAttribStride);
	glGetVertexAttribiv(shader.mPosTexLocation, GL_VERTEX_ATTRIB_ARRAY_TYPE, &previousAttribType);
	glGetVertexAttribiv(shader.mPosTexLocation, GL_VERTEX_ATTRIB_ARRAY_NORMALIZED, &previousAttribNormalized);

	// set desired state
	glDisable(GL_DEPTH_TEST);
	glDisable(GL_CULL_FACE);

	// enable the quad shader
	glUseProgram(shader.mProgramId);

	// set the texture
	glActiveTexture(GL_TEXTURE0);
	glBindTexture(GL_TEXTURE_2D, inputTextureId);


	GLCHECK();

	// bind the quad array
	glBindBuffer(GL_ARRAY_BUFFER, mQuadAttributeArray);

	glEnableVertexAttribArray(shader.mPosTexLocation);
	for (int i = 1; i < 8; ++i)
	{
		glDisableVertexAttribArray(i);
	}

	glVertexAttribPointer(shader.mPosTexLocation, 4, GL_FLOAT, GL_FALSE, 0, nullptr);

	glValidateProgram(shader.mProgramId);
	char log[256];
	GLsizei len = 0;
	glGetProgramInfoLog(shader.mProgramId, sizeof(log), &len, log);

	GLint status = 0;
	glGetProgramiv(shader.mProgramId, GL_VALIDATE_STATUS, &status);
	assert(status == GL_TRUE);

	// draw the quad
	glDrawArrays(GL_TRIANGLES, 0, 6);
	GLCHECK();

	// unbind the texture
	glBindTexture(GL_TEXTURE_2D, 0);

	// restore previous state
	glUseProgram(previousProgramId);
	
	glBindBuffer(GL_ARRAY_BUFFER, previousAttribBuffer);
	glVertexAttribPointer(shader.mPosTexLocation, previousAttribSize, previousAttribType, GLboolean(previousAttribNormalized), previousAttribStride, previousAttribPointer);
	glBindBuffer(GL_ARRAY_BUFFER, previousBufferBinding);
	
	for (int i = 0; i < kNumVertexAttributes; ++i)
	{
		if (previousAttribEnabled[i] != 0)
		{
			glEnableVertexAttribArray(i);
		}
		else
		{
			glDisableVertexAttribArray(i);
		}
	}
	
	if (previousDepthTest)
	{
		glEnable(GL_DEPTH_TEST);
	}
	else
	{
		glDisable(GL_DEPTH_TEST);
	}

	if (previousCullFace)
	{
		glEnable(GL_CULL_FACE);
	}
	else
	{
		glDisable(GL_CULL_FACE);
	}

	glActiveTexture(previousActiveTextureUnit);
	glBindTexture(GL_TEXTURE_2D, previousTextureBinding);

	GLCHECK();
}


ttv::graphics::ios::FrameBuffer::FrameBuffer()
:	mFrameBufferId(0)
{
}


ttv::graphics::ios::FrameBuffer::~FrameBuffer()
{
	Destroy();
}


bool ttv::graphics::ios::FrameBuffer::Create()
{
	if (mFrameBufferId == 0)
	{
		glGenFramebuffers(1, &mFrameBufferId);
	}

	return true;
}


void ttv::graphics::ios::FrameBuffer::Destroy()
{
	if (mFrameBufferId != 0)
	{
		glDeleteFramebuffers(1, &mFrameBufferId);

		mFrameBufferId = 0;
	}
}

#endif
