/********************************************************************************************
* Twitch Broadcasting SDK
*
* This software is supplied under the terms of a license agreement with Twitch Interactive, Inc. and
* may not be copied or used except in accordance with the terms of that agreement
* Copyright (c) 2012-2016 Twitch Interactive, Inc.
*********************************************************************************************/

#include "internal/pch.h"
#include "internal/bindings/graphics_opengl2.h"


#if TTV_SUPPORT_OPENGL2

#include "twitchsdk.h"

#if TTV_PLATFORM_WINDOWS
ttv::graphics::opengl2::DeviceContext::DeviceContext()
:	mHGLRC(0)
,	mHDC(0)

{
}
#elif TTV_PLATFORM_MAC
ttv::graphics::opengl2::DeviceContext::DeviceContext()
:	mCGLContextObj(0)
{
}
#endif


#if TTV_PLATFORM_WINDOWS
void ttv::graphics::opengl2::DeviceContext::Create(HGLRC hglrc, HDC hdc)
{
	mHGLRC = hglrc;
	mHDC = hdc;
}
#elif TTV_PLATFORM_MAC
void ttv::graphics::opengl2::DeviceContext::Create(CGLContextObj context)
{
	mCGLContextObj = context;
}
#endif


void ttv::graphics::opengl2::DeviceContext::CreateFromCurrent()
{
#if TTV_PLATFORM_WINDOWS
	mHGLRC = wglGetCurrentContext();
	mHDC = wglGetCurrentDC();
#elif TTV_PLATFORM_MAC
	mCGLContextObj = CGLGetCurrentContext();
#endif
}


void ttv::graphics::opengl2::DeviceContext::Clear()
{
#if TTV_PLATFORM_WINDOWS
	mHGLRC = 0;
	mHDC = 0;
#elif TTV_PLATFORM_MAC
	mCGLContextObj = 0;
#endif
}


bool ttv::graphics::opengl2::DeviceContext::IsCurrent()
{
#if TTV_PLATFORM_WINDOWS
	HGLRC previousGLContext = wglGetCurrentContext();
	return previousGLContext == mHGLRC;
	
#elif TTV_PLATFORM_MAC
	return mCGLContextObj == CGLGetCurrentContext();
#endif
}


bool ttv::graphics::opengl2::DeviceContext::Bind()
{
#if TTV_PLATFORM_WINDOWS
	return wglMakeCurrent(mHDC, mHGLRC) != 0;
#elif TTV_PLATFORM_MAC
	return CGLSetCurrentContext(mCGLContextObj) == kCGLNoError;
#endif
}



ttv::graphics::opengl2::RenderTexture::RenderTexture()
:	mTextureId(0)
,	mWidth(0)
,	mHeight(0)
{
}


ttv::graphics::opengl2::RenderTexture::~RenderTexture()
{
	Destroy();
}


void ttv::graphics::opengl2::RenderTexture::Create(uint width, uint height)
{
	Destroy();
	
	// save state
	GLint previousTextureBinding = 0;
	glGetIntegerv(GL_TEXTURE_BINDING_2D, &previousTextureBinding);

	glGenTextures(1, &mTextureId);
	glBindTexture(GL_TEXTURE_2D, mTextureId);
	
	mWidth = width;
	mHeight = height;
	
	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, width, height, 0, GL_RGBA, GL_UNSIGNED_BYTE, nullptr);

	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);

	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);

	// restore state
	glBindTexture(GL_TEXTURE_2D, previousTextureBinding);
}


void ttv::graphics::opengl2::RenderTexture::Destroy()
{
	if (mTextureId != 0)
	{
		glDeleteTextures(1, &mTextureId);
	}
	
	Forget();
}


void ttv::graphics::opengl2::RenderTexture::Forget()
{
	mTextureId = 0;
	mWidth = 0;
	mHeight = 0;
}


void ttv::graphics::opengl2::RenderTexture::Clear(float r, float g, float b, float a)
{
	assert(mTextureId != 0);
	
	// save state
	GLint previousTextureBinding = 0;
	glGetIntegerv(GL_TEXTURE_BINDING_2D, &previousTextureBinding);
	
	glBindTexture(GL_TEXTURE_2D, GetId());
	
	glClearColor(r, g, b, a);
	glClear(GL_COLOR_BUFFER_BIT);
	
	// restore state
	glBindTexture(GL_TEXTURE_2D, previousTextureBinding);
}


void ttv::graphics::opengl2::RenderTexture::ManuallyClearTexture()
{
	static int n = 0;
	
	// save state
	GLint previousTextureBinding = 0;
	glGetIntegerv(GL_TEXTURE_BINDING_2D, &previousTextureBinding);
	
	glBindTexture(GL_TEXTURE_2D, GetId());
	
	uint8_t* pixels = new uint8_t[4 * mWidth * mHeight];
	int index = 0;
	for (uint y = 0; y < mHeight; ++y)
	{
		for (uint x = 0; x < mWidth; ++x)
		{
			pixels[index+0] = 0;
			pixels[index+1] = 0;
			pixels[index+2] = 0;
			pixels[index+3] = 255;
			pixels[index+(n%3)] = static_cast<char>(255.0f * static_cast<float>(y) / static_cast<float>(mHeight-1));
			
			index += 4;
		}
		
	}
	
	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, mWidth, mHeight, 0, GL_RGBA, GL_UNSIGNED_BYTE, pixels);
	delete [] pixels;
	
	n++;
	
	// restore state
	glBindTexture(GL_TEXTURE_2D, previousTextureBinding);
}


GLuint ttv::graphics::opengl2::RenderTexture::GetId()
{
	return mTextureId;
}


bool ttv::graphics::opengl2::GLCHECK()
{
	GLenum ret = glGetError();
	
	switch (ret)
	{
		case GL_NO_ERROR:
		{
			break;
		}
		default:
		{
			ttv::trace::Message("OpenGL", TTV_ML_ERROR, "OpenGL error: %d", ret);
			break;
		}
	}
	
	assert(ret == GL_NO_ERROR);
	return ret == GL_NO_ERROR;
}



ttv::graphics::opengl2::QuadShader::QuadShader()
:	mMvpLocation(0)
,	mTextureLocation(0)
,	mPosTexLocation(0)
,	mProgramId(0)
{
}


bool ttv::graphics::opengl2::QuadShader::Create()
{
	const char* vertexSource =
		"uniform mat4 u_MVP;"
		""
		"attribute vec4 a_PosTex;"
		""
		"varying vec2 v_TexCoord;"
		""
		"void main()"
		"{"
		"	gl_Position = u_MVP * vec4(a_PosTex.x, a_PosTex.y, -2, 1);"
		"	v_TexCoord = a_PosTex.zw;"
		"}";
	
	const char* fragmentSource =
		"uniform sampler2D u_Texture;"
		""
		"varying vec2 v_TexCoord;"
		""
		"void main()"
		"{"
		"	gl_FragColor = texture2D(u_Texture, v_TexCoord.xy);"
		""
		"	gl_FragColor.a = 1.0;"
		"}";
		
	GLint status = GL_TRUE;
	
	GLuint vertexShaderId = 0;
	GLuint fragmentShaderId = 0;
	
	// compile the vertex shader
	if (status == GL_TRUE)
	{
		vertexShaderId = glCreateShader(GL_VERTEX_SHADER);
		glShaderSource(vertexShaderId, 1, &vertexSource, NULL);
		glCompileShader(vertexShaderId);

		glGetShaderiv(vertexShaderId, GL_COMPILE_STATUS, &status);
	}
	
	// compile the fragment shader
	if (status == GL_TRUE)
	{
		fragmentShaderId = glCreateShader(GL_FRAGMENT_SHADER);
		glShaderSource(fragmentShaderId, 1, &fragmentSource, NULL);
		glCompileShader(fragmentShaderId);
		
		glGetShaderiv(fragmentShaderId, GL_COMPILE_STATUS, &status);
	}
	
	// create and link the program
	if (status == GL_TRUE)
	{
		mProgramId = glCreateProgram();
		
		glAttachShader(mProgramId, vertexShaderId);
		glAttachShader(mProgramId, fragmentShaderId);
		glLinkProgram(mProgramId);
		
		glGetProgramiv(mProgramId, GL_LINK_STATUS, &status);
	}
	
	// free the shaders, the program will retain the last reference to these
	if (vertexShaderId != 0)
	{
		glDeleteShader(vertexShaderId);
	}
	if (fragmentShaderId != 0)
	{
		glDeleteShader(fragmentShaderId);
	}
	
	if (status == GL_TRUE)
	{
		// extract uniform and attribute locations
		mMvpLocation = glGetUniformLocation(mProgramId, "u_MVP");
		mTextureLocation = glGetUniformLocation(mProgramId, "u_Texture");
		mPosTexLocation = glGetAttribLocation(mProgramId, "a_PosTex");
	}
	else
	{
		Destroy();
		
		assert(false);
		return false;
	}
		
	GLCHECK();
	
	return true;
}


void ttv::graphics::opengl2::QuadShader::Destroy()
{
	if (mProgramId != 0)
	{
		glDeleteProgram(mProgramId);
	}

	Forget();
}


void ttv::graphics::opengl2::QuadShader::Forget()
{
	mProgramId = 0;
	mMvpLocation = 0;
	mTextureLocation = 0;
	mPosTexLocation = 0;
}


void ttv::graphics::opengl2::QuadShader::SetUniforms(float width, float height)
{
	assert(mProgramId != 0);
	
	// set the ortho projection uniform
	float nearPlane = 1;
	float farPlane = 3;
	
	float mvp[] =
	{
		2.0f / width,	0.0f,			0.0f,											0.0f,
		0.0f,			2 / -height,	0,												0,
		0.0f,			0.0f,			-2.0f / (farPlane-nearPlane),					0.0f,
		-1.0f,			1.0f,			-(farPlane+nearPlane) / (farPlane-nearPlane),	1.0f
	};
	
	// save state
	GLint previousProgramId = 0;
	glGetIntegerv(GL_CURRENT_PROGRAM, &previousProgramId);
	
	// set the mvp value
	glUseProgram(mProgramId);
	glUniformMatrix4fv(mMvpLocation, 1, GL_FALSE, mvp);
	
	// set the uniform location
	glUniform1i(mTextureLocation, 0);
	
	// restore state
	glUseProgram(previousProgramId);
	
	GLCHECK();
}



ttv::graphics::opengl2::Quad::Quad()
:	mQuadAttributeArray(0)
,	mWidth(0)
,	mHeight(0)
{
}


ttv::graphics::opengl2::Quad::~Quad()
{
	Destroy();
}


void ttv::graphics::opengl2::Quad::Create(float width, float height, bool verticalFlip)
{
	Destroy();
	
	mWidth = width;
	mHeight = height;
	
	// x, y, tx, ty
	float quad[] =
	{
		0, 0,				0, 0,
		0, height,			0, 1,
		width, height,		1, 1,
		
		0, 0,				0, 0,
		width, height,		1, 1,
		width, 0,			1, 0
	};
	
	if (verticalFlip)
	{
		for (int i = 0; i < kNumQuadVertices * 4; i += 4)
		{
			quad[i+3] = 1 - quad[i+3];
		}
	}

	GLCHECK();
	
	// cache state
	GLint previousBufferBinding = 0;
	glGetIntegerv(GL_ARRAY_BUFFER_BINDING, &previousBufferBinding);
	GLCHECK();

	glGenBuffers(1, &mQuadAttributeArray);
	GLCHECK();
	
	glBindBuffer(GL_ARRAY_BUFFER, mQuadAttributeArray);
	GLCHECK();
	
	glBufferData(GL_ARRAY_BUFFER, sizeof(quad), quad, GL_STATIC_DRAW);
	GLCHECK();

	// restore state
	glBindBuffer(GL_ARRAY_BUFFER, previousBufferBinding);
	GLCHECK();
}


void ttv::graphics::opengl2::Quad::Destroy()
{
	if (mQuadAttributeArray != 0)
	{
		glDeleteBuffers(1, &mQuadAttributeArray);
	}

	Forget();
}


void ttv::graphics::opengl2::Quad::Forget()
{
	mQuadAttributeArray = 0;
	mHeight = 0;
	mWidth = 0;
}


void ttv::graphics::opengl2::Quad::Draw(const QuadShader& shader, GLint inputTextureId)
{
	assert(glCheckFramebufferStatus(GL_FRAMEBUFFER) == GL_FRAMEBUFFER_COMPLETE);
	assert(shader.mProgramId != 0);
	assert(inputTextureId != 0);
	
	GLCHECK();
	
	// cache state
	GLint previousProgramId = 0;
	glGetIntegerv(GL_CURRENT_PROGRAM, &previousProgramId);

	GLint previousBufferBinding = 0;
	glGetIntegerv(GL_ARRAY_BUFFER_BINDING, &previousBufferBinding);
	
	GLint previousDepthTest = 0;
	glGetIntegerv(GL_DEPTH_TEST, &previousDepthTest);
	
	GLint previousCullFace = 0;
	glGetIntegerv(GL_CULL_FACE, &previousCullFace);
	
	GLint previousActiveTextureUnit = 0;
	glGetIntegerv(GL_ACTIVE_TEXTURE, &previousActiveTextureUnit);

	GLint previousTextureBinding = 0;
	glGetIntegerv(GL_TEXTURE_BINDING_2D, &previousTextureBinding);
	
	const int kNumVertexAttributes = 8;
	GLint previousAttribEnabled[kNumVertexAttributes];
	for (int i = 0; i < kNumVertexAttributes; ++i)
	{
		glGetVertexAttribiv(i, GL_VERTEX_ATTRIB_ARRAY_ENABLED, &previousAttribEnabled[i]);
	}

	GLint previousAttribBuffer;
	GLint previousAttribSize;
	GLint previousAttribStride;
	GLint previousAttribType;
	GLint previousAttribNormalized;
	GLvoid *previousAttribPointer;
	glGetVertexAttribPointerv(shader.mPosTexLocation, GL_VERTEX_ATTRIB_ARRAY_POINTER, &previousAttribPointer);
	glGetVertexAttribiv(shader.mPosTexLocation, GL_VERTEX_ATTRIB_ARRAY_BUFFER_BINDING, &previousAttribBuffer);
	glGetVertexAttribiv(shader.mPosTexLocation, GL_VERTEX_ATTRIB_ARRAY_SIZE, &previousAttribSize);
	glGetVertexAttribiv(shader.mPosTexLocation, GL_VERTEX_ATTRIB_ARRAY_STRIDE, &previousAttribStride);
	glGetVertexAttribiv(shader.mPosTexLocation, GL_VERTEX_ATTRIB_ARRAY_TYPE, &previousAttribType);
	glGetVertexAttribiv(shader.mPosTexLocation, GL_VERTEX_ATTRIB_ARRAY_NORMALIZED, &previousAttribNormalized);

	// set desired state
	glDisable(GL_DEPTH_TEST);
	glDisable(GL_CULL_FACE);
	
	// enable the quad shader
	glUseProgram(shader.mProgramId);

	// set the texture
	glActiveTexture(GL_TEXTURE0);
	glBindTexture(GL_TEXTURE_2D, inputTextureId);

	
	GLCHECK();
	
	// bind the quad array
	glBindBuffer(GL_ARRAY_BUFFER, mQuadAttributeArray);

	glEnableVertexAttribArray(shader.mPosTexLocation);
	for (int i = 1; i < 8; ++i)
	{
		glDisableVertexAttribArray(i);
	}

	glVertexAttribPointer(shader.mPosTexLocation, 4, GL_FLOAT, GL_FALSE, 0, nullptr);
	
	glValidateProgram(shader.mProgramId);
	char log[256];
	GLsizei len = 0;
	glGetProgramInfoLog(shader.mProgramId, sizeof(log), &len, log);
	
	GLint status = 0;
	glGetProgramiv(shader.mProgramId, GL_VALIDATE_STATUS, &status);
	assert(status == GL_TRUE);
	
	// draw the quad
	glDrawArrays(GL_TRIANGLES, 0, 6);
	GLCHECK();
	
	// unbind the texture
	glBindTexture(GL_TEXTURE_2D, 0);
	
	// restore state
	glUseProgram(previousProgramId);
	
	glBindBuffer(GL_ARRAY_BUFFER, previousAttribBuffer);
	glVertexAttribPointer(shader.mPosTexLocation, previousAttribSize, previousAttribType, static_cast<GLboolean>(previousAttribNormalized), previousAttribStride, previousAttribPointer);
	glBindBuffer(GL_ARRAY_BUFFER, previousBufferBinding);
	
	for (int i = 0; i < kNumVertexAttributes; ++i)
	{
		if (previousAttribEnabled[i] != 0)
		{
			glEnableVertexAttribArray(i);
		}
		else
		{
			glDisableVertexAttribArray(i);
		}
	}
		
	if (previousDepthTest)
	{
		glEnable(GL_DEPTH_TEST);
	}
	else
	{
		glDisable(GL_DEPTH_TEST);
	}
	
	if (previousCullFace)
	{
		glEnable(GL_CULL_FACE);
	}
	else
	{
		glDisable(GL_CULL_FACE);
	}
	
	glActiveTexture(previousActiveTextureUnit);
	glBindTexture(GL_TEXTURE_2D, previousTextureBinding);
	
	GLCHECK();
}


ttv::graphics::opengl2::FrameBuffer::FrameBuffer()
:	mFrameBufferId(0)
{
}


ttv::graphics::opengl2::FrameBuffer::~FrameBuffer()
{
	Destroy();
}


bool ttv::graphics::opengl2::FrameBuffer::Create()
{
	if (mFrameBufferId == 0)
	{
		glGenFramebuffers(1, &mFrameBufferId);
	}

	return true;
}


void ttv::graphics::opengl2::FrameBuffer::Destroy()
{
	if (mFrameBufferId != 0)
	{
		glDeleteFramebuffers(1, &mFrameBufferId);
	}

	Forget();
}


void ttv::graphics::opengl2::FrameBuffer::Forget()
{
	mFrameBufferId = 0;
}


ttv::graphics::opengl2::GenericBuffer::GenericBuffer()
:	mBufferId(0)
{
}


ttv::graphics::opengl2::GenericBuffer::~GenericBuffer()
{
	Destroy();
}


bool ttv::graphics::opengl2::GenericBuffer::Create()
{
	if (mBufferId == 0)
	{
		glGenBuffers(1, &mBufferId);
	}
	
	return true;
}


void ttv::graphics::opengl2::GenericBuffer::Destroy()
{
	if (mBufferId != 0)
	{
		glDeleteBuffers(1, &mBufferId);
	}

	Forget();
}


void ttv::graphics::opengl2::GenericBuffer::Forget()
{
	mBufferId = 0;
}


ttv::graphics::opengl2::Query::Query()
:	mQuery(0)
{
}


ttv::graphics::opengl2::Query::~Query()
{
	Destroy();
}

				
bool ttv::graphics::opengl2::Query::Create()
{
	Destroy();

	glGenQueries(1, &mQuery);

	return true;
}


void ttv::graphics::opengl2::Query::Destroy()
{
	if (mQuery != 0)
	{
		glDeleteQueries(1, &mQuery);
	}

	Forget();
}


void ttv::graphics::opengl2::Query::Forget()
{
	mQuery = 0;
}


void ttv::graphics::opengl2::Query::Begin()
{
	assert(mQuery);
	
	glBeginQuery(GL_SAMPLES_PASSED, mQuery);
}


void ttv::graphics::opengl2::Query::End()
{
	assert(mQuery);
	
	glEndQuery(GL_SAMPLES_PASSED);
}


bool ttv::graphics::opengl2::Query::IsReady()
{
	GLint ready = GL_FALSE;
	glGetQueryObjectiv(mQuery, GL_QUERY_RESULT_AVAILABLE, &ready);

	return ready == GL_TRUE;
}

#endif
