﻿using System;
using System.Collections.Generic;

using size_t = System.UIntPtr;


namespace Twitch.Broadcast
{
    public class Constants
    {
        public const uint TTV_MIN_BITRATE = 230;
        public const uint TTV_MAX_BITRATE = 3500;
        public const uint TTV_MIN_FPS = 10;
        public const uint TTV_MAX_FPS = 60;
        public const uint TTV_MAX_WIDTH = 1920;
        public const uint TTV_MAX_HEIGHT = 1200;
    }

    public enum RTMPState
    {
        Invalid = -1,
        Idle,
        Initialize,
        Handshake,
        Connect,
        CreateStream,
        Publish,
        SendVideo,
        Shutdown,
        Error
    }

    public enum StartFlags
    {
        None = 0,

        TTV_Start_BandwidthTest = 1 << 0
    }

    public enum AuthFlag
    {
        TTV_AuthFlag_Broadcast = (1 << 0),
        TTV_AuthFlag_Chat = (1 << 1),
    }

    public struct TTV_AuthParams
    {
        internal size_t size;
        internal string userName;
        internal string password;
        internal string clientSecret;
    }

    public class AuthParams : StructWrapperBase<TTV_AuthParams>
    {
        public AuthParams()
        {
            mStruct.size = (size_t)System.Runtime.InteropServices.Marshal.SizeOf(typeof(TTV_AuthParams));
        }

        public string UserName
        {
            get { return mStruct.userName; }
            set { mStruct.userName = value; }
        }

        public string Password
        {
            get { return mStruct.password; }
            set { mStruct.password = value; }
        }

        public string ClientSecret
        {
            get { return mStruct.clientSecret; }
            set { mStruct.clientSecret = value; }
        }
    }

    public unsafe struct TTV_IngestServer
    {
        public fixed byte serverName[256];
        public fixed byte serverUrl[256];
        public byte defaultServer;
    }

    public class IngestServer : StructWrapperBase<TTV_IngestServer>
    {
        internal float mBitrateKbps = 0;

        public IngestServer()
        {
            this.ServerName = "";
            this.ServerUrl = "";
            this.DefaultServer = false;
        }

        internal IngestServer(ref TTV_IngestServer info)
        {
            mStruct = info;
        }

        internal unsafe IngestServer(TTV_IngestServer* info)
        {
            mStruct = *info;
        }

        public unsafe string ServerName
        {
            get
            {
                fixed (byte* b = mStruct.serverName)
                {
                    return MarshalUtil.StringFromBytes(b, 256);
                }
            }
            set
            {
                fixed (byte* b = mStruct.serverName)
                {
                    MarshalUtil.StringToBytes(value, b, 256);
                }
            }
        }

        public unsafe string ServerUrl
        {
            get
            {
                fixed (byte* b = mStruct.serverUrl)
                {
                    return MarshalUtil.StringFromBytes(b, 256);
                }
            }
            set
            {
                fixed (byte* b = mStruct.serverUrl)
                {
                    MarshalUtil.StringToBytes(value, b, 256);
                }
            }
        }

        public bool DefaultServer
        {
            get { return mStruct.defaultServer != 0; }
            set { mStruct.defaultServer = (byte)(value ? 1 : 0); }
        }

        /**
         * The estimated bitrate to the ingest server.  This value may be 0 if testing has not been performed.
         */
        public float BitrateKbps
        {
            get { return mBitrateKbps; }
            set { mBitrateKbps = value; }
        }
    }

    public unsafe struct TTV_IngestList
    {
        public TTV_IngestServer* ingestList;
        public uint ingestCount;
    }

    public class IngestList
    {
        protected IngestServer[] servers = null;
        protected IngestServer defaultServer = null;

        public IngestList()
        {
            servers = new IngestServer[0];
        }

        internal unsafe IngestList(ref TTV_IngestList info)
        {
            if (info.ingestList == null)
            {
                servers = new IngestServer[0];
            }
            else
            {
                servers = new IngestServer[info.ingestCount];
                for (uint i = 0; i < info.ingestCount; ++i)
                {
                    servers[i] = new IngestServer(ref info.ingestList[i]);

                    if (servers[i].DefaultServer)
                    {
                        defaultServer = servers[i];
                    }
                }

                if (defaultServer == null && servers.Length > 0)
                {
                    defaultServer = servers[0];
                }
            }
        }

        internal unsafe IngestList(TTV_IngestList* info)
        {
            if (info->ingestList == null)
            {
                servers = new IngestServer[0];
            }
            else
            {
                servers = new IngestServer[info->ingestCount];
                for (uint i = 0; i < info->ingestCount; ++i)
                {
                    servers[i] = new IngestServer(ref info->ingestList[i]);

                    if (servers[i].DefaultServer)
                    {
                        defaultServer = servers[i];
                    }
                }

                if (defaultServer == null && servers.Length > 0)
                {
                    defaultServer = servers[0];
                }
            }
        }

        public IngestServer[] Servers
        {
            get { return servers; }
            set { servers = value; }
        }

        public IngestServer DefaultServer
        {
            get { return defaultServer; }
        }

        public IngestServer BestServer
        {
            get
            {
                if (servers == null || servers.Length == 0)
                {
                    return null;
                }

                IngestServer best = servers[0];
                for (int i = 1; i < servers.Length; ++i)
                {
                    if (best.BitrateKbps < servers[i].BitrateKbps)
                    {
                        best = servers[i];
                    }
                }

                return best;
            }
        }
    }

    public enum PixelFormat
    {
        TTV_PF_BGRA = 0x00010203,
        TTV_PF_ABGR = 0x01020300,
        TTV_PF_RGBA = 0x02010003,
        TTV_PF_ARGB = 0x03020100
    }

    public enum EncodingCpuUsage
    {
        TTV_ECU_LOW,
        TTV_ECU_MEDIUM,
        TTV_ECU_HIGH
    }

    public struct TTV_VideoParams
    {
        public size_t size;
        public uint outputWidth;
        public uint outputHeight;
        public PixelFormat pixelFormat;
        public uint maxKbps;
        public uint targetFps;
        public EncodingCpuUsage encodingCpuUsage;
        public byte disableAdaptiveBitrate;
        public byte verticalFlip;
        public UIntPtr encoderPluginPointer;
    }

    public class VideoParams : StructWrapperBase<TTV_VideoParams>, ICloneable
    {
        public VideoParams()
        {
            mStruct.size = (size_t)System.Runtime.InteropServices.Marshal.SizeOf(typeof(TTV_VideoParams));
        }

        internal VideoParams(ref TTV_VideoParams p)
        {
            base.mStruct = p;
        }

        public uint OutputWidth
        {
            get { return mStruct.outputWidth; }
            set { mStruct.outputWidth = value; }
        }

        public uint OutputHeight
        {
            get { return mStruct.outputHeight; }
            set { mStruct.outputHeight = value; }
        }

        public PixelFormat PixelFormat
        {
            get { return mStruct.pixelFormat; }
            set { mStruct.pixelFormat = value; }
        }

        public uint MaxKbps
        {
            get { return mStruct.maxKbps; }
            set { mStruct.maxKbps = value; }
        }

        public uint TargetFps
        {
            get { return mStruct.targetFps; }
            set { mStruct.targetFps = value; }
        }

        public EncodingCpuUsage EncodingCpuUsage
        {
            get { return mStruct.encodingCpuUsage; }
            set { mStruct.encodingCpuUsage = value; }
        }

        public bool DisableAdaptiveBitrate
        {
            get { return mStruct.disableAdaptiveBitrate != 0; }
            set { mStruct.disableAdaptiveBitrate = value ? (byte)1 : (byte)0; }
        }

        public bool VerticalFlip
        {
            get { return mStruct.verticalFlip != 0; }
            set { mStruct.verticalFlip = value ? (byte)1 : (byte)0; }
        }

        public object Clone()
        {
            return new VideoParams(ref base.mStruct);
        }
    }

    public enum VideoEncoder
    {
        TTV_VID_ENC_DISABLE = -2,
        TTV_VID_ENC_DEFAULT = -1,

        TTV_VID_ENC_INTEL = 0,
        TTV_VID_ENC_APPLE = 2,
        TTV_VID_ENC_PLUGIN = 100
    }

    public enum AudioEncoder
    {
        TTV_AUD_ENC_DEFAULT = -1,

        TTV_AUD_ENC_LAMEMP3 = 0,
        TTV_AUD_ENC_APPLEAAC = 1,
    }

    public enum AudioSampleFormat
    {
        TTV_ASF_PCM_S16
    }

    public struct TTV_AudioParams
    {
        public size_t size;
        public byte audioEnabled;
        public byte enableMicCapture;
        public byte enablePlaybackCapture;
        public byte enablePassthroughAudio;
    }

    public class AudioParams : StructWrapperBase<TTV_AudioParams>, ICloneable
    {
        public AudioParams()
        {
            mStruct.size = (size_t)System.Runtime.InteropServices.Marshal.SizeOf(typeof(TTV_AudioParams));
        }

        internal AudioParams(ref TTV_AudioParams p)
        {
            base.mStruct = p;
        }

        public bool AudioEnabled
        {
            get { return mStruct.audioEnabled != 0; }
            set { mStruct.audioEnabled = (byte)(value ? 1 : 0); }
        }

        public bool EnableMicCapture
        {
            get { return mStruct.enableMicCapture != 0; }
            set { mStruct.enableMicCapture = (byte)(value ? 1 : 0); }
        }

        public bool EnablePlaybackCapture
        {
            get { return mStruct.enablePlaybackCapture != 0; }
            set { mStruct.enablePlaybackCapture = (byte)(value ? 1 : 0); }
        }

        public bool EnablePassthroughAudio
        {
            get { return mStruct.enablePassthroughAudio != 0; }
            set { mStruct.enablePassthroughAudio = (byte)(value ? 1 : 0); }
        }

        public object Clone()
        {
            return new AudioParams(ref base.mStruct);
        }
    }

    public enum TTV_AudioDeviceType
    {
        TTV_PLAYBACK_DEVICE,
        TTV_RECORDER_DEVICE,
        TTV_PASSTHROUGH_DEVICE,

        TTV_DEVICE_NUM
    }

    public enum StatType
    {
        TTV_ST_RTMPSTATE,
        TTV_ST_RTMPDATASENT
    }

    public unsafe struct TTV_UserInfo
    {
        public size_t size;
        public fixed byte displayName[64];
        public fixed byte name[64];
    }

    public class UserInfo
    {
        protected string name = string.Empty;
        protected string displayName = string.Empty;

        public UserInfo()
        {
        }

        internal unsafe UserInfo(ref TTV_UserInfo info)
        {
            fixed (byte* b = info.displayName)
            {
                displayName = MarshalUtil.StringFromBytes(b, 64);
            }
            fixed (byte* b = info.name)
            {
                name = MarshalUtil.StringFromBytes(b, 64);
            }
        }

        internal unsafe UserInfo(TTV_UserInfo* info)
        {
            displayName = MarshalUtil.StringFromBytes(info->displayName, 64);
            name = MarshalUtil.StringFromBytes(info->name, 64);
        }

        public string DisplayName
        {
            get { return displayName; }
            set { displayName = value; }
        }

        public string Name
        {
            get { return name; }
            set { name = value; }
        }
    }

    public unsafe struct TTV_ChannelInfo
    {
        public size_t size;
        public fixed byte name[64];
        public fixed byte displayName[64];
        public fixed byte channelUrl[256];
    }

    public class ChannelInfo
    {
        protected string name = string.Empty;
        protected string displayName = string.Empty;
        protected string channelUrl = string.Empty;

        public ChannelInfo()
        {

        }

        internal unsafe ChannelInfo(ref TTV_ChannelInfo info)
        {
            fixed (byte* b = info.name)
            {
                name = MarshalUtil.StringFromBytes(b, 64);
            }

            fixed (byte* b = info.displayName)
            {
                displayName = MarshalUtil.StringFromBytes(b, 64);
            }

            fixed (byte* b = info.channelUrl)
            {
                channelUrl = MarshalUtil.StringFromBytes(b, 256);
            }
        }

        internal unsafe ChannelInfo(TTV_ChannelInfo* info)
        {
            name = MarshalUtil.StringFromBytes(info->name, 64);
            displayName = MarshalUtil.StringFromBytes(info->displayName, 64);
            channelUrl = MarshalUtil.StringFromBytes(info->channelUrl, 256);
        }

        public string Name
        {
            get { return name; }
            set { name = value; }
        }

        public string DisplayName
        {
            get { return displayName; }
            set { displayName = value; }
        }

        public string ChannelUrl
        {
            get { return channelUrl; }
            set { channelUrl = value; }
        }
    }

    public unsafe struct TTV_StreamInfo
    {
        public size_t size;
        public int viewers;
        public UInt64 streamId;
    }

    public class StreamInfo
    {
        protected int viewers = 0;
        protected UInt64 streamId = 0;

        public StreamInfo()
        {
        }

        public StreamInfo(ref TTV_StreamInfo info)
        {
            viewers = info.viewers;
            streamId = info.streamId;
        }

        public unsafe StreamInfo(TTV_StreamInfo* info)
        {
            viewers = info->viewers;
            streamId = info->streamId;
        }

        public int Viewers
        {
            get { return viewers; }
            set { viewers = value; }
        }

        public UInt64 StreamId
        {
            get { return streamId; }
            set { streamId = value; }
        }
    }

    public unsafe struct TTV_StreamInfoForSetting
    {
        public size_t size;
        public fixed byte streamTitle[256];
        public fixed byte gameName[256];
    }

    public class StreamInfoForSetting : StructWrapperBase<TTV_StreamInfoForSetting>
    {
        public StreamInfoForSetting()
        {
            mStruct.size = (size_t)System.Runtime.InteropServices.Marshal.SizeOf(typeof(TTV_StreamInfoForSetting));
        }

        public unsafe string StreamTitle
        {
            set
            {
                fixed (byte* b = mStruct.streamTitle)
                {
                    MarshalUtil.StringToBytes(value, b, 256);
                }
            }
            get
            {
                fixed (byte* b = mStruct.streamTitle)
                {
                    return MarshalUtil.StringFromBytes(b, 256);
                }
            }
        }

        public unsafe string GameName
        {
            set
            {
                fixed (byte* b = mStruct.gameName)
                {
                    MarshalUtil.StringToBytes(value, b, 256);
                }
            }
            get
            {
                fixed (byte* b = mStruct.gameName)
                {
                    return MarshalUtil.StringFromBytes(b, 256);
                }
            }
        }
    }

    public unsafe struct TTV_ArchivingState
    {
        public size_t size;
        public byte recordingEnabled;
        public fixed byte pad[3];
        public fixed byte cureUrl[256];
    }

    public class ArchivingState
    {
        protected string cureUrl = string.Empty;
        protected bool recordingEnabled = false;

        public ArchivingState()
        {
        }

        internal unsafe ArchivingState(ref TTV_ArchivingState info)
        {
            fixed (byte* b = info.cureUrl)
            {
                cureUrl = MarshalUtil.StringFromBytes(b, 256);
            }

            recordingEnabled = info.recordingEnabled != 0;
        }

        internal unsafe ArchivingState(TTV_ArchivingState* info)
        {
            cureUrl = MarshalUtil.StringFromBytes(info->cureUrl, 256);
            recordingEnabled = info->recordingEnabled != 0;
        }

        public string CureUrl
        {
            get { return cureUrl; }
            set { cureUrl = value; }
        }

        public bool RecordingEnabled
        {
            get { return recordingEnabled; }
            set { recordingEnabled = value; }
        }
    }

    public unsafe struct TTV_GameInfo
    {
        public fixed byte name[256];
        public int popularity;
        public int id;
    }

    public class GameInfo
    {
        protected string name = null;
        protected int popularity = 0;
        protected int id = 0;

        public GameInfo()
        {
        }

        internal unsafe GameInfo(ref TTV_GameInfo info)
        {
            fixed (byte* b = info.name)
            {
                name = MarshalUtil.StringFromBytes(b, 256);
            }

            popularity = info.popularity;
            id = info.id;
        }

        public string Name
        {
            get { return name; }
            set { name = value; }
        }

        public int Popularity
        {
            get { return popularity; }
            set { popularity = value; }
        }

        public int Id
        {
            get { return id; }
            set { id = value; }
        }
    }

    public unsafe struct TTV_GameInfoList
    {
        public TTV_GameInfo* list;
        public uint count;
    }

    public class GameInfoList
    {
        protected GameInfo[] gameList = null;

        public GameInfoList()
        {
        }

        internal unsafe GameInfoList(ref TTV_GameInfoList data)
        {
            if (data.list == null)
            {
                gameList = new GameInfo[0];
            }
            else
            {
                gameList = new GameInfo[data.count];
                for (uint i = 0; i < data.count; ++i)
                {
                    gameList[i] = new GameInfo(ref data.list[i]);
                }
            }
        }

        public GameInfo[] List
        {
            get { return gameList; }
            set { gameList = value; }
        }
    }
}
