﻿using System;
using System.Collections.Generic;
using System.Runtime.InteropServices;

using SequenceId = System.UIntPtr; // To handle ulong size properly


namespace Twitch.Broadcast
{
    public class DesktopBroadcastApi : BroadcastApi
    {
        #region Native Functions

        #region BroadcastApi

        [DllImport("twitchsdk", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_Init")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_Init(ref StandardBroadcastApiHelper.ManagedBroadcastApiListener listener);
        [DllImport("twitchsdk", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_Shutdown")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_Shutdown();

        [DllImport("twitchsdk", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_RequestAuthToken")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_RequestAuthToken(ref TTV_AuthParams authParams, UInt32 flags);
        [DllImport("twitchsdk", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_Login")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_Login(ref TTV_AuthToken authToken);
        [DllImport("twitchsdk", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_GetIngestServers")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_GetIngestServers(ref TTV_AuthToken authToken);
        [DllImport("twitchsdk", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_GetUserInfo")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_GetUserInfo(ref TTV_AuthToken authToken);
        [DllImport("twitchsdk", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_GetStreamInfo")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_GetStreamInfo(ref TTV_AuthToken authToken, string channel);
        [DllImport("twitchsdk", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_RunCommercial")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_RunCommercial(ref TTV_AuthToken authToken);
        [DllImport("twitchsdk", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_GetArchivingState")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_GetArchivingState(ref TTV_AuthToken authToken);
        [DllImport("twitchsdk", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_GetGameNameList")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_GetGameNameList(string str);

        [DllImport("twitchsdk", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_SetStreamInfo")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_SetStreamInfo(ref TTV_AuthToken authToken, string channel, ref TTV_StreamInfoForSetting streamInfoToSet);
        [DllImport("twitchsdk", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_SetVolume")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_SetVolume(TTV_AudioDeviceType device, float volume);
        [DllImport("twitchsdk", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_GetVolume")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_GetVolume(TTV_AudioDeviceType device, ref float volume);
        [DllImport("twitchsdk", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_GetDefaultParams")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_GetDefaultParams(ref TTV_VideoParams vidParams);
        [DllImport("twitchsdk", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_GetMaxResolution")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_GetMaxResolution(uint maxKbps, uint frameRate, float bitsPerPixel, float aspectRatio, ref uint width, ref uint height);
        [DllImport("twitchsdk", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_PollTasks")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_PollTasks();
        [DllImport("twitchsdk", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_SendActionMetaData")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_SendActionMetaData(ref TTV_AuthToken authToken, string name, UInt64 streamTime, string humanDescription, string data);
        [DllImport("twitchsdk", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_SendStartSpanMetaData")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_SendStartSpanMetaData(ref TTV_AuthToken authToken, string name, UInt64 streamTime, out SequenceId sequenceId, string humanDescription, string data);
        [DllImport("twitchsdk", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_SendEndSpanMetaData")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_SendEndSpanMetaData(ref TTV_AuthToken authToken, string name, UInt64 streamTime, SequenceId sequenceId, string humanDescription, string data);
        [DllImport("twitchsdk", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_SubmitVideoFrame")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_SubmitVideoFrame(UIntPtr frameBuffer);
        [DllImport("twitchsdk", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_SubmitAudioSamples")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_SubmitAudioSamples(Int16[] samples, uint numSamples);
        [DllImport("twitchsdk", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_Start")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_Start(ref TTV_VideoParams videoParams, ref TTV_AudioParams audioParams, ref TTV_IngestServer ingestServer, UInt32 flags, bool async);
        [DllImport("twitchsdk", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_Stop")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_Stop(bool async);
        [DllImport("twitchsdk", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_PauseVideo")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_PauseVideo();
        [DllImport("twitchsdk", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_GetStreamTime")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_GetStreamTime(out UInt64 timeMs);

        [DllImport("twitchsdk", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_AllocateFrameBuffer")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_AllocateFrameBuffer(uint size, out UIntPtr buffer);
        [DllImport("twitchsdk", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_FreeFrameBuffer")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_FreeFrameBuffer(UIntPtr buffer);
        [DllImport("twitchsdk", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_RandomizeFrameBuffer")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_RandomizeFrameBuffer(UIntPtr buffer, uint size);

        [DllImport("twitchsdk", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_PollStats")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_PollStats();
        [DllImport("twitchsdk", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_RegisterStatsCallback")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_RegisterStatsCallback(ref StandardBroadcastApiHelper.ManagedStatsListener listener);
        [DllImport("twitchsdk", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_RemoveStatsCallback")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_RemoveStatsCallback(ref StandardBroadcastApiHelper.ManagedStatsListener listener);

        #endregion

        #endregion

        #region BroadcastApi Implementation

        private StandardBroadcastApiHelper m_Helper = null;
        private StandardBroadcastApiHelper.ApiListenerAdapter m_ApiListenerAdapter = null;
        private StandardBroadcastApiHelper.StatsListenerAdapter m_StatsListenerAdapter = null;
   
        public override IBroadcastApiListener BroadcastApiListener
        {
            get { return m_BroadcastApiListener; }
            set
            {
                if (m_BroadcastApiListener == value)
                {
                    return;
                }

                if (m_BroadcastApiListener != null)
                {
                    m_ApiListenerAdapter.ClientListener = null;
                }

                m_BroadcastApiListener = value;

                if (m_BroadcastApiListener != null)
                {
                    m_BroadcastApiListener = value;

                    m_ApiListenerAdapter.ClientListener = m_BroadcastApiListener;
                }
            }
        }

        public DesktopBroadcastApi()
        {
            m_ApiListenerAdapter = new StandardBroadcastApiHelper.ApiListenerAdapter();
            m_StatsListenerAdapter = new StandardBroadcastApiHelper.StatsListenerAdapter();
            m_Helper = new StandardBroadcastApiHelper(m_ApiListenerAdapter, m_StatsListenerAdapter);
        }

        public override ErrorCode Initialize()
        {
            m_ApiListenerAdapter.ClientListener = m_BroadcastApiListener;

            StandardBroadcastApiHelper.ManagedBroadcastApiListener nativeListener = m_ApiListenerAdapter.ManagedListener;
            return TTV_CSharp_Broadcast_Init(ref nativeListener);
        }

        public override ErrorCode Shutdown()
        {
            ErrorCode ec = TTV_CSharp_Broadcast_Shutdown();

            if (Error.Succeeded(ec))
            {
                m_ApiListenerAdapter.ClientListener = null;
            }

            return ec;
        }

        public override ErrorCode RequestAuthToken(AuthParams authParams, AuthFlag flags)
        {
            return TTV_CSharp_Broadcast_RequestAuthToken(ref authParams.mStruct, (UInt32)flags);
        }

        public override ErrorCode Login(AuthToken authToken)
        {
            return TTV_CSharp_Broadcast_Login(ref authToken.mStruct);
        }

        public override ErrorCode GetIngestServers(AuthToken authToken)
        {
            return TTV_CSharp_Broadcast_GetIngestServers(ref authToken.mStruct);
        }

        public override ErrorCode GetUserInfo(AuthToken authToken)
        {
            return TTV_CSharp_Broadcast_GetUserInfo(ref authToken.mStruct);
        }

        public override ErrorCode GetStreamInfo(AuthToken authToken, string channel)
        {
            return TTV_CSharp_Broadcast_GetStreamInfo(ref authToken.mStruct, channel);
        }

        public override ErrorCode SetStreamInfo(AuthToken authToken, string channel, StreamInfoForSetting streamInfoToSet)
        {
            return TTV_CSharp_Broadcast_SetStreamInfo(ref authToken.mStruct, channel, ref streamInfoToSet.mStruct);
        }

        public override ErrorCode GetArchivingState(AuthToken authToken)
        {
            return TTV_CSharp_Broadcast_GetArchivingState(ref authToken.mStruct);
        }

        public override ErrorCode RunCommercial(AuthToken authToken)
        {
            return TTV_CSharp_Broadcast_RunCommercial(ref authToken.mStruct);
        }

        public override ErrorCode SetVolume(TTV_AudioDeviceType device, float volume)
        {
            return TTV_CSharp_Broadcast_SetVolume(device, volume);
        }

        public override ErrorCode GetVolume(TTV_AudioDeviceType device, ref float volume)
        {
            return TTV_CSharp_Broadcast_GetVolume(device, ref volume);
        }

        public override ErrorCode GetGameNameList(string str)
        {
            return TTV_CSharp_Broadcast_GetGameNameList(str);
        }

        public override ErrorCode GetDefaultParams(VideoParams vidParams)
        {
            return TTV_CSharp_Broadcast_GetDefaultParams(ref vidParams.mStruct);
        }

        public override ErrorCode GetMaxResolution(uint maxKbps, uint frameRate, float bitsPerPixel, float aspectRatio, ref uint width, ref uint height)
        {
            return TTV_CSharp_Broadcast_GetMaxResolution(maxKbps, frameRate, bitsPerPixel, aspectRatio, ref width, ref height);
        }

        public override ErrorCode PollTasks()
        {
            try
            {
                m_InsidePollTasks = true;
                
                return TTV_CSharp_Broadcast_PollTasks();
            }
            finally
            {
                m_InsidePollTasks = false;
            }
        }

        public override ErrorCode SendActionMetaData(AuthToken authToken, string name, UInt64 streamTime, string humanDescription, string data)
        {
            return TTV_CSharp_Broadcast_SendActionMetaData(ref authToken.mStruct, name, streamTime, humanDescription, data);
        }

        public override ErrorCode SendStartSpanMetaData(AuthToken authToken, string name, UInt64 streamTime, out ulong sequenceId, string humanDescription, string data)
        {
            SequenceId s;
            ErrorCode ec = TTV_CSharp_Broadcast_SendStartSpanMetaData(ref authToken.mStruct, name, streamTime, out s, humanDescription, data);
            sequenceId = (ulong)s;
            return ec;
        }

        public override ErrorCode SendEndSpanMetaData(AuthToken authToken, string name, UInt64 streamTime, ulong sequenceId, string humanDescription, string data)
        {
            return TTV_CSharp_Broadcast_SendEndSpanMetaData(ref authToken.mStruct, name, streamTime, (SequenceId)sequenceId, humanDescription, data);
        }

        public override ErrorCode SubmitVideoFrame(UIntPtr frameBuffer)
        {
            return TTV_CSharp_Broadcast_SubmitVideoFrame(frameBuffer);
        }

        public override ErrorCode SubmitAudioSamples(Int16[] samples, uint numSamples)
        {
            return TTV_CSharp_Broadcast_SubmitAudioSamples(samples, numSamples);
        }

        public override ErrorCode Start(VideoParams videoParams, AudioParams audioParams, IngestServer ingestServer, StartFlags flags, bool async)
        {
            return TTV_CSharp_Broadcast_Start(ref videoParams.mStruct, ref audioParams.mStruct, ref ingestServer.mStruct, (uint)flags, async);
        }

        public override ErrorCode Stop(bool async)
        {
            return TTV_CSharp_Broadcast_Stop(async);
        }

        public override ErrorCode PauseVideo()
        {
            return TTV_CSharp_Broadcast_PauseVideo();
        }

        public override ErrorCode AllocateFrameBuffer(uint size, out UIntPtr buffer)
        {
            return TTV_CSharp_Broadcast_AllocateFrameBuffer(size, out buffer);
        }

        public override ErrorCode FreeFrameBuffer(UIntPtr buffer)
        {
            return TTV_CSharp_Broadcast_FreeFrameBuffer(buffer);
        }

        public override ErrorCode RandomizeFrameBuffer(UIntPtr buffer, uint size)
        {
            return TTV_CSharp_Broadcast_RandomizeFrameBuffer(buffer, size);
        }

        public override ErrorCode GetStreamTime(out UInt64 timeMs)
        {
            return TTV_CSharp_Broadcast_GetStreamTime(out timeMs);
        }

        #region Stats

        public override ErrorCode RegisterStatsCallback(IStatsListener listener)
        {
            if (m_StatsListenerAdapter.ClientListener != null)
            {
                return ErrorCode.TTV_EC_ALREADY_INITIALIZED;
            }

            StandardBroadcastApiHelper.ManagedStatsListener nativeListener = m_StatsListenerAdapter.ManagedListener;

            ErrorCode ec = TTV_CSharp_Broadcast_RegisterStatsCallback(ref nativeListener);
            if (Error.Succeeded(ec))
            {
                m_StatsListenerAdapter.ClientListener = listener;
            }

            return ec;
        }

        public override ErrorCode RemoveStatsCallback(IStatsListener listener)
        {
            StandardBroadcastApiHelper.ManagedStatsListener nativeListener = m_StatsListenerAdapter.ManagedListener;

            ErrorCode ec = TTV_CSharp_Broadcast_RemoveStatsCallback(ref nativeListener);
            if (Error.Succeeded(ec))
            {
                m_StatsListenerAdapter.ClientListener = null;
            }

            return ec;
        }

        public override ErrorCode PollStats()
        {
            return TTV_CSharp_Broadcast_PollStats();
        }

        #endregion

        #endregion
    }
}
