﻿using System;
using System.Collections.Generic;
using System.Runtime.InteropServices;

using SequenceId = System.UIntPtr;


namespace Twitch.Broadcast
{
    /// <summary>
    /// This class provides the common functionality for all delegate-based callback implementations of BroadcastApi.
    /// </summary>
    internal class StandardBroadcastApiHelper
    {
        #region Callback Signatures

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        internal delegate void TTV_CSharp_RequestAuthTokenCallback(ErrorCode ec, ref TTV_AuthToken result);
        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        internal delegate void TTV_CSharp_LoginCallback(ErrorCode ec, ref TTV_ChannelInfo result);
        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        internal delegate void TTV_CSharp_GetIngestListCallback(ErrorCode ec, ref TTV_IngestList result);
        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        internal delegate void TTV_CSharp_GetUserInfoCallback(ErrorCode ec, ref TTV_UserInfo result);
        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        internal delegate void TTV_CSharp_GetStreamInfoCallback(ErrorCode ec, ref TTV_StreamInfo result);
        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        internal delegate void TTV_CSharp_GetArchivingStateCallback(ErrorCode ec, ref TTV_ArchivingState result);
        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        internal delegate void TTV_CSharp_RunCommercialCallback(ErrorCode ec);
        //[UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        //internal delegate void TTV_CSharp_GetGameLiveStreamsCallback(ErrorCode ec, ref TTV_LiveGameStreamList result);
        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        internal delegate void TTV_CSharp_GetGameNameListCallback(ErrorCode ec, ref TTV_GameInfoList result);
        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        internal delegate void TTV_CSharp_BufferUnlockCallback(UIntPtr buffer);
        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        internal delegate void TTV_CSharp_StartCallback(ErrorCode ret);
        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        internal delegate void TTV_CSharp_StopCallback(ErrorCode ret);
        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        internal delegate void TTV_CSharp_SetStreamInfoCallback(ErrorCode ret);
        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        internal delegate void TTV_CSharp_SendActionMetaDataCallback(ErrorCode ret);
        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        internal delegate void TTV_CSharp_SendStartSpanMetaDataCallback(ErrorCode ret);
        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        internal delegate void TTV_CSharp_SendEndSpanMetaDataCallback(ErrorCode ret);

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        internal delegate void TTV_CSharp_StatCallback(StatType type, UInt64 data);

        // NOTE: This struct needs to be kept perfectly in sync with the strucy in csharp_broadcast.h
        internal struct ManagedBroadcastApiListener
        {
            public TTV_CSharp_RequestAuthTokenCallback requestAuthTokenCallback;
            public TTV_CSharp_LoginCallback loginCallback;
            public TTV_CSharp_GetIngestListCallback getIngestListCallback;
            public TTV_CSharp_GetUserInfoCallback getUserInfoCallback;
            public TTV_CSharp_GetStreamInfoCallback getStreamInfoCallback;
            public TTV_CSharp_GetArchivingStateCallback getArchivingStateCallback;
            public TTV_CSharp_RunCommercialCallback runCommercialCallback;
            //public TTV_CSharp_GetGameLiveStreamsCallback getGameLiveStreamsCallback;
            public TTV_CSharp_GetGameNameListCallback getGameNameListCallback;
            public TTV_CSharp_StartCallback startCallback;
            public TTV_CSharp_StopCallback stopCallback;
            public TTV_CSharp_SendActionMetaDataCallback sendActionMetaDataCallback;
            public TTV_CSharp_SendStartSpanMetaDataCallback sendStartSpanMetaDataCallback;
            public TTV_CSharp_SendEndSpanMetaDataCallback sendEndSpanMetaDataCallback;
            public TTV_CSharp_SetStreamInfoCallback setStreamInfoCallback;
            public TTV_CSharp_BufferUnlockCallback bufferUnlockCallback;
        }

        internal struct ManagedStatsListener
        {
            public TTV_CSharp_StatCallback statCallback;
        }

        internal class ApiListenerAdapter
        {
            private ManagedBroadcastApiListener m_ManagedListener;
            private IBroadcastApiListener m_ClientListener;

            public ManagedBroadcastApiListener ManagedListener
            {
                get { return m_ManagedListener; }
            }

            public IBroadcastApiListener ClientListener
            {
                get { return m_ClientListener; }
                set { m_ClientListener = value; }
            }

            public ApiListenerAdapter()
            {
                m_ManagedListener.requestAuthTokenCallback = RequestAuthTokenCallback;
                m_ManagedListener.loginCallback = LoginCallback;
                m_ManagedListener.getIngestListCallback = GetIngestListCallback;
                m_ManagedListener.getUserInfoCallback = GetUserInfoCallback;
                m_ManagedListener.getStreamInfoCallback = GetStreamInfoCallback;
                m_ManagedListener.getArchivingStateCallback = GetArchivingStateCallback;
                m_ManagedListener.runCommercialCallback = RunCommercialCallback;
                //m_ManagedListener.getGameLiveStreamsCallback = RequestAuthTokenCallback;
                m_ManagedListener.getGameNameListCallback = GetGameNameListCallback;
                m_ManagedListener.startCallback = StartCallback;
                m_ManagedListener.stopCallback = StopCallback;
                m_ManagedListener.sendActionMetaDataCallback = SendActionMetaDataCallback;
                m_ManagedListener.sendStartSpanMetaDataCallback = SendStartSpanMetaDataCallback;
                m_ManagedListener.sendEndSpanMetaDataCallback = SendEndSpanMetaDataCallback;
                m_ManagedListener.setStreamInfoCallback = SetStreamInfoCallback;
                m_ManagedListener.setStreamInfoCallback = SetStreamInfoCallback;
                m_ManagedListener.bufferUnlockCallback = BufferUnlockCallback;
            }

            private void RequestAuthTokenCallback(ErrorCode ec, ref TTV_AuthToken result)
            {
                try
                {
                    m_ClientListener.RequestAuthTokenCallback(ec, new AuthToken(ref result));
                }
                catch (Exception x)
                {
                    System.Diagnostics.Debug.WriteLine(x);
                }
            }

            private void LoginCallback(ErrorCode ec, ref TTV_ChannelInfo result)
            {
                try
                {
                    m_ClientListener.LoginCallback(ec, new ChannelInfo(ref result));
                }
                catch (Exception x)
                {
                    System.Diagnostics.Debug.WriteLine(x);
                }
            }

            private void GetIngestListCallback(ErrorCode ec, ref TTV_IngestList result)
            {
                try
                {
                    m_ClientListener.GetIngestServersCallback(ec, new IngestList(ref result));
                }
                catch (Exception x)
                {
                    System.Diagnostics.Debug.WriteLine(x);
                }
            }

            private void GetUserInfoCallback(ErrorCode ec, ref TTV_UserInfo result)
            {
                try
                {
                    m_ClientListener.GetUserInfoCallback(ec, new UserInfo(ref result));
                }
                catch (Exception x)
                {
                    System.Diagnostics.Debug.WriteLine(x);
                }
            }

            private void GetStreamInfoCallback(ErrorCode ec, ref TTV_StreamInfo result)
            {
                try
                {
                    m_ClientListener.GetStreamInfoCallback(ec, new StreamInfo(ref result));
                }
                catch (Exception x)
                {
                    System.Diagnostics.Debug.WriteLine(x);
                }
            }

            private void GetArchivingStateCallback(ErrorCode ec, ref TTV_ArchivingState result)
            {
                try
                {
                    m_ClientListener.GetArchivingStateCallback(ec, new ArchivingState(ref result));
                }
                catch (Exception x)
                {
                    System.Diagnostics.Debug.WriteLine(x);
                }
            }

            private void RunCommercialCallback(ErrorCode ec)
            {
                try
                {
                    m_ClientListener.RunCommercialCallback(ec);
                }
                catch (Exception x)
                {
                    System.Diagnostics.Debug.WriteLine(x);
                }
            }

            private void GetGameNameListCallback(ErrorCode ec, ref TTV_GameInfoList result)
            {
                try
                {
                    m_ClientListener.GetGameNameListCallback(ec, new GameInfoList(ref result));
                }
                catch (Exception x)
                {
                    System.Diagnostics.Debug.WriteLine(x);
                }
            }

            private void StartCallback(ErrorCode ec)
            {
                try
                {
                    m_ClientListener.StartCallback(ec);
                }
                catch (Exception x)
                {
                    System.Diagnostics.Debug.WriteLine(x);
                }
            }

            private void StopCallback(ErrorCode ec)
            {
                try
                {
                    m_ClientListener.StopCallback(ec);
                }
                catch (Exception x)
                {
                    System.Diagnostics.Debug.WriteLine(x);
                }
            }

            private void SendActionMetaDataCallback(ErrorCode ec)
            {
                try
                {
                    m_ClientListener.SendActionMetaDataCallback(ec);
                }
                catch (Exception x)
                {
                    System.Diagnostics.Debug.WriteLine(x);
                }
            }

            private void SendStartSpanMetaDataCallback(ErrorCode ec)
            {
                try
                {
                    m_ClientListener.SendStartSpanMetaDataCallback(ec);
                }
                catch (Exception x)
                {
                    System.Diagnostics.Debug.WriteLine(x);
                }
            }

            private void SendEndSpanMetaDataCallback(ErrorCode ec)
            {
                try
                {
                    m_ClientListener.SendEndSpanMetaDataCallback(ec);
                }
                catch (Exception x)
                {
                    System.Diagnostics.Debug.WriteLine(x);
                }
            }

            private void SetStreamInfoCallback(ErrorCode ec)
            {
                try
                {
                    m_ClientListener.SetStreamInfoCallback(ec);
                }
                catch (Exception x)
                {
                    System.Diagnostics.Debug.WriteLine(x);
                }
            }

            private void BufferUnlockCallback(UIntPtr buffer)
            {
                try
                {
                    m_ClientListener.BufferUnlockCallback(buffer);
                }
                catch (Exception x)
                {
                    System.Diagnostics.Debug.WriteLine(x);
                }
            }
        }

        internal class StatsListenerAdapter
        {
            private ManagedStatsListener m_ManagedListener;
            private IStatsListener m_ClientListener;

            public ManagedStatsListener ManagedListener
            {
                get { return m_ManagedListener; }
            }

            public IStatsListener ClientListener
            {
                get { return m_ClientListener; }
                set { m_ClientListener = value; }
            }

            public StatsListenerAdapter()
            {
                m_ManagedListener.statCallback = StatCallback;
            }

            private void StatCallback(StatType type, UInt64 data)
            {
                try
                {
                    m_ClientListener.StatCallback(type, data);
                }
                catch (Exception x)
                {
                    System.Diagnostics.Debug.WriteLine(x);
                }
            }
        }

        private ApiListenerAdapter m_ApiListenerAdapter = null;
        private StatsListenerAdapter m_StatsListenerAdapter = null;

        #endregion

        internal StandardBroadcastApiHelper(ApiListenerAdapter apiAdapter, StatsListenerAdapter statsAdapter)
        {
            m_ApiListenerAdapter = apiAdapter;
            m_StatsListenerAdapter = statsAdapter;
        }
    }
}
