﻿using System;
using System.Collections.Generic;
using System.Runtime.InteropServices;
using Twitch.Json;

using SequenceId = System.UIntPtr; // To handle ulong size properly


namespace Twitch.Broadcast
{
    /// <summary>
    /// http://docs.unity3d.com/Documentation/Manual/PluginsForIOS.html
    /// </summary>
    public class UnityIosBroadcastApi : UnityBroadcastApi
    {
        #region Callback Signatures

        internal delegate void QueuedEventDelegate();

        internal struct ManagedBroadcastApiListener
        {
            // nothing needs to be passed to native
        }

        internal struct ManagedStatsListener
        {
            // nothing needs to be passed to native
        }

        public unsafe void RequestAuthTokenCallback(string json)
        {
            (BroadcastApi.Instance as UnityIosBroadcastApi).Enqueue(() =>
                {
                    // deserialize json
                    Dictionary<string, object> values = JsonReader.Deserialize(json) as Dictionary<string, object>;
                    if (values == null)
                    {
                        throw new Exception("RequestAuthTokenCallback error parsing json: " + json);
                    }

                    // extract parameters
                    if (!values.ContainsKey("ec") || !(values["ec"] is Int64))
                    {
                        throw new Exception("RequestAuthTokenCallback missing valid 'ec' field");
                    }
                    if (!values.ContainsKey("result") || !(values["result"] is Int64))
                    {
                        throw new Exception("RequestAuthTokenCallback does not contain 'result'");
                    }

                    IntPtr authTokenPtr = IntPtr.Zero;

                    try
                    {
                        ErrorCode ec = (ErrorCode)(Int64)values["ec"];

                        authTokenPtr = new IntPtr((Int64)values["result"]);
                        AuthToken authToken = new AuthToken((TTV_AuthToken*)authTokenPtr);

                        m_ApiListenerAdapter.RequestAuthTokenCallback(ec, authToken);
                    }
                    finally
                    {
                        if (authTokenPtr != IntPtr.Zero)
                        {
                            TTV_CSharp_Broadcast_FreeAuthToken(authTokenPtr);
                        }
                    }
                });
        }

        public unsafe void LoginCallback(string json)
        {
            (BroadcastApi.Instance as UnityIosBroadcastApi).Enqueue(() =>
               {
                   // deserialize json
                   Dictionary<string, object> values = JsonReader.Deserialize(json) as Dictionary<string, object>;
                   if (values == null)
                   {
                       throw new Exception("LoginCallback error parsing json: " + json);
                   }

                   // extract parameters
                   if (!values.ContainsKey("ec") || !(values["ec"] is Int64))
                   {
                       throw new Exception("LoginCallback missing valid 'ec' field");
                   }
                   if (!values.ContainsKey("result") || !(values["result"] is Int64))
                   {
                       throw new Exception("LoginCallback does not contain 'result'");
                   }

                   IntPtr channelInfoPtr = IntPtr.Zero;

                   try
                   {
                       ErrorCode ec = (ErrorCode)(Int64)values["ec"];

                       channelInfoPtr = new IntPtr((Int64)values["result"]);
                       ChannelInfo channelInfo = new ChannelInfo((TTV_ChannelInfo*)channelInfoPtr);

                       m_ApiListenerAdapter.LoginCallback(ec, channelInfo);
                   }
                   finally
                   {
                       if (channelInfoPtr != IntPtr.Zero)
                       {
                           TTV_CSharp_Broadcast_FreeChannelInfo(channelInfoPtr);
                       }
                   }
               });
        }

        public unsafe void GetIngestListCallback(string json)
        {
            (BroadcastApi.Instance as UnityIosBroadcastApi).Enqueue(() =>
               {
                   // deserialize json
                   Dictionary<string, object> values = JsonReader.Deserialize(json) as Dictionary<string, object>;
                   if (values == null)
                   {
                       throw new Exception("GetIngestListCallback error parsing json: " + json);
                   }

                   // extract parameters
                   if (!values.ContainsKey("ec") || !(values["ec"] is Int64))
                   {
                       throw new Exception("GetIngestListCallback missing valid 'ec' field");
                   }
                   if (!values.ContainsKey("result") || !(values["result"] is Int64))
                   {
                       throw new Exception("GetIngestListCallback does not contain 'result'");
                   }

                   IntPtr ingestListPtr = IntPtr.Zero;

                   try
                   {
                       ErrorCode ec = (ErrorCode)(Int64)values["ec"];

                       ingestListPtr = new IntPtr((Int64)values["result"]);
                       IngestList ingestList = new IngestList((TTV_IngestList*)ingestListPtr);

                       m_ApiListenerAdapter.GetIngestListCallback(ec, ingestList);
                   }
                   finally
                   {
                       if (ingestListPtr != IntPtr.Zero)
                       {
                           TTV_CSharp_Broadcast_FreeIngestList(ingestListPtr);
                       }
                   }
               });
        }

        public unsafe void GetUserInfoCallback(string json)
        {
            (BroadcastApi.Instance as UnityIosBroadcastApi).Enqueue(() =>
                {
                    // deserialize json
                    Dictionary<string, object> values = JsonReader.Deserialize(json) as Dictionary<string, object>;
                    if (values == null)
                    {
                        throw new Exception("GetUserInfoCallback error parsing json: " + json);
                    }

                    // extract parameters
                    if (!values.ContainsKey("ec") || !(values["ec"] is Int64))
                    {
                        throw new Exception("GetUserInfoCallback missing valid 'ec' field");
                    }
                    if (!values.ContainsKey("result") || !(values["result"] is Int64))
                    {
                        throw new Exception("GetUserInfoCallback does not contain 'result'");
                    }

                    IntPtr userInfoPtr = IntPtr.Zero;

                    try
                    {
                        ErrorCode ec = (ErrorCode)(Int64)values["ec"];

                        userInfoPtr = new IntPtr((Int64)values["result"]);
                        UserInfo userInfo = new UserInfo((TTV_UserInfo*)userInfoPtr);

                        m_ApiListenerAdapter.GetUserInfoCallback(ec, userInfo);
                    }
                    finally
                    {
                        if (userInfoPtr != IntPtr.Zero)
                        {
                            TTV_CSharp_Broadcast_FreeUserInfo(userInfoPtr);
                        }
                    }
                });
        }

        public unsafe void GetStreamInfoCallback(string json)
        {
            (BroadcastApi.Instance as UnityIosBroadcastApi).Enqueue(() =>
                {
                    // deserialize json
                    Dictionary<string, object> values = JsonReader.Deserialize(json) as Dictionary<string, object>;
                    if (values == null)
                    {
                        throw new Exception("GetStreamInfoCallback error parsing json: " + json);
                    }

                    // extract parameters
                    if (!values.ContainsKey("ec") || !(values["ec"] is Int64))
                    {
                        throw new Exception("GetStreamInfoCallback missing valid 'ec' field");
                    }
                    if (!values.ContainsKey("result") || !(values["result"] is Int64))
                    {
                        throw new Exception("GetStreamInfoCallback does not contain 'result'");
                    }

                    IntPtr streamInfoPtr = IntPtr.Zero;

                    try
                    {
                        ErrorCode ec = (ErrorCode)(Int64)values["ec"];

                        streamInfoPtr = new IntPtr((Int64)values["result"]);
                        StreamInfo streamInfo = new StreamInfo((TTV_StreamInfo*)streamInfoPtr);

                        m_ApiListenerAdapter.GetStreamInfoCallback(ec, streamInfo);
                    }
                    finally
                    {
                        if (streamInfoPtr != IntPtr.Zero)
                        {
                            TTV_CSharp_Broadcast_FreeStreamInfo(streamInfoPtr);
                        }
                    }
                });
        }

        public unsafe void GetArchivingStateCallback(string json)
        {
            (BroadcastApi.Instance as UnityIosBroadcastApi).Enqueue(() =>
                {
                    // deserialize json
                    Dictionary<string, object> values = JsonReader.Deserialize(json) as Dictionary<string, object>;
                    if (values == null)
                    {
                        throw new Exception("GetArchivingStateCallback error parsing json: " + json);
                    }

                    // extract parameters
                    if (!values.ContainsKey("ec") || !(values["ec"] is Int64))
                    {
                        throw new Exception("GetArchivingStateCallback missing valid 'ec' field");
                    }
                    if (!values.ContainsKey("result") || !(values["result"] is Int64))
                    {
                        throw new Exception("GetArchivingStateCallback does not contain 'result'");
                    }

                    IntPtr archivingStatePtr = IntPtr.Zero;

                    try
                    {
                        ErrorCode ec = (ErrorCode)(Int64)values["ec"];

                        archivingStatePtr = new IntPtr((Int64)values["result"]);
                        ArchivingState state = new ArchivingState((TTV_ArchivingState*)archivingStatePtr);

                        m_ApiListenerAdapter.GetArchivingStateCallback(ec, state);
                    }
                    finally
                    {
                        if (archivingStatePtr != IntPtr.Zero)
                        {
                            TTV_CSharp_Broadcast_FreeArchivingState(archivingStatePtr);
                        }
                    }
                });
        }

        public unsafe void RunCommercialCallback(string json)
        {
            (BroadcastApi.Instance as UnityIosBroadcastApi).Enqueue(() =>
                {
                    // deserialize json
                    Dictionary<string, object> values = JsonReader.Deserialize(json) as Dictionary<string, object>;
                    if (values == null)
                    {
                        throw new Exception("RunCommercialCallback error parsing json: " + json);
                    }

                    // extract parameters
                    if (!values.ContainsKey("ec") || !(values["ec"] is Int64))
                    {
                        throw new Exception("RunCommercialCallback missing valid 'ec' field");
                    }

                    ErrorCode ec = (ErrorCode)(Int64)values["ec"];

                    m_ApiListenerAdapter.RunCommercialCallback(ec);
                });
        }

        //public unsafe void GetGameLiveStreamsCallback(string json)
        //{

        //}

        public unsafe void StartCallback(string json)
        {
            (BroadcastApi.Instance as UnityIosBroadcastApi).Enqueue(() =>
                {
                    // deserialize json
                    Dictionary<string, object> values = JsonReader.Deserialize(json) as Dictionary<string, object>;
                    if (values == null)
                    {
                        throw new Exception("StartCallback error parsing json: " + json);
                    }

                    // extract parameters
                    if (!values.ContainsKey("ec") || !(values["ec"] is Int64))
                    {
                        throw new Exception("StartCallback missing valid 'ec' field");
                    }

                    ErrorCode ec = (ErrorCode)(Int64)values["ec"];

                    m_ApiListenerAdapter.StartCallback(ec);
                });
        }

        public unsafe void StopCallback(string json)
        {
            (BroadcastApi.Instance as UnityIosBroadcastApi).Enqueue(() =>
                {
                    // deserialize json
                    Dictionary<string, object> values = JsonReader.Deserialize(json) as Dictionary<string, object>;
                    if (values == null)
                    {
                        throw new Exception("StopCallback error parsing json: " + json);
                    }

                    // extract parameters
                    if (!values.ContainsKey("ec") || !(values["ec"] is Int64))
                    {
                        throw new Exception("StopCallback missing valid 'ec' field");
                    }

                    ErrorCode ec = (ErrorCode)(Int64)values["ec"];

                    m_ApiListenerAdapter.StopCallback(ec);
                });
        }

        public unsafe void SendActionMetaDataCallback(string json)
        {
            (BroadcastApi.Instance as UnityIosBroadcastApi).Enqueue(() =>
               {
                   // deserialize json
                   Dictionary<string, object> values = JsonReader.Deserialize(json) as Dictionary<string, object>;
                   if (values == null)
                   {
                       throw new Exception("SendActionMetaDataCallback error parsing json: " + json);
                   }

                   // extract parameters
                   if (!values.ContainsKey("ec") || !(values["ec"] is Int64))
                   {
                       throw new Exception("SendActionMetaDataCallback missing valid 'ec' field");
                   }

                   ErrorCode ec = (ErrorCode)(Int64)values["ec"];

                   m_ApiListenerAdapter.SendActionMetaDataCallback(ec);
               });
        }

        public unsafe void SendStartSpanMetaDataCallback(string json)
        {
            (BroadcastApi.Instance as UnityIosBroadcastApi).Enqueue(() =>
                {
                    // deserialize json
                    Dictionary<string, object> values = JsonReader.Deserialize(json) as Dictionary<string, object>;
                    if (values == null)
                    {
                        throw new Exception("SendStartSpanMetaDataCallback error parsing json: " + json);
                    }

                    // extract parameters
                    if (!values.ContainsKey("ec") || !(values["ec"] is Int64))
                    {
                        throw new Exception("SendStartSpanMetaDataCallback missing valid 'ec' field");
                    }

                    ErrorCode ec = (ErrorCode)(Int64)values["ec"];

                    m_ApiListenerAdapter.SendStartSpanMetaDataCallback(ec);
                });
        }

        public unsafe void SendEndSpanMetaDataCallback(string json)
        {
            (BroadcastApi.Instance as UnityIosBroadcastApi).Enqueue(() =>
               {
                   // deserialize json
                   Dictionary<string, object> values = JsonReader.Deserialize(json) as Dictionary<string, object>;
                   if (values == null)
                   {
                       throw new Exception("SendEndSpanMetaDataCallback error parsing json: " + json);
                   }

                   // extract parameters
                   if (!values.ContainsKey("ec") || !(values["ec"] is Int64))
                   {
                       throw new Exception("SendEndSpanMetaDataCallback missing valid 'ec' field");
                   }

                   ErrorCode ec = (ErrorCode)(Int64)values["ec"];

                   m_ApiListenerAdapter.SendEndSpanMetaDataCallback(ec);
               });
        }

        public unsafe void SetStreamInfoCallback(string json)
        {
            (BroadcastApi.Instance as UnityIosBroadcastApi).Enqueue(() =>
                {
                    // deserialize json
                    Dictionary<string, object> values = JsonReader.Deserialize(json) as Dictionary<string, object>;
                    if (values == null)
                    {
                        throw new Exception("SetStreamInfoCallback error parsing json: " + json);
                    }

                    // extract parameters
                    if (!values.ContainsKey("ec") || !(values["ec"] is Int64))
                    {
                        throw new Exception("SetStreamInfoCallback missing valid 'ec' field");
                    }

                    ErrorCode ec = (ErrorCode)(Int64)values["ec"];

                    m_ApiListenerAdapter.SetStreamInfoCallback(ec);
                });
        }

        public unsafe void BufferUnlockCallback(string json)
        {
            (BroadcastApi.Instance as UnityIosBroadcastApi).Enqueue(() =>
                {
                    // deserialize json
                    Dictionary<string, object> values = JsonReader.Deserialize(json) as Dictionary<string, object>;
                    if (values == null)
                    {
                        throw new Exception("BufferUnlockCallback error parsing json: " + json);
                    }

                    // extract parameters
                    if (!values.ContainsKey("buffer") || !(values["buffer"] is Int64))
                    {
                        throw new Exception("BufferUnlockCallback does not contain 'buffer'");
                    }

                    UIntPtr buffer = new UIntPtr((UInt64)(Int64)values["buffer"]);

                    m_ApiListenerAdapter.BufferUnlockCallback(buffer);
                });
        }

        public unsafe void StatCallback(string json)
        {
            (BroadcastApi.Instance as UnityIosBroadcastApi).Enqueue(() =>
                {
                    // deserialize json
                    Dictionary<string, object> values = JsonReader.Deserialize(json) as Dictionary<string, object>;
                    if (values == null)
                    {
                        throw new Exception("StatCallback error parsing json: " + json);
                    }

                    // extract parameters
                    if (!values.ContainsKey("type") || !(values["type"] is Int64))
                    {
                        throw new Exception("StatCallback does not contain 'type'");
                    }
                    if (!values.ContainsKey("data") || !(values["data"] is Int64))
                    {
                        throw new Exception("StatCallback does not contain 'data'");
                    }

                    StatType type = (StatType)(Int64)values["type"];
                    UInt64 data = (UInt64)(Int64)values["data"];

                    m_StatsListenerAdapter.StatCallback(type, data);
                });
        }

        private class BroadcastListenerAdapter
        {
            private ManagedBroadcastApiListener m_ManagedListener;
            private IBroadcastApiListener m_ClientListener;

            public ManagedBroadcastApiListener ManagedListener
            {
                get { return m_ManagedListener; }
            }

            public IBroadcastApiListener ClientListener
            {
                get { return m_ClientListener; }
                set { m_ClientListener = value; }
            }

            public BroadcastListenerAdapter()
            {
            }

            internal void RequestAuthTokenCallback(ErrorCode ec, AuthToken result)
            {
                try
                {
                    if (m_ClientListener != null)
                    {
                        m_ClientListener.RequestAuthTokenCallback(ec, result);
                    }
                }
                catch (Exception x)
                {
                    System.Diagnostics.Debug.WriteLine(x);
                }
            }

            public unsafe void LoginCallback(ErrorCode ec, ChannelInfo result)
            {
                try
                {
                    if (m_ClientListener != null)
                    {
                        m_ClientListener.LoginCallback(ec, result);
                    }
                }
                catch (Exception x)
                {
                    System.Diagnostics.Debug.WriteLine(x);
                }
            }

            public void GetIngestListCallback(ErrorCode ec, IngestList result)
            {
                try
                {
                    if (m_ClientListener != null)
                    {
                        m_ClientListener.GetIngestServersCallback(ec, result);
                    }
                }
                catch (Exception x)
                {
                    System.Diagnostics.Debug.WriteLine(x);
                }
            }

            public void GetUserInfoCallback(ErrorCode ec, UserInfo result)
            {
                try
                {
                    if (m_ClientListener != null)
                    {
                        m_ClientListener.GetUserInfoCallback(ec, result);
                    }
                }
                catch (Exception x)
                {
                    System.Diagnostics.Debug.WriteLine(x);
                }
            }

            public void GetStreamInfoCallback(ErrorCode ec, StreamInfo result)
            {
                try
                {
                    if (m_ClientListener != null)
                    {
                        m_ClientListener.GetStreamInfoCallback(ec, result);
                    }
                }
                catch (Exception x)
                {
                    System.Diagnostics.Debug.WriteLine(x);
                }
            }

            public void GetArchivingStateCallback(ErrorCode ec, ArchivingState result)
            {
                try
                {
                    if (m_ClientListener != null)
                    {
                        m_ClientListener.GetArchivingStateCallback(ec, result);
                    }
                }
                catch (Exception x)
                {
                    System.Diagnostics.Debug.WriteLine(x);
                }
            }

            public void RunCommercialCallback(ErrorCode ec)
            {
                try
                {
                    if (m_ClientListener != null)
                    {
                        m_ClientListener.RunCommercialCallback(ec);
                    }
                }
                catch (Exception x)
                {
                    System.Diagnostics.Debug.WriteLine(x);
                }
            }

            //public void GetGameLiveStreamsCallback(ErrorCode ec, GameInfoList result)
            //{

            //}

            public void StartCallback(ErrorCode ec)
            {
                try
                {
                    if (m_ClientListener != null)
                    {
                        m_ClientListener.StartCallback(ec);
                    }
                }
                catch (Exception x)
                {
                    System.Diagnostics.Debug.WriteLine(x);
                }
            }

            public void StopCallback(ErrorCode ec)
            {
                try
                {
                    if (m_ClientListener != null)
                    {
                        m_ClientListener.StopCallback(ec);
                    }
                }
                catch (Exception x)
                {
                    System.Diagnostics.Debug.WriteLine(x);
                }
            }

            public void SendActionMetaDataCallback(ErrorCode ec)
            {
                try
                {
                    if (m_ClientListener != null)
                    {
                        m_ClientListener.SendActionMetaDataCallback(ec);
                    }
                }
                catch (Exception x)
                {
                    System.Diagnostics.Debug.WriteLine(x);
                }
            }

            public void SendStartSpanMetaDataCallback(ErrorCode ec)
            {
                try
                {
                    if (m_ClientListener != null)
                    {
                        m_ClientListener.SendStartSpanMetaDataCallback(ec);
                    }
                }
                catch (Exception x)
                {
                    System.Diagnostics.Debug.WriteLine(x);
                }
            }

            public void SendEndSpanMetaDataCallback(ErrorCode ec)
            {
                try
                {
                    if (m_ClientListener != null)
                    {
                        m_ClientListener.SendEndSpanMetaDataCallback(ec);
                    }
                }
                catch (Exception x)
                {
                    System.Diagnostics.Debug.WriteLine(x);
                }
            }

            public void SetStreamInfoCallback(ErrorCode ec)
            {
                try
                {
                    if (m_ClientListener != null)
                    {
                        m_ClientListener.SetStreamInfoCallback(ec);
                    }
                }
                catch (Exception x)
                {
                    System.Diagnostics.Debug.WriteLine(x);
                }
            }

            public void BufferUnlockCallback(UIntPtr buffer)
            {
                try
                {
                    if (m_ClientListener != null)
                    {
                        m_ClientListener.BufferUnlockCallback(buffer);
                    }
                }
                catch (Exception x)
                {
                    System.Diagnostics.Debug.WriteLine(x);
                }
            }
        }

        internal class StatsListenerAdapter
        {
            private ManagedStatsListener m_ManagedListener;
            private IStatsListener m_ClientListener;

            public ManagedStatsListener ManagedListener
            {
                get { return m_ManagedListener; }
            }

            public IStatsListener ClientListener
            {
                get { return m_ClientListener; }
                set { m_ClientListener = value; }
            }

            public StatsListenerAdapter()
            {
            }

            public void StatCallback(StatType type, UInt64 data)
            {
                try
                {
                    if (m_ClientListener != null)
                    {
                        m_ClientListener.StatCallback(type, data);
                    }
                }
                catch (Exception x)
                {
                    System.Diagnostics.Debug.WriteLine(x);
                }
            }
        }

        #endregion

        #region Native Functions

        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_Init")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_Init(ref ManagedBroadcastApiListener listener);
        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_Shutdown")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_Shutdown();

        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_RequestAuthToken")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_RequestAuthToken(ref TTV_AuthParams authParams, UInt32 flags);
        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_Login")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_Login(ref TTV_AuthToken authToken);
        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_GetIngestServers")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_GetIngestServers(ref TTV_AuthToken authToken);
        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_GetUserInfo")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_GetUserInfo(ref TTV_AuthToken authToken);
        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_GetStreamInfo")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_GetStreamInfo(ref TTV_AuthToken authToken, string channel);
        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_SetStreamInfo")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_SetStreamInfo(ref TTV_AuthToken authToken, string channel, ref TTV_StreamInfoForSetting streamInfoToSet);
        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_GetArchivingState")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_GetArchivingState(ref TTV_AuthToken authToken);
        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_RunCommercial")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_RunCommercial(ref TTV_AuthToken authToken);
        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_SetVolume")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_SetVolume(TTV_AudioDeviceType device, float volume);
        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_GetVolume")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_GetVolume(TTV_AudioDeviceType device, ref float volume);
        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_GetGameNameList")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_GetGameNameList(string str);
        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_FreeGameNameList")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_FreeGameNameList(IntPtr p);
        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_GetDefaultParams")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_GetDefaultParams(ref TTV_VideoParams vidParams);
        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_GetMaxResolution")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_GetMaxResolution(uint maxKbps, uint frameRate, float bitsPerPixel, float aspectRatio, ref uint width, ref uint height);
        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_PollTasks")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_PollTasks();
        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_SendActionMetaData")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_SendActionMetaData(ref TTV_AuthToken authToken, string name, UInt64 streamTime, string humanDescription, string data);
        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_SendStartSpanMetaData")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_SendStartSpanMetaData(ref TTV_AuthToken authToken, string name, UInt64 streamTime, out SequenceId sequenceId, string humanDescription, string data);
        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_SendEndSpanMetaData")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_SendEndSpanMetaData(ref TTV_AuthToken authToken, string name, UInt64 streamTime, SequenceId sequenceId, string humanDescription, string data);
        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_SubmitVideoFrame")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_SubmitVideoFrame(UIntPtr frameBuffer);
        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_SubmitAudioSamples")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_SubmitAudioSamples(Int16[] samples, uint numSamples);
        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_PauseVideo")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_PauseVideo();
        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_AllocateFrameBuffer")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_AllocateFrameBuffer(uint size, out UIntPtr buffer);
        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_FreeFrameBuffer")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_FreeFrameBuffer(UIntPtr buffer);
        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_RandomizeFrameBuffer")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_RandomizeFrameBuffer(UIntPtr buffer, uint size);
        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_GetStreamTime")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_GetStreamTime(out UInt64 timeMs);

        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_FreeAuthToken")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_FreeAuthToken(IntPtr p);
        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_FreeChannelInfo")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_FreeChannelInfo(IntPtr p);
        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_FreeUserInfo")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_FreeUserInfo(IntPtr p);
        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_FreeIngestList")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_FreeIngestList(IntPtr p);
        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_FreeStreamInfo")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_FreeStreamInfo(IntPtr p);
        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_FreeArchivingState")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_FreeArchivingState(IntPtr p);

        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_RegisterStatsCallback")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_RegisterStatsCallback(ref ManagedStatsListener listener);
        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_RemoveStatsCallback")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_RemoveStatsCallback(ref ManagedStatsListener listener);
        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Broadcast_PollStats")]
        internal static extern ErrorCode TTV_CSharp_Broadcast_PollStats();

        #region UnityBroadcastApi

        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_Unity_Broadcast_Init")]
        internal static extern ErrorCode TTV_Unity_Broadcast_Init();
        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_Unity_Broadcast_Shutdown")]
        internal static extern ErrorCode TTV_Unity_Broadcast_Shutdown();
        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_Unity_Broadcast_Start")]
        internal static extern ErrorCode TTV_Unity_Broadcast_Start(ref TTV_VideoParams videoParams, ref TTV_AudioParams audioParams, ref TTV_IngestServer ingestServer, UInt32 flags, bool async);
        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_Unity_Broadcast_Stop")]
        internal static extern ErrorCode TTV_Unity_Broadcast_Stop(bool async);
        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_Unity_Broadcast_SubmitTexture")]
        internal static extern ErrorCode TTV_Unity_Broadcast_SubmitTexture(IntPtr texture, int width, int height);
        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_Unity_Broadcast_SubmitAudioSamples")]
        internal static extern ErrorCode TTV_Unity_Broadcast_SubmitAudioSamples(float[] samples, uint numSamples, uint numChannels, uint sampleRate);
        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_Unity_Broadcast_GetCapturePixelFormat")]
        internal static extern ErrorCode TTV_Unity_Broadcast_GetCapturePixelFormat(ref PixelFormat format);

        #endregion

        #endregion

        #region BroadcastApi Implementation

        private BroadcastListenerAdapter m_ApiListenerAdapter = null;
        private StatsListenerAdapter m_StatsListenerAdapter = null;
        private List<QueuedEventDelegate> m_QueuedEvents = new List<QueuedEventDelegate>();

        internal void Enqueue(QueuedEventDelegate func)
        {
            lock (m_QueuedEvents)
            {
                m_QueuedEvents.Add(func);
            }
        }

        public override IBroadcastApiListener BroadcastApiListener
        {
            get { return m_BroadcastApiListener; }
            set
            {
                if (m_BroadcastApiListener == value)
                {
                    return;
                }

                if (m_BroadcastApiListener != null)
                {
                    m_ApiListenerAdapter.ClientListener = null;
                }

                m_BroadcastApiListener = value;

                if (m_BroadcastApiListener != null)
                {
                    m_BroadcastApiListener = value;

                    m_ApiListenerAdapter.ClientListener = m_BroadcastApiListener;
                }
            }
        }

        public UnityIosBroadcastApi()
        {
            m_ApiListenerAdapter = new BroadcastListenerAdapter();
            m_StatsListenerAdapter = new StatsListenerAdapter();
        }

        public override ErrorCode Initialize()
        {
            m_ApiListenerAdapter.ClientListener = m_BroadcastApiListener;

            ManagedBroadcastApiListener nativeListener = m_ApiListenerAdapter.ManagedListener;
            ErrorCode ec = TTV_CSharp_Broadcast_Init(ref nativeListener);

            if (Error.Succeeded(ec))
            {
                ec = TTV_Unity_Broadcast_Init();
            }

            return ec;
        }

        public override ErrorCode Shutdown()
        {
            ErrorCode ec = TTV_CSharp_Broadcast_Shutdown();

            if (Error.Succeeded(ec))
            {
                m_ApiListenerAdapter.ClientListener = null;
            }

            ec = TTV_Unity_Broadcast_Shutdown();

            return ec;
        }

        public override ErrorCode RequestAuthToken(AuthParams authParams, AuthFlag flags)
        {
            return TTV_CSharp_Broadcast_RequestAuthToken(ref authParams.mStruct, (UInt32)flags);
        }

        public override ErrorCode Login(AuthToken authToken)
        {
            return TTV_CSharp_Broadcast_Login(ref authToken.mStruct);
        }

        public override ErrorCode GetIngestServers(AuthToken authToken)
        {
            return TTV_CSharp_Broadcast_GetIngestServers(ref authToken.mStruct);
        }

        public override ErrorCode GetUserInfo(AuthToken authToken)
        {
            return TTV_CSharp_Broadcast_GetUserInfo(ref authToken.mStruct);
        }

        public override ErrorCode GetStreamInfo(AuthToken authToken, string channel)
        {
            return TTV_CSharp_Broadcast_GetStreamInfo(ref authToken.mStruct, channel);
        }

        public override ErrorCode SetStreamInfo(AuthToken authToken, string channel, StreamInfoForSetting streamInfoToSet)
        {
            return TTV_CSharp_Broadcast_SetStreamInfo(ref authToken.mStruct, channel, ref streamInfoToSet.mStruct);
        }

        public override ErrorCode GetArchivingState(AuthToken authToken)
        {
            return TTV_CSharp_Broadcast_GetArchivingState(ref authToken.mStruct);
        }

        public override ErrorCode RunCommercial(AuthToken authToken)
        {
            return TTV_CSharp_Broadcast_RunCommercial(ref authToken.mStruct);
        }

        public override ErrorCode SetVolume(TTV_AudioDeviceType device, float volume)
        {
            return TTV_CSharp_Broadcast_SetVolume(device, volume);
        }

        public override ErrorCode GetVolume(TTV_AudioDeviceType device, ref float volume)
        {
            return TTV_CSharp_Broadcast_GetVolume(device, ref volume);
        }

        public override ErrorCode GetGameNameList(string str)
        {
            return TTV_CSharp_Broadcast_GetGameNameList(str);
        }

        public override ErrorCode GetDefaultParams(VideoParams vidParams)
        {
            return TTV_CSharp_Broadcast_GetDefaultParams(ref vidParams.mStruct);
        }

        public override ErrorCode GetMaxResolution(uint maxKbps, uint frameRate, float bitsPerPixel, float aspectRatio, ref uint width, ref uint height)
        {
            return TTV_CSharp_Broadcast_GetMaxResolution(maxKbps, frameRate, bitsPerPixel, aspectRatio, ref width, ref height);
        }

        public override ErrorCode PollTasks()
        {
            try
            {
                m_InsidePollTasks = true;

                // fire queued up events
                List<QueuedEventDelegate> copy = new List<QueuedEventDelegate>();

                lock (m_QueuedEvents)
                {
                    copy.AddRange(m_QueuedEvents);
                    m_QueuedEvents.Clear();
                }

                foreach (QueuedEventDelegate e in copy)
                {
                    e();
                }

                return TTV_CSharp_Broadcast_PollTasks();
            }
            finally
            {
                m_InsidePollTasks = false;
            }
        }

        public override ErrorCode SendActionMetaData(AuthToken authToken, string name, UInt64 streamTime, string humanDescription, string data)
        {
            return TTV_CSharp_Broadcast_SendActionMetaData(ref authToken.mStruct, name, streamTime, humanDescription, data);
        }

        public override ErrorCode SendStartSpanMetaData(AuthToken authToken, string name, UInt64 streamTime, out ulong sequenceId, string humanDescription, string data)
        {
            SequenceId s;
            ErrorCode ec = TTV_CSharp_Broadcast_SendStartSpanMetaData(ref authToken.mStruct, name, streamTime, out s, humanDescription, data);
            sequenceId = (ulong)s;
            return ec;
        }

        public override ErrorCode SendEndSpanMetaData(AuthToken authToken, string name, UInt64 streamTime, ulong sequenceId, string humanDescription, string data)
        {
            return TTV_CSharp_Broadcast_SendEndSpanMetaData(ref authToken.mStruct, name, streamTime, (SequenceId)sequenceId, humanDescription, data);
        }

        public override ErrorCode SubmitVideoFrame(UIntPtr frameBuffer)
        {
            return TTV_CSharp_Broadcast_SubmitVideoFrame(frameBuffer);
        }

        public override ErrorCode SubmitAudioSamples(Int16[] samples, uint numSamples)
        {
            return TTV_CSharp_Broadcast_SubmitAudioSamples(samples, numSamples);
        }

        public override ErrorCode PauseVideo()
        {
            return TTV_CSharp_Broadcast_PauseVideo();
        }

        public override ErrorCode AllocateFrameBuffer(uint size, out UIntPtr buffer)
        {
            return TTV_CSharp_Broadcast_AllocateFrameBuffer(size, out buffer);
        }

        public override ErrorCode FreeFrameBuffer(UIntPtr buffer)
        {
            return TTV_CSharp_Broadcast_FreeFrameBuffer(buffer);
        }

        public override ErrorCode RandomizeFrameBuffer(UIntPtr buffer, uint size)
        {
            return TTV_CSharp_Broadcast_RandomizeFrameBuffer(buffer, size);
        }

        public override ErrorCode Start(VideoParams videoParams, AudioParams audioParams, IngestServer ingestServer, StartFlags flags, bool async)
        {
            return TTV_Unity_Broadcast_Start(ref videoParams.mStruct, ref audioParams.mStruct, ref ingestServer.mStruct, (uint)flags, async);
        }

        public override ErrorCode GetStreamTime(out UInt64 timeMs)
        {
            return TTV_CSharp_Broadcast_GetStreamTime(out timeMs);
        }

        public override ErrorCode SubmitTexturePointer(IntPtr texturePointer, int width, int height)
        {
            return TTV_Unity_Broadcast_SubmitTexture(texturePointer, width, height);
        }

        public override ErrorCode SubmitAudioSamples(float[] samples, uint numSamples, uint numChannels, uint sampleRate)
        {
            return TTV_Unity_Broadcast_SubmitAudioSamples(samples, numSamples, numChannels, sampleRate);
        }

        public override ErrorCode Stop(bool async)
        {
            return TTV_Unity_Broadcast_Stop(async);
        }

        public override ErrorCode GetCapturePixelFormat(ref PixelFormat format)
        {
            return TTV_Unity_Broadcast_GetCapturePixelFormat(ref format);
        }

        #region Stats

        public override ErrorCode RegisterStatsCallback(IStatsListener listener)
        {
            if (m_StatsListenerAdapter.ClientListener != null)
            {
                return ErrorCode.TTV_EC_ALREADY_INITIALIZED;
            }

            m_StatsListenerAdapter.ClientListener = listener;

            ManagedStatsListener nativeListener = m_StatsListenerAdapter.ManagedListener;
            return TTV_CSharp_Broadcast_RegisterStatsCallback(ref nativeListener);
        }

        public override ErrorCode RemoveStatsCallback(IStatsListener listener)
        {
            ManagedStatsListener nativeListener = m_StatsListenerAdapter.ManagedListener;

            ErrorCode ec = TTV_CSharp_Broadcast_RemoveStatsCallback(ref nativeListener);

            if (Error.Succeeded(ec))
            {
                m_StatsListenerAdapter.ClientListener = null;
            }

            return ec;
        }

        public override ErrorCode PollStats()
        {
            return TTV_CSharp_Broadcast_PollStats();
        }

        #endregion

        #endregion
    }
}
