﻿using System;
using System.Collections.Generic;
using System.Runtime.InteropServices;


namespace Twitch.Chat
{
    public static class Constants
    {
        public static readonly int kMaxStringLength = 1000;
    }

    /**
     * TTV_ChatUserMode - A list of the mode flags a user may have.
     */
    public enum TTV_ChatUserMode
    {
        TTV_CHAT_USERMODE_VIEWER = 0,		            //!< A regular viewer.

        TTV_CHAT_USERMODE_MODERATOR = 1 << 0,	        //!< A moderator.
        TTV_CHAT_USERMODE_BROADCASTER = 1 << 1,	        //!< The broadcaster.
        TTV_CHAT_USERMODE_ADMINISTRATOR = 1 << 2,	    //!< An admin.
        TTV_CHAT_USERMODE_STAFF = 1 << 3,	            //!< A member of Twitch.
        TTV_CHAT_USERMODE_SYSTEM = 1 << 4,	            //!< The Twitch system.
        TTV_CHAT_USERMODE_GLOBALMODERATOR = 1 << 5,	    //!< A global mod.

        TTV_CHAT_USERMODE_BANNED = 1 << 30,	            //!< The user has been banned.  This flag may not always be up to date.
    }

    /**
     * TTV_ChatUserSubscription - A list of the subscription flags a user may have.
     */
    public enum TTV_ChatUserSubscription
    {
        TTV_CHAT_USERSUB_NONE = 0,		        //!< A standard user.

        TTV_CHAT_USERSUB_SUBSCRIBER = 1 << 0,	//!< A subscriber in the current channel.
        TTV_CHAT_USERSUB_TURBO = 1 << 1,	    //!< A Twitch Turbo account.
    }

    /**
     * Controls the way chat messages are passed to the client.
     */
    public enum TTV_ChatTokenizationOption
    {
        TTV_CHAT_TOKENIZATION_OPTION_NONE = 0,

        TTV_CHAT_TOKENIZATION_OPTION_EMOTICONS = 1 << 0,
        TTV_CHAT_TOKENIZATION_OPTION_MENTIONS = 1 << 1,
        TTV_CHAT_TOKENIZATION_OPTION_URLS = 1 << 2,

        TTV_CHAT_TOKENIZATION_OPTION_ALL =  TTV_CHAT_TOKENIZATION_OPTION_EMOTICONS | 
                                            TTV_CHAT_TOKENIZATION_OPTION_MENTIONS |
                                            TTV_CHAT_TOKENIZATION_OPTION_URLS
    }

    /**
     * The current state of the chat connection.
     */
    public enum TTV_ChatChannelState
    {
        TTV_CHAT_CHANNEL_STATE_DISCONNECTED,
        TTV_CHAT_CHANNEL_STATE_CONNECTING,
        TTV_CHAT_CHANNEL_STATE_CONNECTED,
        TTV_CHAT_CHANNEL_STATE_DISCONNECTING,
    }

    /**
     * TTV_ChatRestrictionReason - The reason for not allowing the local user to chat right now.  The UI for entering messages should be enabled only if TTV_CHAT_RESTRICTION_REASON_NONE is received.
     */
    public enum TTV_ChatRestrictionReason
    {
        TTV_CHAT_RESTRICTION_REASON_NONE = 0,		//!< Allowed to chat right now.
        TTV_CHAT_RESTRICTION_REASON_ANONYMOUS = 1 << 0,	//!< Logged in anonymously so can't chat.
        TTV_CHAT_RESTRICTION_REASON_SUBSCRIBERS_ONLY = 1 << 1,	//!< The channel is in subscribers only mode.
        TTV_CHAT_RESTRICTION_REASON_SLOW_MODE = 1 << 2,	//!< The channel is in slow mode and messages are being paced.  Expect TTV_CHAT_RESTRICTION_REASON_NONE when the timeout is up.
        TTV_CHAT_RESTRICTION_REASON_TIMEOUT = 1 << 3,	//!< The user has been banned for an amount of time.  Expect TTV_CHAT_RESTRICTION_REASON_NONE when the timeout is up.
        TTV_CHAT_RESTRICTION_REASON_BANNED = 1 << 4,	//!< The user has been permanently banned.  Expect TTV_CHAT_RESTRICTION_REASON_NONE if a moderator ever unbans.
        TTV_CHAT_RESTRICTION_REASON_LANGUAGE = 1 << 5	//!< The user does not speak the active broadcaster language and has not opted in yet.
    }

    /**
     * TTV_ChatMessageFlag - Flags decribing the message variation.  This are intended to be ORed as needed.
     */
    public enum TTV_ChatMessageFlag
    {
        TTV_CHAT_MESSAGE_FLAG_NONE = 0,		//!< A regular message.
        TTV_CHAT_MESSAGE_FLAG_ACTION = 1 << 0,	//!< Whether or not the message is an action.  If true, it should be displayed entirely in the name text color and of the form "<userName> <message>".
        TTV_CHAT_MESSAGE_FLAG_WHISPER = 1 << 1,	//!< Whether or not the message is a received whisper.
        TTV_CHAT_MESSAGE_FLAG_WHISPER_SENT = 1 << 2,	//!< Whether or not the message is a sent whisper.
        TTV_CHAT_MESSAGE_FLAG_IGNORED = 1 << 3  //!< Whether or not the message was sent by a blocked user.
    }

    #region ChatUserInfo

    /**
     * TTV_ChatUserInfo - The user information.
     */
    public unsafe struct TTV_ChatUserInfo
    {
        public fixed byte userName[64];
        public TTV_ChatUserMode modes;
        public TTV_ChatUserSubscription subscriptions;
        public uint nameColorARGB;
    }

    public class ChatUserInfo
    {
        protected string userName = null;
        protected TTV_ChatUserMode modes;
        public TTV_ChatUserSubscription subscriptions;				//!< The user's subscriptions for the channel.
        public uint nameColorARGB;							//!< The current ARGB color of the user's name text.

        public ChatUserInfo()
        {
            userName = "";
            modes = TTV_ChatUserMode.TTV_CHAT_USERMODE_VIEWER;
            subscriptions = TTV_ChatUserSubscription.TTV_CHAT_USERSUB_NONE;
            nameColorARGB = 0;
        }

        internal unsafe ChatUserInfo(ref TTV_ChatUserInfo info)
        {
            fixed (byte* b = info.userName)
            {
                userName = MarshalUtil.StringFromBytes(b, 64);
            }

            modes = info.modes;
            subscriptions = info.subscriptions;
            nameColorARGB = info.nameColorARGB;
        }

        internal unsafe ChatUserInfo(TTV_ChatUserInfo* info)
        {
            userName = MarshalUtil.StringFromBytes(info->userName, 64);

            modes = info->modes;
            subscriptions = info->subscriptions;
            nameColorARGB = info->nameColorARGB;
        }

        public ChatUserInfo Copy()
        {
            ChatUserInfo copy = new ChatUserInfo();
            copy.userName = userName;
            copy.modes = modes;
            copy.subscriptions = subscriptions;
            copy.nameColorARGB = nameColorARGB;
            return copy;
        }

        public unsafe string UserName
        {
            get { return userName; }
        }

        public TTV_ChatUserMode Modes
        {
            get { return modes; }
        }

        public TTV_ChatUserSubscription Subscriptions
        {
            get { return subscriptions; }
        }

        public uint NameColorARGB
        {
            get { return nameColorARGB; }
        }

        public bool IsBanned
        {
            get { return (modes & TTV_ChatUserMode.TTV_CHAT_USERMODE_BANNED) != 0; }
        }

        public bool IsModerator
        {
            get { return (modes & TTV_ChatUserMode.TTV_CHAT_USERMODE_MODERATOR) != 0; }
        }

        public bool IsGlobalModerator
        {
            get { return (modes & TTV_ChatUserMode.TTV_CHAT_USERMODE_GLOBALMODERATOR) != 0; }
        }

        public bool IsBroadcaster
        {
            get { return (modes & TTV_ChatUserMode.TTV_CHAT_USERMODE_BROADCASTER) != 0; }
        }

        public bool IsViewer
        {
            get { return (modes & TTV_ChatUserMode.TTV_CHAT_USERMODE_VIEWER) != 0; }
        }

        public bool IsAdministrator
        {
            get { return (modes & TTV_ChatUserMode.TTV_CHAT_USERMODE_ADMINISTRATOR) != 0; }
        }

        public bool IsStaff
        {
            get { return (modes & TTV_ChatUserMode.TTV_CHAT_USERMODE_STAFF) != 0; }
        }

        public bool IsSubscriber
        {
            get { return (subscriptions & TTV_ChatUserSubscription.TTV_CHAT_USERSUB_SUBSCRIBER) != 0; }
        }

        public bool IsTurbo
        {
            get { return (subscriptions & TTV_ChatUserSubscription.TTV_CHAT_USERSUB_TURBO) != 0; }
        }

        public override bool Equals(object obj)
        {
            ChatUserInfo other = obj as ChatUserInfo;
            if (other == null)
            {
                return false;
            }
            return other.UserName.Equals(userName);
        }

        public override int GetHashCode()
        {
            return userName.GetHashCode();
        }

        public override string ToString()
        {
            System.Text.StringBuilder sb = new System.Text.StringBuilder();

            sb.Append(userName).Append(" [");

            if (this.IsAdministrator)
            {
                sb.Append("Admin");
            }
            if (this.IsBroadcaster)
            {
                sb.Append("Broad");
            }
            if (this.IsBanned)
            {
                sb.Append("Ban");
            }
            if (this.IsModerator)
            {
                sb.Append("Mod");
            }
            if (this.IsGlobalModerator)
            {
                sb.Append("Gblmod");
            }
            if (this.IsStaff)
            {
                sb.Append("Staff");
            }
            if (this.IsViewer)
            {
                sb.Append("Viewer");
            }
            if (this.IsSubscriber)
            {
                sb.Append("Sub");
            }
            if (this.IsTurbo)
            {
                sb.Append("Turbo");
            }

            sb.Append("]");

            return sb.ToString();
        }
    }

    #endregion

    #region ChatUserList

    /**
     * TTV_ChatUserList - A list of chat users.
     */
    public unsafe struct TTV_ChatUserList
    {
        public TTV_ChatUserInfo* userList;		//!< The array of user info entries.
        public uint userCount;			//!< The number of entries in the array.
    }

    public class ChatUserList
    {
        protected ChatUserInfo[] users = null;

        internal unsafe ChatUserList(ref TTV_ChatUserList list)
        {
            users = new ChatUserInfo[list.userCount];
            for (int i = 0; i < users.Length; ++i)
            {
                users[i] = new ChatUserInfo(ref list.userList[i]);
            }
        }

        internal unsafe ChatUserList(TTV_ChatUserList* list)
        {
            if (list == null)
            {
                users = new ChatUserInfo[0];
            }
            else
            {
                users = new ChatUserInfo[list->userCount];
                for (int i = 0; i < users.Length; ++i)
                {
                    users[i] = new ChatUserInfo(ref list->userList[i]);
                }
            }
        }

        public ChatUserInfo[] List
        {
            get { return users; }
        }
    }

    #endregion

    #region ChatChannelInfo

    /**
     * TTV_ChatChannelInfo - Information about a chat channel.
     */
    public unsafe struct TTV_ChatChannelInfo
    {
        public fixed byte name[64];
        public fixed byte broadcasterLanguage[10];
        public TTV_ChatRestrictionReason localUserRestriction;
        public UInt32 slowModeTimeout;
        public byte subscribersOnly;
        public byte slowMode;
    }

    public class ChatChannelInfo : StructWrapperBase<TTV_ChatChannelInfo>
    {
        public unsafe ChatChannelInfo()
        {
            fixed (byte* name = mStruct.name)
            {
                name[0] = 0;
            }

            fixed (byte* broadcasterLanguage = mStruct.broadcasterLanguage)
            {
                broadcasterLanguage[0] = 0;
            }

            mStruct.localUserRestriction = TTV_ChatRestrictionReason.TTV_CHAT_RESTRICTION_REASON_NONE;
            mStruct.slowModeTimeout = 0;
            mStruct.subscribersOnly = 0;
            mStruct.slowMode = 0;
        }

        internal ChatChannelInfo(ref TTV_ChatChannelInfo info)
        {
            mStruct = info;
        }

        internal unsafe ChatChannelInfo(TTV_ChatChannelInfo* p)
        {
            if (p != null)
            {
                mStruct = *p;
            }
        }

        public ChatChannelInfo Copy()
        {
            return new ChatChannelInfo(ref mStruct);
        }

        public unsafe string Name
        {
            get
            {
                fixed (byte* b = mStruct.name)
                {
                    return MarshalUtil.StringFromBytes(b, 64);
                }
            }
        }

        public unsafe string BroadcasterLanguage
        {
            get
            {
                fixed (byte* b = mStruct.broadcasterLanguage)
                {
                    return MarshalUtil.StringFromBytes(b, 10);
                }
            }
        }

        public TTV_ChatRestrictionReason LocalUserRestriction
        {
            get
            {
                return mStruct.localUserRestriction;
            }
        }

        public UInt32 SlowModeTimeout
        {
            get
            {
                return mStruct.slowModeTimeout;
            }
        }

        public bool SlowMode
        {
            get
            {
                return mStruct.slowMode != 0;
            }
        }

        public bool SubscribersOnly
        {
            get
            {
                return mStruct.subscribersOnly != 0;
            }
        }
    }

    #endregion

    #region ChatBadgeImage

    /**
     * TTV_ChatBadgeImage - Information for rendering a badge.
     */
    public unsafe struct TTV_ChatBadgeImage
    {
        public fixed byte rasterImageUrl[139 + 1];
        public fixed byte svgImageUrl[139 + 1];
    }

    /**
     * ChatBadgeImage - Information for rendering a badge.
     */
    public class ChatBadgeImage
    {
        protected string rasterImageUrl;
        protected string svgImageUrl;

        internal unsafe ChatBadgeImage(ref TTV_ChatBadgeImage badge)
        {
            fixed (TTV_ChatBadgeImage* p = &badge)
            {
                rasterImageUrl = MarshalUtil.StringFromBytes(p->rasterImageUrl, 140);
                svgImageUrl = MarshalUtil.StringFromBytes(p->svgImageUrl, 140);
            }
        }

        public string RasterImageUrl
        {
            get { return rasterImageUrl; }
        }

        public string SvgImageUrl
        {
            get { return svgImageUrl; }
        }
    }

    #endregion

    /**
     * TTV_ChatMessageTokenType - The types of tokens that can be generated from a chat message.
     */
    public enum TTV_ChatMessageTokenType
    {
        TTV_CHAT_MSGTOKEN_TEXT,
        TTV_CHAT_MSGTOKEN_EMOTICON,
        TTV_CHAT_MSGTOKEN_MENTION,
        TTV_CHAT_MSGTOKEN_URL
    }

    /**
     * TTV_ChatTextMessageToken - Information about a text token.
     */
    public unsafe struct TTV_ChatTextToken
    {
        public byte* buffer;
    }

    /**
     * TTV_ChatEmoticonMessageToken - Information about an image token that is specified by a URL.
     */
    public unsafe struct TTV_ChatEmoticonToken
    {
        public byte* emoticonText;
        public UInt32 emoticonId;
    };

    /**
     * TTV_ChatMentionToken - Information about a user mention.
     */
    public unsafe struct TTV_ChatMentionToken
    {
        public byte* userName;
    }

    /**
     * TTV_ChatUrlToken - Information about a hyperlink.
     */
    public unsafe struct TTV_ChatUrlToken
    {
        public byte* url;
    }

    /**
     * TTV_ChatMessageToken - Information about an image token.
     */
    [StructLayout(LayoutKind.Explicit)]
    public unsafe struct TTV_ChatMessageToken
    {
        [System.Runtime.InteropServices.FieldOffset(0)]
        public TTV_ChatMessageTokenType type;	//!< The way to interpret the data.

        [System.Runtime.InteropServices.FieldOffset(4)]
        public TTV_ChatTextToken text;

        [System.Runtime.InteropServices.FieldOffset(4)]
        public TTV_ChatEmoticonToken emoticon;

        [System.Runtime.InteropServices.FieldOffset(4)]
        public TTV_ChatMentionToken mention;

        [System.Runtime.InteropServices.FieldOffset(4)]
        public TTV_ChatUrlToken url;
    }

    public class ChatMessageToken
    {
        protected TTV_ChatMessageTokenType type;	//!< The way to interpret the data.

        public TTV_ChatMessageTokenType Type
        {
            get { return type; }
        }
    }

    public class ChatTextToken : ChatMessageToken
    {
        protected string message;

        internal unsafe ChatTextToken(ref TTV_ChatTextToken token)
        {
            base.type = TTV_ChatMessageTokenType.TTV_CHAT_MSGTOKEN_TEXT;
            message = MarshalUtil.StringFromBytes(token.buffer, Constants.kMaxStringLength);
        }

        internal unsafe ChatTextToken(TTV_ChatTextToken* token)
        {
            base.type = TTV_ChatMessageTokenType.TTV_CHAT_MSGTOKEN_TEXT;
            message = MarshalUtil.StringFromBytes(token->buffer, Constants.kMaxStringLength);
        }

        public string Message
        {
            get { return message; }
        }
    }

    public class ChatEmoticonToken : ChatMessageToken
    {
        protected string emoticonText;
        protected UInt32 emoticonId;

        internal unsafe ChatEmoticonToken(ref TTV_ChatEmoticonToken token)
        {
            base.type = TTV_ChatMessageTokenType.TTV_CHAT_MSGTOKEN_EMOTICON;
            emoticonText = MarshalUtil.StringFromBytes(token.emoticonText, Constants.kMaxStringLength);
            emoticonId = token.emoticonId;
        }


        internal unsafe ChatEmoticonToken(TTV_ChatEmoticonToken* token)
        {
            base.type = TTV_ChatMessageTokenType.TTV_CHAT_MSGTOKEN_EMOTICON;
            emoticonText = MarshalUtil.StringFromBytes(token->emoticonText, Constants.kMaxStringLength);
            emoticonId = token->emoticonId;
        }

        public string GetUrl(float scale)
        {
            return String.Format("http://static-cdn.jtvnw.net/emoticons/v1/{0}/{1}", emoticonId, scale);
        }

        public string EmoticonText
        {
            get { return emoticonText; }
        }

        public UInt32 EmoticonId
        {
            get { return emoticonId; }
        }
    }

    public class ChatMentionToken : ChatMessageToken
    {
        protected string userName;

        internal unsafe ChatMentionToken(ref TTV_ChatMentionToken token)
        {
            base.type = TTV_ChatMessageTokenType.TTV_CHAT_MSGTOKEN_MENTION;
            userName = MarshalUtil.StringFromBytes(token.userName, Constants.kMaxStringLength);
        }

        internal unsafe ChatMentionToken(TTV_ChatMentionToken* token)
        {
            base.type = TTV_ChatMessageTokenType.TTV_CHAT_MSGTOKEN_MENTION;
            userName = MarshalUtil.StringFromBytes(token->userName, Constants.kMaxStringLength);
        }

        public string UserName
        {
            get { return userName; }
        }
    }

    public class ChatUrlToken : ChatMessageToken
    {
        protected string url;

        internal unsafe ChatUrlToken(ref TTV_ChatUrlToken token)
        {
            base.type = TTV_ChatMessageTokenType.TTV_CHAT_MSGTOKEN_URL;
            url = MarshalUtil.StringFromBytes(token.url, Constants.kMaxStringLength);
        }

        public string Url
        {
            get { return url; }
        }
    }

    #region ChatMessage

    /**
     * TTV_ChatMessage - A list of tokens parsed from a text message.
     */
    public unsafe struct TTV_ChatMessage
    {
        public byte* userName;		                        //!< The UTF-8 encoded displayed name.  Currently restricted to the ASCII subset.
        public UInt32 userId;
        public byte* threadId;		                        //!< The UTF-8 encoded displayed name.  Currently restricted to the ASCII subset.
        public TTV_ChatUserMode modes;					    //!< The modes of the user who sent the message.
        public TTV_ChatUserSubscription subscriptions;	    //!< The subscriptions of the user who sent the message.
        public uint nameColorARGB;							//!< The current ARGB color of the user's name text.
        public TTV_ChatMessageToken* tokenList;			    //!< The array of message tokens.
        public uint tokenCount;							    //!< The number of entries in tokenList.
        public TTV_ChatMessageFlag flags;			        //!< Any additional details about the message.
        public UInt32 timestamp;
        public UInt32 messageId;
    }

    public class ChatMessage
    {
        protected string userName;
        protected UInt32 userId;
        protected TTV_ChatUserMode modes;								//!< The modes of the user who sent the message.
        protected string threadId;
        protected TTV_ChatUserSubscription subscriptions;				//!< The subscriptions of the user who sent the message.
        protected uint nameColorARGB;							//!< The current ARGB color of the user's name text.
        protected ChatMessageToken[] tokenList;					//!< The array of message tokens.
        protected TTV_ChatMessageFlag flags;                    //!< Any additional details about the message.
        protected UInt32 timestamp;
        protected UInt32 messageId;

        internal static ChatMessage[] GetArray(ref TTV_ChatMessageList messageList)
        {
            ChatMessage[] list = new ChatMessage[messageList.messageCount];
            for (int i = 0; i < list.Length; ++i)
            {
                unsafe
                {
                    list[i] = new ChatMessage(ref messageList.messageList[i]);
                }
            }

            return list;
        }

        internal unsafe static ChatMessage[] GetArray(TTV_ChatMessageList* messageList)
        {
            if (messageList == null)
            {
                return new ChatMessage[0];
            }
            else
            {
                ChatMessage[] list = new ChatMessage[messageList->messageCount];
                for (int i = 0; i < list.Length; ++i)
                {
                    unsafe
                    {
                        list[i] = new ChatMessage(ref messageList->messageList[i]);
                    }
                }

                return list;
            }
        }

        internal unsafe ChatMessage(ref TTV_ChatMessage data)
        {
            fixed (TTV_ChatMessage* d = &data)
            {
                Bind(d);
            }
        }

        internal unsafe ChatMessage(TTV_ChatMessage* data)
        {
            Bind(data);
        }

        internal unsafe ChatMessage(IntPtr p)
        {
            TTV_ChatMessage* data = (TTV_ChatMessage*)p;
            Bind(data);
        }

        internal unsafe void Bind(TTV_ChatMessage* data)
        {
            userName = MarshalUtil.StringFromBytes(data->userName, Constants.kMaxStringLength);
            threadId = MarshalUtil.StringFromBytes(data->threadId, Constants.kMaxStringLength);

            userId = data->userId;
            modes = data->modes;
            subscriptions = data->subscriptions;
            nameColorARGB = data->nameColorARGB;
            flags = data->flags;
            tokenList = new ChatMessageToken[data->tokenCount];
            timestamp = data->timestamp;
            messageId = data->messageId;

            for (int i = 0; i < tokenList.Length; ++i)
            {
                switch (data->tokenList[i].type)
                {
                    case TTV_ChatMessageTokenType.TTV_CHAT_MSGTOKEN_TEXT:
                    {
                        tokenList[i] = new ChatTextToken(ref data->tokenList[i].text);
                        break;
                    }
                    case TTV_ChatMessageTokenType.TTV_CHAT_MSGTOKEN_EMOTICON:
                    {
                        tokenList[i] = new ChatEmoticonToken(ref data->tokenList[i].emoticon);
                        break;
                    }
                    case TTV_ChatMessageTokenType.TTV_CHAT_MSGTOKEN_MENTION:
                    {
                        tokenList[i] = new ChatMentionToken(ref data->tokenList[i].mention);
                        break;
                    }
                    case TTV_ChatMessageTokenType.TTV_CHAT_MSGTOKEN_URL:
                    {
                        tokenList[i] = new ChatUrlToken(ref data->tokenList[i].url);
                        break;
                    }
                    default:
                    {
                        tokenList[i] = null;
                        break;
                    }
                }
            }
        }

        public string UserName
        {
            get { return userName; }
        }

        public string ThreadId
        {
            get { return threadId; }
        }

        public TTV_ChatUserMode Modes
        {
            get { return modes; }
        }

        public TTV_ChatUserSubscription Subscriptions
        {
            get { return subscriptions; }
        }

        public uint NameColorARGB
        {
            get { return nameColorARGB; }
        }

        public ChatMessageToken[] Tokens
        {
            get { return tokenList; }
        }

        public UInt32 UserId 
        {
            get 
            { 
                return userId; 
            } 
        }

        public UInt32 MessageId
        {
            get
            {
                return messageId;
            }
        }

        public UInt32 Timestamp
        {
            get
            {
                return timestamp;
            }
        }

        public bool IsAction
        {
            get { return (flags & TTV_ChatMessageFlag.TTV_CHAT_MESSAGE_FLAG_ACTION) != 0; }
        }
    }

    #endregion

    #region ChatMessageList

    /**
     * TTV_ChatMessageList - A list of chat messages.
     */
    public unsafe struct TTV_ChatMessageList
    {
        public TTV_ChatMessage* messageList;            //!< The ordered array of chat messages.
        public UInt32 messageCount;						//!< The number of messages in the list.
    }

    #endregion

    #region ChatUserEmoticonSets

    public unsafe struct TTV_ChatUserEmoticonSets
    {
        internal fixed byte userName[64];
        internal UInt32 emoticonSetCount;
        internal UInt32* emoticonSetIds;
    }


    public class ChatUserEmoticonSets
    {
        protected string userName;
        protected uint[] emoticonSetIds;

        internal unsafe ChatUserEmoticonSets(ref TTV_ChatUserEmoticonSets data)
        {
            fixed (byte* b = data.userName)
            {
                userName = MarshalUtil.StringFromBytes(b, 64);
            }

            emoticonSetIds = new uint[data.emoticonSetCount];

            for (int i = 0; i < data.emoticonSetCount; ++i)
            {
                emoticonSetIds[i] = data.emoticonSetIds[i];
            }
        }
        
        internal unsafe ChatUserEmoticonSets(IntPtr p)
        {
            TTV_ChatUserEmoticonSets* data = (TTV_ChatUserEmoticonSets*)p;

            userName = MarshalUtil.StringFromBytes(data->userName, 64);
            emoticonSetIds = new uint[data->emoticonSetCount];

            for (int i = 0; i < data->emoticonSetCount; ++i)
            {
                emoticonSetIds[i] = data->emoticonSetIds[i];
            }
        }

        public string Username
        {
            get { return userName; }
        }

        public uint[] EmoticonSetIds
        {
            get { return emoticonSetIds; }
        }
    }

    #endregion

    #region ChatEmoticonData

    public unsafe struct TTV_ChatEmoticonData
    {
        internal UInt32 emoticonId;
        internal fixed byte match[64];
        internal byte isRegex;
    }

    public class ChatEmoticonData
    {
        protected uint emoticonId;
        protected string match;
        protected bool isRegex;

        internal unsafe ChatEmoticonData(ref TTV_ChatEmoticonData data)
        {
            emoticonId = data.emoticonId;

            fixed (byte* b = data.match)
            {
                match = MarshalUtil.StringFromBytes(b, 64);
            }

            isRegex = data.isRegex != 0;
        }

        public uint EmoticonId
        {
            get { return emoticonId; }
        }

        public string Match
        {
            get { return match; }
        }

        public bool IsRegex
        {
            get { return isRegex; }
        }
    }

    #endregion

    #region ChatEmoticonSetData

    public unsafe struct TTV_ChatEmoticonSetData
    {
        internal UInt32 emoticonSetId;
        internal UInt32 emoticonCount;
        internal TTV_ChatEmoticonData* emoticons;
    }

    public class ChatEmoticonSetData
    {
        protected uint emoticonSetId;
        protected ChatEmoticonData[] emoticons;

        internal unsafe ChatEmoticonSetData(ref TTV_ChatEmoticonSetData data)
        {
            emoticonSetId = data.emoticonSetId;
            emoticons = new ChatEmoticonData[data.emoticonCount];

            for (int i = 0; i < data.emoticonCount; ++i)
            {
                emoticons[i] = new ChatEmoticonData(ref data.emoticons[i]);
            }
        }

        internal unsafe ChatEmoticonSetData(IntPtr p)
        {
            TTV_ChatEmoticonSetData* data = (TTV_ChatEmoticonSetData*)p;

            emoticonSetId = data->emoticonSetId;
            emoticons = new ChatEmoticonData[data->emoticonCount];

            for (int i = 0; i < data->emoticonCount; ++i)
            {
                emoticons[i] = new ChatEmoticonData(ref data->emoticons[i]);
            }
        }

        public uint EmoticonSetId
        {
            get { return emoticonSetId; }
        }

        public ChatEmoticonData[] Emoticons
        {
            get { return emoticons; }
        }
    }

    #endregion

    #region ChatBadgeData

    /**
     * TTV_ChatBadgeData - Information about how to render badges based on the subscriptions and user modes.
     */
    public unsafe struct TTV_ChatBadgeData
    {
        internal fixed byte channel[64];
        internal TTV_ChatBadgeImage turboIcon;
        internal TTV_ChatBadgeImage channelSubscriberIcon;
        internal TTV_ChatBadgeImage broadcasterIcon;
        internal TTV_ChatBadgeImage staffIcon;
        internal TTV_ChatBadgeImage adminIcon;
        internal TTV_ChatBadgeImage moderatorIcon;
        internal TTV_ChatBadgeImage globalModeratorIcon;
    }

    public class ChatBadgeData
    {
        protected string channel;
        protected ChatBadgeImage turboIcon;
        protected ChatBadgeImage channelSubscriberIcon;
        protected ChatBadgeImage broadcasterIcon;
        protected ChatBadgeImage staffIcon;
        protected ChatBadgeImage adminIcon;
        protected ChatBadgeImage moderatorIcon;
        protected ChatBadgeImage globalModeratorIcon;

        internal unsafe ChatBadgeData(IntPtr p)
        {
            TTV_ChatBadgeData* data = (TTV_ChatBadgeData*)p;

            channel = MarshalUtil.StringFromBytes(data->channel, 64);

            turboIcon = new ChatBadgeImage(ref data->turboIcon);
            channelSubscriberIcon = new ChatBadgeImage(ref data->channelSubscriberIcon);
            broadcasterIcon = new ChatBadgeImage(ref data->broadcasterIcon);
            staffIcon = new ChatBadgeImage(ref data->staffIcon);
            adminIcon = new ChatBadgeImage(ref data->adminIcon);
            moderatorIcon = new ChatBadgeImage(ref data->moderatorIcon);
            globalModeratorIcon = new ChatBadgeImage(ref data->globalModeratorIcon);
        }

        public string Channel
        {
            get { return channel; }
        }

        public ChatBadgeImage Turbo
        {
            get { return turboIcon; }
        }

        public ChatBadgeImage ChannelSubscriber
        {
            get { return channelSubscriberIcon; }
        }

        public ChatBadgeImage Broadcaster
        {
            get { return broadcasterIcon; }
        }

        public ChatBadgeImage Staff
        {
            get { return staffIcon; }
        }

        public ChatBadgeImage Admin
        {
            get { return adminIcon; }
        }

        public ChatBadgeImage Moderator
        {
            get { return moderatorIcon; }
        }

        public ChatBadgeImage GlobalModerator
        {
            get { return globalModeratorIcon; }
        }
    }

    #endregion

    #region ChatThread

    public unsafe struct TTV_ChatThread
    {
        public byte* threadId;
	    public TTV_ChatUserList participants;
	    public TTV_ChatMessage* lastMessage;
	    public UInt32 lastUpdatedTimestamp;
	    public UInt32 lastReadMessageId;
	    public byte archived;
        public byte muted;
    }

    public class ChatThread
    {
        private string threadId;
        private ChatUserList participants;
        private ChatMessage lastMessage;
        private DateTime lastUpdatedTimestamp;
        private uint lastReadMessageId;
        private bool archived;
        private bool muted;

        internal unsafe ChatThread(TTV_ChatThread* data)
        {
            threadId = MarshalUtil.StringFromBytes(data->threadId, Constants.kMaxStringLength);
            participants = new ChatUserList(ref data->participants);
            lastMessage = data->lastMessage != null ? new ChatMessage(data->lastMessage) : null;
            lastUpdatedTimestamp = new DateTime(data->lastUpdatedTimestamp);
            lastReadMessageId = data->lastReadMessageId;
            archived = data->archived != 0;
            muted = data->muted != 0;
        }

        public string ThreadId
        {
            get { return threadId; }
        }

        public ChatUserList Participants
        {
            get { return participants; }
        }

        public ChatMessage LastMessage
        {
            get { return lastMessage; }
        }

        public DateTime LastUpdatedTimestamp
        {
            get { return lastUpdatedTimestamp; }
        }

        public uint LastReadMessageId
        {
            get { return lastReadMessageId; }
        }

        public bool IsMuted
        {
            get { return muted; }
        }

        public bool IsArchived
        {
            get { return archived; }
        }
    }

    #endregion

    #region ChatThreadList

    public unsafe struct TTV_ChatThreadList
    {
        public TTV_ChatThread* threadList;
	    public UInt32 threadCount;
    }

    public class ChatThreadList
    {
        private ChatThread[] threads;

        internal unsafe ChatThreadList(TTV_ChatThreadList* data)
        {
            if (data != null)
            {
                threads = new ChatThread[data->threadCount];

                for (int i = 0; i < data->threadCount; ++i)
                {
                    threads[i] = new ChatThread(&data->threadList[i]);
                }
            }
            else
            {
                threads = new ChatThread[0];
            }
        }

        public ChatThread[] Threads
        {
            get { return threads; }
        }
    }

    #endregion

    #region Helper structs

    public unsafe struct TTV_ChatKeyValuePair
    {
        public fixed byte key[64];
        public fixed byte value[192];
    }

    public unsafe struct TTV_ChatKeyValuePairList
    {
        public TTV_ChatKeyValuePair* list;
        public UInt32 count;
    }

    #endregion
}
