﻿using System;
using System.Collections.Generic;
using System.Runtime.InteropServices;
using ErrorCode = Twitch.ErrorCode;

// TODO: This needs to be update to match the current API

namespace Twitch.Chat
{
    public class StandardChatApi : ChatApi
    {
        #region Callback Signatures

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        internal delegate void ChatInitializationCallback(ErrorCode result);
        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        internal delegate void ChatShutdownCallback(ErrorCode result);
        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        internal delegate void ChatUserEmoticonSetsChangedCallback(ref TTV_ChatUserEmoticonSets emoticonSets);
        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        internal delegate void ChatEmoticonSetDataCallback(ref TTV_ChatEmoticonSetData emoticonSet);
        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        internal delegate void ChatUserBlockChangeCallback(string userName, string blockUsername, bool block, ErrorCode ec);

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        internal delegate void ChatChannelStateChangedCallback(string channelName, TTV_ChatChannelState state, ErrorCode ec);
        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        internal delegate void ChatChannelInfoChangedCallback(string channelName, ref TTV_ChatChannelInfo info);
        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        internal delegate void ChatChannelLocalUserChangedCallback(string channelName, ref TTV_ChatUserInfo userInfo);
        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        internal delegate void ChatChannelUserChangeCallback(string channelName, ref TTV_ChatUserList joinList, ref TTV_ChatUserList leaveList, ref TTV_ChatUserList infoChangeList);
        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        internal delegate void ChatChannelMessageCallback(string channelName, ref TTV_ChatMessageList messageList);
        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        internal delegate void ChatChannelClearCallback(string channelName, string username);
        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        internal delegate void ChatChannelBadgeDataDownloadCallback(string channelName, ErrorCode result);
        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        internal delegate void ChatChannelHostTargetChangedCallback(string channelName, string targetChannel, int numViewers);
        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        internal delegate void ChatChannelNoticeCallback(string channelName, string id, ref TTV_ChatKeyValuePairList parameters);
        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        internal delegate void ChatChannelSetBroadcasterLanguageCallback(string channelName, ErrorCode ec);

        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        internal delegate void ChatWhisperStateChangedCallback(TTV_ChatChannelState state, ErrorCode ec);
        [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
        internal delegate void ChatWhisperMessageCallback(ref TTV_ChatMessageList messageList);

        internal struct ManagedChatAPIListener
        {
            public ChatInitializationCallback initializationCallback;
            public ChatShutdownCallback shutdownCallback;
            public ChatUserEmoticonSetsChangedCallback userEmoticonSetsChangedCallback;
            public ChatEmoticonSetDataCallback emoticonSetDataCallback;
            public ChatUserBlockChangeCallback chatUserBlockChangeCallback;
        }

        internal struct ManagedChatChannelListener
        {
            public ChatChannelStateChangedCallback channelStateChangedCallback;
            public ChatChannelInfoChangedCallback channelInfoChangedCallback;
            public ChatChannelLocalUserChangedCallback localUserInfoChangedCallback;
            public ChatChannelUserChangeCallback userCallback;
            public ChatChannelMessageCallback messageCallback;
            public ChatChannelClearCallback clearCallback;
            public ChatChannelBadgeDataDownloadCallback badgeDownloadCallback;
            public ChatChannelHostTargetChangedCallback hostTargetChangedCallback;
            public ChatChannelNoticeCallback noticeCallback;
            public ChatChannelSetBroadcasterLanguageCallback setBroadcasterLanguageCallback;
        }

        internal struct ManagedChatWhisperListener
        {
            public ChatWhisperStateChangedCallback whisperStateChangedCallback;
            public ChatWhisperMessageCallback whisperMessageCallback;
        }


        private class ApiListenerAdapter
        {
            private StandardChatApi m_Api;
            private ManagedChatAPIListener m_ManagedListener;
            private IChatApiListener m_ClientListener;

            public ManagedChatAPIListener ManagedListener
            {
                get { return m_ManagedListener; }
            }

            public IChatApiListener ClientListener
            {
                get { return m_ClientListener; }
                set { m_ClientListener = value; }
            }

            public ApiListenerAdapter(StandardChatApi api)
            {
                m_Api = api;

                m_ManagedListener.initializationCallback = ChatInitializationCallback;
                m_ManagedListener.shutdownCallback = ChatShutdownCallback;
                m_ManagedListener.userEmoticonSetsChangedCallback = ChatUserEmoticonSetsChangedCallback;
                m_ManagedListener.emoticonSetDataCallback = ChatEmoticonSetDataCallback;
                m_ManagedListener.chatUserBlockChangeCallback = ChatUserBlockChangeCallback;
            }

            private void ChatInitializationCallback(ErrorCode result)
            {
                m_ClientListener.ChatInitializationCallback(result);

                if (Error.Failed(result))
                {
                    m_Api.m_ApiListenerAdapter.ClientListener = null;
                }
            }

            private void ChatShutdownCallback(ErrorCode result)
            {
                m_ClientListener.ChatShutdownCallback(result);

                if (Error.Succeeded(result))
                {
                    m_Api.m_ApiListenerAdapter.ClientListener = null;
                }
            }

            private void ChatUserEmoticonSetsChangedCallback(ref TTV_ChatUserEmoticonSets emoticonSets)
            {
                m_ClientListener.ChatUserEmoticonSetsChangedCallback(new ChatUserEmoticonSets(ref emoticonSets));
            }

            private void ChatEmoticonSetDataCallback(ref TTV_ChatEmoticonSetData emoticonSet)
            {
                m_ClientListener.ChatEmoticonSetDataCallback(new ChatEmoticonSetData(ref emoticonSet));
            }

            private void ChatUserBlockChangeCallback(string userName, string blockUsername, bool block, ErrorCode ec)
            {
                m_ClientListener.ChatUserBlockChangeCallback(userName, blockUsername, block, ec);
            }
        }

        private class ChannelListenerAdapter
        {
            private StandardChatApi m_Api;
            private ManagedChatChannelListener m_ManagedListener;
            private IChatChannelListener m_ClientListener;

            public ManagedChatChannelListener ManagedListener
            {
                get { return m_ManagedListener; }
            }

            public IChatChannelListener ClientListener
            {
                get { return m_ClientListener; }
                set { m_ClientListener = value; }
            }

            public ChannelListenerAdapter(StandardChatApi api)
            {
                m_Api = api;

                m_ManagedListener.channelStateChangedCallback = ChatChannelStateChangedCallback;
                m_ManagedListener.channelInfoChangedCallback = ChatChannelInfoChangedCallback;
                m_ManagedListener.localUserInfoChangedCallback = ChatChannelLocalUserChangedCallback;
                m_ManagedListener.userCallback = ChatChannelUserChangeCallback;
                m_ManagedListener.messageCallback = ChatChannelMessageCallback;
                m_ManagedListener.clearCallback = ChatChannelClearCallback;
                m_ManagedListener.badgeDownloadCallback = ChatChannelBadgeDataDownloadCallback;
                m_ManagedListener.hostTargetChangedCallback = ChatChannelHostTargetChangedCallback;
                m_ManagedListener.noticeCallback = ChatChannelNoticeCallback;
                m_ManagedListener.setBroadcasterLanguageCallback = ChatChannelSetBroadcasterLanguageCallback;
            }

            private void ChatChannelStateChangedCallback(string channelName, TTV_ChatChannelState state, ErrorCode ec)
            {
                m_ClientListener.ChatChannelStateChangedCallback(channelName, state, ec);

                switch (state)
                {
                    case TTV_ChatChannelState.TTV_CHAT_CHANNEL_STATE_DISCONNECTED:
                        m_ClientListener = null;
                        break;
                    default:
                        break;
                }
            }

            private void ChatChannelInfoChangedCallback(string channelName, ref TTV_ChatChannelInfo info)
            {
                m_ClientListener.ChatChannelInfoChangedCallback(channelName, new ChatChannelInfo(ref info));
            }

            private void ChatChannelLocalUserChangedCallback(string channelName, ref TTV_ChatUserInfo userInfo)
            {
                m_ClientListener.ChatChannelLocalUserChangedCallback(channelName, new ChatUserInfo(ref userInfo));
            }

            private void ChatChannelUserChangeCallback(string channelName, ref TTV_ChatUserList joinList, ref TTV_ChatUserList leaveList, ref TTV_ChatUserList infoChangeList)
            {
                m_ClientListener.ChatChannelUserChangeCallback(channelName, new ChatUserList(ref joinList).List, new ChatUserList(ref leaveList).List, new ChatUserList(ref infoChangeList).List);

                m_Api.FreeUserList(ref joinList);
                m_Api.FreeUserList(ref leaveList);
                m_Api.FreeUserList(ref infoChangeList);
            }

            private void ChatChannelMessageCallback(string channelName, ref TTV_ChatMessageList messageList)
            {
                m_ClientListener.ChatChannelMessageCallback(channelName, ChatMessage.GetArray(ref messageList));

                m_Api.FreeMessageList(ref messageList);
            }

            private void ChatChannelClearCallback(string channelName, string username)
            {
                m_ClientListener.ChatChannelClearCallback(channelName, username);
            }

            private void ChatChannelBadgeDataDownloadCallback(string channelName, ErrorCode result)
            {
                m_ClientListener.ChatChannelBadgeDataDownloadCallback(channelName, result);
            }

            private void ChatChannelHostTargetChangedCallback(string channelName, string targetChannel, int numViewers)
            {
                m_ClientListener.ChatChannelHostTargetChangedCallback(channelName, targetChannel, numViewers);
            }

            private unsafe void ChatChannelNoticeCallback(string channelName, string id, ref TTV_ChatKeyValuePairList parameters)
            {
                Dictionary<string, string> kvpList = new Dictionary<string,string>();

                for (int i = 0; i < parameters.count; ++i)
                {
                    string key = MarshalUtil.StringFromBytes(parameters.list[i].key, 64);
                    string value = MarshalUtil.StringFromBytes(parameters.list[i].value, 192);
                    kvpList[key] = value;
                }

                m_ClientListener.ChatChannelNoticeCallback(channelName, id, kvpList);
            }

            private void ChatChannelSetBroadcasterLanguageCallback(string channelName, ErrorCode ec)
            {
                m_ClientListener.ChatChannelSetBroadcasterLanguageCallback(channelName, ec);
            }
        }

        private ApiListenerAdapter m_ApiListenerAdapter = null;
        private Dictionary<string, ChannelListenerAdapter> m_ChannelListenerAdapters = new Dictionary<string, ChannelListenerAdapter>();

        #endregion
        
        #region Native Functions

        [DllImport("twitchsdk", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Chat_Initialize")]
        internal static extern ErrorCode TTV_CSharp_Chat_Initialize(TTV_ChatTokenizationOption tokenizationOptions, ref ManagedChatAPIListener listener);
        [DllImport("twitchsdk", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Chat_Shutdown")]
        internal static extern ErrorCode TTV_CSharp_Chat_Shutdown();

        [DllImport("twitchsdk", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Chat_LogIn")]
        internal static extern ErrorCode TTV_CSharp_Chat_LogIn(string userName, string oauthToken);
        [DllImport("twitchsdk", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Chat_LogOut")]
        internal static extern ErrorCode TTV_CSharp_Chat_LogOut(string userName);

        [DllImport("twitchsdk", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Chat_Connect")]
        internal static extern ErrorCode TTV_CSharp_Chat_Connect(string username, string channelName, ref ManagedChatChannelListener listener);
        [DllImport("twitchsdk", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Chat_Disconnect")]
        internal static extern ErrorCode TTV_CSharp_Chat_Disconnect(string username, string channelName);

        [DllImport("twitchsdk", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Chat_BlockUser")]
        internal static extern ErrorCode TTV_CSharp_Chat_BlockUser(string userName, string blockUserName);
        [DllImport("twitchsdk", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Chat_UnblockUser")]
        internal static extern ErrorCode TTV_CSharp_Chat_UnblockUser(string userName, string blockUserName);
        [DllImport("twitchsdk", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Chat_GetUserBlocked")]
        internal static extern ErrorCode TTV_CSharp_Chat_GetUserBlocked(string userName, string blockUserName, ref bool blocked);
        
        [DllImport("twitchsdk", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Chat_SendMessage")]
        internal static extern ErrorCode TTV_CSharp_Chat_SendMessage(string username, string channelName, string message);
        [DllImport("twitchsdk", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Chat_FlushEvents")]
        internal static extern ErrorCode TTV_CSharp_Chat_FlushEvents();
        [DllImport("twitchsdk", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Chat_ForceUserListUpdate")]
        internal static extern ErrorCode TTV_CSharp_Chat_ForceUserListUpdate(string username, string channelName);

        [DllImport("twitchsdk", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Chat_GetBadgeData")]
        internal static extern ErrorCode TTV_CSharp_Chat_GetBadgeData(string channelName, out IntPtr /*TTV_ChatBadgeData*/ data);
        [DllImport("twitchsdk", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Chat_FreeBadgeData")]
        internal static extern ErrorCode TTV_CSharp_Chat_FreeBadgeData(IntPtr data);

        [DllImport("twitchsdk", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Chat_FreeUserList")]
        internal static extern ErrorCode TTV_CSharp_Chat_FreeUserList(ref TTV_ChatUserList list);
        [DllImport("twitchsdk", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Chat_FreeMessageList")]
        internal static extern ErrorCode TTV_CSharp_Chat_FreeMessageList(ref TTV_ChatMessageList list);

        [DllImport("twitchsdk", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Chat_GetMessageFlushInterval")]
        internal static extern UInt64 TTV_CSharp_Chat_GetMessageFlushInterval();
        [DllImport("twitchsdk", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Chat_SetMessageFlushInterval")]
        internal static extern ErrorCode TTV_CSharp_Chat_SetMessageFlushInterval(UInt64 milliseconds);
        [DllImport("twitchsdk", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Chat_GetUserListUpdateInterval")]
        internal static extern UInt64 TTV_CSharp_Chat_GetUserListUpdateInterval();
        [DllImport("twitchsdk", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Chat_SetUserListUpdateInterval")]
        internal static extern ErrorCode TTV_CSharp_Chat_SetUserListUpdateInterval(UInt64 milliseconds);

        [DllImport("twitchsdk", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Chat_SetBroadcasterLanguageChatEnabled")]
        internal static extern ErrorCode TTV_CSharp_Chat_SetBroadcasterLanguageChatEnabled(string username, string channelName, bool enabled);
        [DllImport("twitchsdk", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Chat_OptInToBroadcasterLanguageChat")]
        internal static extern ErrorCode TTV_CSharp_Chat_OptInToBroadcasterLanguageChat(string username, string channelName, string language);
        [DllImport("twitchsdk", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Chat_SetLocalLanguage")]
        internal static extern ErrorCode TTV_CSharp_Chat_SetLocalLanguage(string language);

        #endregion

        public StandardChatApi()
        {
            m_ApiListenerAdapter = new ApiListenerAdapter(this);
        }

        public override ErrorCode Initialize(TTV_ChatTokenizationOption tokenizationOptions, IChatApiListener listener)
        {
            if (m_ApiListenerAdapter.ClientListener != null)
            {
                return ErrorCode.TTV_EC_ALREADY_INITIALIZED;
            }

            m_ApiListenerAdapter.ClientListener = listener;

            ManagedChatAPIListener nativeListener = m_ApiListenerAdapter.ManagedListener;
            return TTV_CSharp_Chat_Initialize(tokenizationOptions, ref nativeListener);
        }

        public override ErrorCode Shutdown()
        {
            return TTV_CSharp_Chat_Shutdown();
        }

        public override ErrorCode LogIn(string userName, string oauthToken)
        {
            return TTV_CSharp_Chat_LogIn(userName, oauthToken);
        }

        public override ErrorCode LogOut(string userName)
        {
            return TTV_CSharp_Chat_LogOut(userName);
        }

        public override ErrorCode Connect(string userName, string channelName, IChatChannelListener listener)
        {
            ChannelListenerAdapter adapter = null;
            if (m_ChannelListenerAdapters.ContainsKey(channelName))
            {
                adapter = m_ChannelListenerAdapters[channelName];
            }
            else
            {
                adapter = new ChannelListenerAdapter(this);
            }

            adapter.ClientListener = listener;
            m_ChannelListenerAdapters[channelName] = adapter;

            ManagedChatChannelListener nativeListener = adapter.ManagedListener;
            ErrorCode ec = TTV_CSharp_Chat_Connect(channelName, userName, ref nativeListener);
            if (Error.Failed(ec))
            {
                m_ChannelListenerAdapters.Remove(channelName);
            }

            return ec;
        }

        public override ErrorCode Disconnect(string userName, string channelName)
        {
            return TTV_CSharp_Chat_Disconnect(userName, channelName);
        }

        public override ErrorCode SendMessage(string userName, string channelName, string message)
        {
            return TTV_CSharp_Chat_SendMessage(userName, channelName, message);
        }

        public override ErrorCode BlockUser(string userName, string blockUserName)
        {
            return TTV_CSharp_Chat_BlockUser(userName, blockUserName);
        }

        public override ErrorCode UnblockUser(string userName, string blockUserName)
        {
            return TTV_CSharp_Chat_UnblockUser(userName, blockUserName);
        }

        public override ErrorCode GetUserBlocked(string userName, string blockUserName, ref bool blocked)
        {
            return TTV_CSharp_Chat_GetUserBlocked(userName, blockUserName, ref blocked);
        }

        public override ErrorCode FlushEvents()
        {
            return TTV_CSharp_Chat_FlushEvents();
        }

        public override ErrorCode ForceUserListUpdate(string userName, string channelName)
        {
            return TTV_CSharp_Chat_ForceUserListUpdate(userName, channelName);
        }

        public override ErrorCode GetBadgeData(string channelName, out ChatBadgeData data)
        {
            data = null;

            IntPtr p = IntPtr.Zero;

            ErrorCode ret = TTV_CSharp_Chat_GetBadgeData(channelName, out p);
            if (Error.Succeeded(ret))
            {
                data = new ChatBadgeData(p);
                ret = TTV_CSharp_Chat_FreeBadgeData(p);
            }

            return ret;
        }

        public override ErrorCode FreeMessageList(ref TTV_ChatMessageList messageList)
        {
            return TTV_CSharp_Chat_FreeMessageList(ref messageList);
        }

        public override ErrorCode FreeUserList(ref TTV_ChatUserList userList)
        {
            return TTV_CSharp_Chat_FreeUserList(ref userList);
        }

        public override int GetMessageFlushInterval()
        {
            return (int)TTV_CSharp_Chat_GetMessageFlushInterval();
        }

        public override ErrorCode SetMessageFlushInterval(int milliseconds)
        {
            return TTV_CSharp_Chat_SetMessageFlushInterval((UInt64)milliseconds);
        }

        public override int GetUserListUpdateInterval()
        {
            return (int)TTV_CSharp_Chat_GetUserListUpdateInterval();
        }

        public override ErrorCode SetUserListUpdateInterval(int milliseconds)
        {
            return TTV_CSharp_Chat_SetUserListUpdateInterval((UInt64)milliseconds);
        }

        public override ErrorCode SetBroadcasterLanguageChatEnabled(string userName, string channelName, bool enabled)
        {
            return TTV_CSharp_Chat_SetBroadcasterLanguageChatEnabled(userName, channelName, enabled);
        }

        public override ErrorCode OptInToBroadcasterLanguageChat(string userName, string channelName, string language)
        {
            return TTV_CSharp_Chat_OptInToBroadcasterLanguageChat(userName, channelName, language);
        }

        public override ErrorCode SetLocalLanguage(string language)
        {
            return TTV_CSharp_Chat_SetLocalLanguage(language);
        }
    }
}
