﻿using System;
using System.Collections.Generic;
using System.Runtime.InteropServices;
using System.Runtime.Serialization;
using Twitch.Json;
using ErrorCode = Twitch.ErrorCode;

namespace Twitch.Chat
{
    /// <summary>
    /// https://docs.unity3d.com/Documentation/Manual/PluginsForIOS.html
    /// http://docs.xamarin.com/guides/ios/advanced_topics/limitations/
    /// </summary>
    public class UnityIosChatApi : ChatApi
    {
        #region Callback Signatures
        
        internal delegate void QueuedEventDelegate();

        internal struct ManagedChatAPIListener
        {
            // nothing needs to be passed to native
        }

        internal struct ManagedChatChannelListener
        {
            // nothing needs to be passed to native
        }

        public void ChatInitializationCallback(string json)
        {
            (ChatApi.Instance as UnityIosChatApi).Enqueue(() =>
                {
                    // deserialize json
                    Dictionary<string, object> values = JsonReader.Deserialize(json) as Dictionary<string, object>;
                    if (values == null)
                    {
                        throw new Exception("ChatInitializationCallback error parsing json: " + json);
                    }

                    // extract parameters
                    if (!values.ContainsKey("ec") || !(values["ec"] is Int64))
                    {
                        throw new Exception("ChatInitializationCallback missing valid 'ec' field");
                    }

                    ErrorCode ec = (ErrorCode)(Int64)values["ec"];

                    m_ApiListenerAdapter.ChatInitializationCallback(ec);
                });
        }

        public void ChatShutdownCallback(string json)
        {
            (ChatApi.Instance as UnityIosChatApi).Enqueue(() =>
                {
                    // deserialize json
                    Dictionary<string, object> values = JsonReader.Deserialize(json) as Dictionary<string, object>;
                    if (values == null)
                    {
                        throw new Exception("ChatShutdownCallback error parsing json: " + json);
                    }

                    // extract parameters
                    if (!values.ContainsKey("ec") || !(values["ec"] is Int64))
                    {
                        throw new Exception("ChatShutdownCallback missing valid 'ec' field");
                    }

                    ErrorCode ec = (ErrorCode)(Int64)values["ec"];

                    m_ApiListenerAdapter.ChatShutdownCallback(ec);
                });
        }

        public void ChatUserEmoticonSetsChangedCallback(string json)
        {
            (ChatApi.Instance as UnityIosChatApi).Enqueue(() =>
            {
                // deserialize json
                Dictionary<string, object> values = JsonReader.Deserialize(json) as Dictionary<string, object>;
                if (values == null)
                {
                    throw new Exception("ChatUserEmoticonSetsChangedCallback error parsing json: " + json);
                }

                // extract parameters
                if (!values.ContainsKey("emoticonSets") || !(values["emoticonSets"] is Int64))
                {
                    throw new Exception("ChatUserEmoticonSetsChangedCallback missing valid 'emoticonSets' field");
                }

                IntPtr emoticonSetsPtr = (IntPtr)values["emoticonSets"];
                ChatUserEmoticonSets emoticonSets = new ChatUserEmoticonSets(emoticonSetsPtr);
                m_ApiListenerAdapter.ChatUserEmoticonSetsChangedCallback(emoticonSets);
            });
        }

        public void ChatEmoticonSetDataCallback(string json)
        {
            (ChatApi.Instance as UnityIosChatApi).Enqueue(() =>
            {
                // deserialize json
                Dictionary<string, object> values = JsonReader.Deserialize(json) as Dictionary<string, object>;
                if (values == null)
                {
                    throw new Exception("ChatEmoticonSetDataCallback error parsing json: " + json);
                }

                // extract parameters
                if (!values.ContainsKey("emoticonSet") || !(values["emoticonSet"] is Int64))
                {
                    throw new Exception("ChatEmoticonSetDataCallback missing valid 'emoticonSet' field");
                }

                IntPtr emoticonSetPtr = (IntPtr)values["emoticonSet"];
                ChatEmoticonSetData emoticonSet = new ChatEmoticonSetData(emoticonSetPtr);
                m_ApiListenerAdapter.ChatEmoticonSetDataCallback(emoticonSet);
            });
        }

        public unsafe void ChatChannelStateChangedCallback(string json)
        {
            (ChatApi.Instance as UnityIosChatApi).Enqueue(() =>
            {
                // deserialize json
                Dictionary<string, object> values = JsonReader.Deserialize(json) as Dictionary<string, object>;
                if (values == null)
                {
                    throw new Exception("ChatChannelStateChangedCallback error parsing json: " + json);
                }

                // extract parameters
                if (!values.ContainsKey("channelName") || !(values["channelName"] is string))
                {
                    throw new Exception("ChatChannelStateChangedCallback missing valid 'channelName' field");
                }
                if (!values.ContainsKey("ec") || !(values["ec"] is Int64))
                {
                    throw new Exception("ChatChannelStateChangedCallback missing valid 'ec' field");
                }
                if (!values.ContainsKey("state") || !(values["state"] is Int64))
                {
                    throw new Exception("ChatChannelStateChangedCallback does not contain 'state'");
                }

                string channelName = values["channelName"] as string;
                ErrorCode ec = (ErrorCode)(Int64)values["ec"];
                TTV_ChatChannelState state = (TTV_ChatChannelState)(Int64)values["state"];

                if (!m_ChannelListenerAdapters.ContainsKey(channelName))
                {
                    throw new Exception("Channel not found: " + channelName);
                }

                ChannelListenerAdapter adapter = m_ChannelListenerAdapters[channelName];
                adapter.ChatChannelStateChangedCallback(channelName, state, ec);
            });
        }

        public unsafe void ChatChannelInfoChangedCallback(string json)
        {
            (ChatApi.Instance as UnityIosChatApi).Enqueue(() =>
            {
                // deserialize json
                Dictionary<string, object> values = JsonReader.Deserialize(json) as Dictionary<string, object>;
                if (values == null)
                {
                    throw new Exception("ChatChannelInfoChangedCallback error parsing json: " + json);
                }

                // extract parameters
                if (!values.ContainsKey("channelName") || !(values["channelName"] is string))
                {
                    throw new Exception("ChatChannelInfoChangedCallback missing valid 'channelName' field");
                }
                if (!values.ContainsKey("channelInfo") || !(values["channelInfo"] is Int64))
                {
                    throw new Exception("ChatChannelInfoChangedCallback does not contain 'state'");
                }

                IntPtr channelInfoPtr = IntPtr.Zero;

                try
                {
                    string channelName = values["channelName"] as string;

                    channelInfoPtr = new IntPtr((Int64)values["channelInfo"]);
                    ChatChannelInfo channelInfo = null;
                    if (channelInfoPtr != IntPtr.Zero)
                    {
                        channelInfo = new ChatChannelInfo((TTV_ChatChannelInfo*)channelInfoPtr);
                    }

                    if (!m_ChannelListenerAdapters.ContainsKey(channelName))
                    {
                        throw new Exception("Channel not found: " + channelName);
                    }

                    ChannelListenerAdapter adapter = m_ChannelListenerAdapters[channelName];
                    adapter.ChatChannelInfoChangedCallback(channelName, channelInfo);
                }
                finally
                {
                    // free the channel info that was allocated on the heap for this iOS callback
                    if (channelInfoPtr != IntPtr.Zero)
                    {
                        TTV_CSharp_Chat_FreeChannelInfo(channelInfoPtr);
                    }
                }
            });
        }

        public unsafe void ChatChannelLocalUserChangedCallback(string json)
        {
            (ChatApi.Instance as UnityIosChatApi).Enqueue(() =>
            {
                // deserialize json
                Dictionary<string, object> values = JsonReader.Deserialize(json) as Dictionary<string, object>;
                if (values == null)
                {
                    throw new Exception("ChatChannelLocalUserChangedCallback error parsing json: " + json);
                }

                // extract parameters
                if (!values.ContainsKey("channelName") || !(values["channelName"] is string))
                {
                    throw new Exception("ChatChannelLocalUserChangedCallback missing valid 'channelName' field");
                }
                if (!values.ContainsKey("userInfo") || !(values["userInfo"] is Int64))
                {
                    throw new Exception("ChatChannelLocalUserChangedCallback does not contain 'userInfo'");
                }

                IntPtr userInfoPtr = IntPtr.Zero;

                try
                {
                    string channelName = values["channelName"] as string;

                    userInfoPtr = new IntPtr((Int64)values["userInfo"]);
                    ChatUserInfo userInfo = new ChatUserInfo((TTV_ChatUserInfo*)userInfoPtr);

                    if (!m_ChannelListenerAdapters.ContainsKey(channelName))
                    {
                        throw new Exception("Channel not found: " + channelName);
                    }

                    ChannelListenerAdapter adapter = m_ChannelListenerAdapters[channelName];
                    adapter.ChatChannelLocalUserChangedCallback(channelName, userInfo);
                }
                finally
                {
                    if (userInfoPtr != IntPtr.Zero)
                    {
                        TTV_CSharp_Chat_FreeUserInfo(userInfoPtr);
                    }
                }
            });
        }

        public unsafe void ChatChannelUserChangeCallback(string json)
        {
            (ChatApi.Instance as UnityIosChatApi).Enqueue(() =>
                {
                    // deserialize json
                    Dictionary<string, object> values = JsonReader.Deserialize(json) as Dictionary<string, object>;
                    if (values == null)
                    {
                        throw new Exception("ChatChannelUserChangeCallback error parsing json: " + json);
                    }

                    // extract parameters
                    if (!values.ContainsKey("channelName") || !(values["channelName"] is string))
                    {
                        throw new Exception("ChatChannelUserChangeCallback missing valid 'channelName' field");
                    }
                    if (!values.ContainsKey("joinList") || !(values["joinList"] is Int64))
                    {
                        throw new Exception("ChatChannelUserChangeCallback does not contain 'joinList'");
                    }
                    if (!values.ContainsKey("leaveList") || !(values["leaveList"] is Int64))
                    {
                        throw new Exception("ChatChannelUserChangeCallback does not contain 'leaveList'");
                    }
                    if (!values.ContainsKey("infoChangeList") || !(values["infoChangeList"] is Int64))
                    {
                        throw new Exception("ChatChannelUserChangeCallback does not contain 'infoChangeList'");
                    }

                    IntPtr joinListPtr = IntPtr.Zero;
                    IntPtr leaveListPtr = IntPtr.Zero;
                    IntPtr infoChangeListPtr = IntPtr.Zero;

                    try
                    {
                        string channelName = values["channelName"] as string;

                        joinListPtr = new IntPtr((Int64)values["joinList"]);
                        ChatUserList joinList = new ChatUserList((TTV_ChatUserList*)joinListPtr);

                        leaveListPtr = new IntPtr((Int64)values["leaveList"]);
                        ChatUserList leaveList = new ChatUserList((TTV_ChatUserList*)leaveListPtr);

                        infoChangeListPtr = new IntPtr((Int64)values["infoChangeList"]);
                        ChatUserList infoChangeList = new ChatUserList((TTV_ChatUserList*)infoChangeListPtr);

                        if (!m_ChannelListenerAdapters.ContainsKey(channelName))
                        {
                            throw new Exception("Channel not found: " + channelName);
                        }

                        ChannelListenerAdapter adapter = m_ChannelListenerAdapters[channelName];
                        adapter.ChatChannelUserChangeCallback(channelName, joinList, leaveList, infoChangeList);
                    }
                    finally
                    {
                        if (joinListPtr != IntPtr.Zero)
                        {
                            TTV_CSharp_Chat_FreeUserList_Pointer(joinListPtr);
                        }
                        if (leaveListPtr != IntPtr.Zero)
                        {
                            TTV_CSharp_Chat_FreeUserList_Pointer(leaveListPtr);
                        }
                        if (infoChangeListPtr != IntPtr.Zero)
                        {
                            TTV_CSharp_Chat_FreeUserList_Pointer(infoChangeListPtr);
                        }
                    }
                });
        }

        public unsafe void ChatChannelMessageCallback(string json)
        {
            (ChatApi.Instance as UnityIosChatApi).Enqueue(() =>
                {
                    // deserialize json
                    Dictionary<string, object> values = JsonReader.Deserialize(json) as Dictionary<string, object>;
                    if (values == null)
                    {
                        throw new Exception("ChatChannelMessageCallback error parsing json: " + json);
                    }

                    // extract parameters
                    if (!values.ContainsKey("channelName") || !(values["channelName"] is string))
                    {
                        throw new Exception("ChatChannelMessageCallback missing valid 'channelName' field");
                    }
                    if (!values.ContainsKey("messageList") || !(values["messageList"] is Int64))
                    {
                        throw new Exception("ChatChannelMessageCallback does not contain 'messageList'");
                    }

                    IntPtr messageListPtr = IntPtr.Zero;

                    try
                    {
                        string channelName = values["channelName"] as string;
                        messageListPtr = new IntPtr((Int64)values["messageList"]);

                        ChatMessage[] messageList = ChatMessage.GetArray((TTV_ChatMessageList*)messageListPtr);

                        if (!m_ChannelListenerAdapters.ContainsKey(channelName))
                        {
                            throw new Exception("Channel not found: " + channelName);
                        }

                        ChannelListenerAdapter adapter = m_ChannelListenerAdapters[channelName];
                        adapter.ChatChannelMessageCallback(channelName, messageList);
                    }
                    finally
                    {
                        // free the list
                        if (messageListPtr != IntPtr.Zero)
                        {
                            TTV_CSharp_Chat_FreeTokenizedMessageList_Pointer(messageListPtr);
                        }
                    }
                });
        }

        public void ChatChannelClearCallback(string json)
        {
            (ChatApi.Instance as UnityIosChatApi).Enqueue(() =>
                {
                    Dictionary<string, object> values = JsonReader.Deserialize(json) as Dictionary<string, object>;
                    if (values == null)
                    {
                        throw new Exception("ChatClearCallback error parsing json: " + json);
                    }

                    // extract parameters
                    if (!values.ContainsKey("channelName") || !(values["channelName"] is string))
                    {
                        throw new Exception("ChatClearCallback missing valid 'channelName' field");
                    }
                    if (!values.ContainsKey("username") || !(values["username"] is string))
                    {
                        throw new Exception("ChatClearCallback missing valid 'username' field");
                    }

                    string channelName = values["channelName"] as string;
                    string username = values["username"] as string;

                    if (!m_ChannelListenerAdapters.ContainsKey(channelName))
                    {
                        throw new Exception("Channel not found: " + channelName);
                    }

                    ChannelListenerAdapter adapter = m_ChannelListenerAdapters[channelName];
                    adapter.ChatChannelClearCallback(channelName, username);
                });
        }

        public void ChatChannelBadgeDataDownloadCallback(string json)
        {
            (ChatApi.Instance as UnityIosChatApi).Enqueue(() =>
                {
                    Dictionary<string, object> values = JsonReader.Deserialize(json) as Dictionary<string, object>;
                    if (values == null)
                    {
                        throw new Exception("ChatBadgeDataDownloadCallback error parsing json: " + json);
                    }

                    // extract parameters
                    if (!values.ContainsKey("channelName") || !(values["channelName"] is string))
                    {
                        throw new Exception("ChatBadgeDataDownloadCallback missing valid 'channelName' field");
                    }
                    if (!values.ContainsKey("ec") || !(values["ec"] is Int64))
                    {
                        throw new Exception("ChatBadgeDataDownloadCallback missing valid 'ec' field");
                    }

                    string channelName = values["channelName"] as string;
                    ErrorCode ec = (ErrorCode)(Int64)values["ec"];

                    if (!m_ChannelListenerAdapters.ContainsKey(channelName))
                    {
                        throw new Exception("Channel not found: " + channelName);
                    }

                    ChannelListenerAdapter adapter = m_ChannelListenerAdapters[channelName];
                    adapter.ChatChannelBadgeDataDownloadCallback(channelName, ec);
                });
        }

        public void ChatChannelHostTargetChangedCallback(string json)
        {
            (ChatApi.Instance as UnityIosChatApi).Enqueue(() =>
            {
                Dictionary<string, object> values = JsonReader.Deserialize(json) as Dictionary<string, object>;
                if (values == null)
                {
                    throw new Exception("ChatHostTargetChangedCallback error parsing json: " + json);
                }

                // extract parameters
                if (!values.ContainsKey("channelName") || !(values["channelName"] is string))
                {
                    throw new Exception("ChatHostTargetChangedCallback missing valid 'channelName' field");
                }
                if (!values.ContainsKey("targetChannel") || !(values["targetChannel"] is string))
                {
                    throw new Exception("ChatHostTargetChangedCallback missing valid 'targetChannel' field");
                }
                if (!values.ContainsKey("numViewers") || !(values["numViewers"] is Int64))
                {
                    throw new Exception("ChatHostTargetChangedCallback missing valid 'numViewers' field");
                }

                string channelName = values["channelName"] as string;
                string targetChannel = values["targetChannel"] as string;
                int numViewers = (int)(Int64)values["numViewers"];

                if (!m_ChannelListenerAdapters.ContainsKey(channelName))
                {
                    throw new Exception("Channel not found: " + channelName);
                }

                ChannelListenerAdapter adapter = m_ChannelListenerAdapters[channelName];
                adapter.ChatChannelHostTargetChangedCallback(channelName, targetChannel, numViewers);
            });
        }

        public void ChatChannelNoticeCallback(string json)
        {
            (ChatApi.Instance as UnityIosChatApi).Enqueue(() =>
            {
                Dictionary<string, object> values = JsonReader.Deserialize(json) as Dictionary<string, object>;
                if (values == null)
                {
                    throw new Exception("ChatChannelNoticeCallback error parsing json: " + json);
                }

                // extract parameters
                if (!values.ContainsKey("channelName") || !(values["channelName"] is string))
                {
                    throw new Exception("ChatChannelNoticeCallback missing valid 'channelName' field");
                }
                if (!values.ContainsKey("id") || !(values["id"] is string))
                {
                    throw new Exception("ChatChannelNoticeCallback missing valid 'id' field");
                }
                if (!values.ContainsKey("params") || !(values["params"] is Dictionary<string, object>))
                {
                    throw new Exception("ChatChannelNoticeCallback missing valid 'params' field");
                }

                string channelName = values["channelName"] as string;
                string id = values["id"] as string;
                Dictionary<string, object> parameters = values["params"] as Dictionary<string, object>;

                if (!m_ChannelListenerAdapters.ContainsKey(channelName))
                {
                    throw new Exception("Channel not found: " + channelName);
                }

                Dictionary<string, string> stringParameters = new Dictionary<string, string>();
                foreach (var kvp in parameters)
                {
                    stringParameters[kvp.Key] = kvp.Value as string;
                }

                ChannelListenerAdapter adapter = m_ChannelListenerAdapters[channelName];
                adapter.ChatChannelNoticeCallback(channelName, id, stringParameters);
            });
        }

        public void ChatChannelSetBroadcasterLanguageCallback(string json)
        {
            (ChatApi.Instance as UnityIosChatApi).Enqueue(() =>
            {
                Dictionary<string, object> values = JsonReader.Deserialize(json) as Dictionary<string, object>;
                if (values == null)
                {
                    throw new Exception("ChatChannelSetBroadcasterLanguageCallback error parsing json: " + json);
                }

                // extract parameters
                if (!values.ContainsKey("channelName") || !(values["channelName"] is string))
                {
                    throw new Exception("ChatChannelSetBroadcasterLanguageCallback missing valid 'channelName' field");
                }
                if (!values.ContainsKey("ec") || !(values["ec"] is Int64))
                {
                    throw new Exception("ChatChannelSetBroadcasterLanguageCallback missing valid 'ec' field");
                }

                string channelName = values["channelName"] as string;
                ErrorCode ec = (ErrorCode)(Int64)values["ec"];

                if (!m_ChannelListenerAdapters.ContainsKey(channelName))
                {
                    throw new Exception("Channel not found: " + channelName);
                }

                ChannelListenerAdapter adapter = m_ChannelListenerAdapters[channelName];
                adapter.ChatChannelSetBroadcasterLanguageCallback(channelName, ec);
            });
        }

        private class ApiListenerAdapter
        {
            private UnityIosChatApi m_Api;
            private ManagedChatAPIListener m_ManagedListener;
            private IChatApiListener m_ClientListener;

            public ManagedChatAPIListener ManagedListener
            {
                get { return m_ManagedListener; }
            }

            public IChatApiListener ClientListener
            {
                get { return m_ClientListener; }
                set { m_ClientListener = value; }
            }

            public ApiListenerAdapter(UnityIosChatApi api)
            {
                m_Api = api;
            }

            internal void ChatInitializationCallback(ErrorCode result)
            {
                m_ClientListener.ChatInitializationCallback(result);

                if (Error.Failed(result))
                {
                    m_Api.m_ApiListenerAdapter.ClientListener = null;
                }
            }

            internal void ChatShutdownCallback(ErrorCode result)
            {
                m_ClientListener.ChatShutdownCallback(result);

                if (Error.Succeeded(result))
                {
                    m_Api.m_ApiListenerAdapter.ClientListener = null;
                }
            }

            internal void ChatUserEmoticonSetsChangedCallback(ChatUserEmoticonSets emoticonSets)
            {
                m_ClientListener.ChatUserEmoticonSetsChangedCallback(emoticonSets);
            }

            internal void ChatEmoticonSetDataCallback(ChatEmoticonSetData emoticonSet)
            {
                m_ClientListener.ChatEmoticonSetDataCallback(emoticonSet);
            }
        }

        private class ChannelListenerAdapter
        {
            private UnityIosChatApi m_Api;
            private ManagedChatChannelListener m_ManagedListener;
            private IChatChannelListener m_ClientListener;

            public ManagedChatChannelListener ManagedListener
            {
                get { return m_ManagedListener; }
            }

            public IChatChannelListener ClientListener
            {
                get { return m_ClientListener; }
                set { m_ClientListener = value; }
            }

            public ChannelListenerAdapter(UnityIosChatApi api)
            {
                m_Api = api;
            }

            internal void ChatChannelStateChangedCallback(string channelName, TTV_ChatChannelState state, ErrorCode ec)
            {
                m_ClientListener.ChatChannelStateChangedCallback(channelName, state, ec);

                switch (state)
                {
                    case TTV_ChatChannelState.TTV_CHAT_CHANNEL_STATE_DISCONNECTED:
                        m_ClientListener = null;
                        break;
                    default:
                        break;
                }
            }

            internal void ChatChannelLocalUserChangedCallback(string channelName, ChatUserInfo userInfo)
            {
                m_ClientListener.ChatChannelLocalUserChangedCallback(channelName, userInfo);
            }

            internal void ChatChannelUserChangeCallback(string channelName, ChatUserList joinList, ChatUserList leaveList, ChatUserList infoChangeList)
            {
                m_ClientListener.ChatChannelUserChangeCallback(channelName, joinList.List, leaveList.List, infoChangeList.List);
            }

            internal void ChatChannelInfoChangedCallback(string channelName, ChatChannelInfo channelInfo)
            {
                m_ClientListener.ChatChannelInfoChangedCallback(channelName, channelInfo);
            }

            internal void ChatChannelMessageCallback(string channelName, ChatMessage[] messageList)
            {
                m_ClientListener.ChatChannelMessageCallback(channelName, messageList);
            }

            internal void ChatChannelClearCallback(string channelName, string username)
            {
                m_ClientListener.ChatChannelClearCallback(channelName, username);
            }

            internal void ChatChannelBadgeDataDownloadCallback(string channelName, ErrorCode result)
            {
                m_ClientListener.ChatChannelBadgeDataDownloadCallback(channelName, result);
            }

            internal void ChatChannelHostTargetChangedCallback(string channelName, string targetChannel, int numViewers)
            {
                m_ClientListener.ChatChannelHostTargetChangedCallback(channelName, targetChannel, numViewers);
            }

            internal void ChatChannelNoticeCallback(string channelName, string id, Dictionary<string, string> parameters)
            {
                m_ClientListener.ChatChannelNoticeCallback(channelName, id, parameters);
            }

            internal void ChatChannelSetBroadcasterLanguageCallback(string channelName, ErrorCode ec)
            {
                m_ClientListener.ChatChannelSetBroadcasterLanguageCallback(channelName, ec);
            }
        }

        private ApiListenerAdapter m_ApiListenerAdapter = null;
        private Dictionary<string, ChannelListenerAdapter> m_ChannelListenerAdapters = new Dictionary<string, ChannelListenerAdapter>();

        #endregion

        #region Native Functions

        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Chat_Initialize")]
        internal static extern ErrorCode TTV_CSharp_Chat_Initialize(TTV_ChatTokenizationOption tokenizationOptions, ref ManagedChatAPIListener listener);
        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Chat_Shutdown")]
        internal static extern ErrorCode TTV_CSharp_Chat_Shutdown();

        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Chat_LogIn")]
        internal static extern ErrorCode TTV_CSharp_Chat_LogIn(string userName, string oauthToken);
        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Chat_LogOut")]
        internal static extern ErrorCode TTV_CSharp_Chat_LogOut(string userName);
        
        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Chat_Connect")]
        internal static extern ErrorCode TTV_CSharp_Chat_Connect(string userName, string channelName, ref ManagedChatChannelListener listener);
        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Chat_Disconnect")]
        internal static extern ErrorCode TTV_CSharp_Chat_Disconnect(string userName, string channelName);

        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Chat_SendMessage")]
        internal static extern ErrorCode TTV_CSharp_Chat_SendMessage(string userName, string channelName, string message);
        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Chat_BlockUser")]
        internal static extern ErrorCode TTV_CSharp_Chat_BlockUser(string userName, string blockUserName);
        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Chat_UnblockUser")]
        internal static extern ErrorCode TTV_CSharp_Chat_UnblockUser(string userName, string blockUserName);
        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Chat_GetUserBlocked")]
        internal static extern ErrorCode TTV_CSharp_Chat_GetUserBlocked(string userName, string blockUserName, ref bool blocked);
        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Chat_FlushEvents")]
        internal static extern ErrorCode TTV_CSharp_Chat_FlushEvents();
        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Chat_ForceUserListUpdate")]
        internal static extern ErrorCode TTV_CSharp_Chat_ForceUserListUpdate(string userName, string channelName);

        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Chat_GetBadgeData")]
        internal static extern ErrorCode TTV_CSharp_Chat_GetBadgeData(string channelName, out IntPtr /*TTV_ChatBadgeData*/ data);
        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Chat_FreeBadgeData")]
        internal static extern ErrorCode TTV_CSharp_Chat_FreeBadgeData(IntPtr data);

        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Chat_FreeUserList")]
        internal static extern ErrorCode TTV_CSharp_Chat_FreeUserList(ref TTV_ChatUserList list);
        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Chat_FreeMessageList")]
        internal static extern ErrorCode TTV_CSharp_Chat_FreeMessageList(ref TTV_ChatMessageList list);

        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Chat_GetMessageFlushInterval")]
        internal static extern UInt64 TTV_CSharp_Chat_GetMessageFlushInterval();
        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Chat_SetMessageFlushInterval")]
        internal static extern ErrorCode TTV_CSharp_Chat_SetMessageFlushInterval(UInt64 milliseconds);
        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Chat_GetUserListUpdateInterval")]
        internal static extern UInt64 TTV_CSharp_Chat_GetUserListUpdateInterval();
        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Chat_SetUserListUpdateInterval")]
        internal static extern ErrorCode TTV_CSharp_Chat_SetUserListUpdateInterval(UInt64 milliseconds);

        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Chat_FreeUserInfo")]
        internal static extern ErrorCode TTV_CSharp_Chat_FreeUserInfo(IntPtr data);
        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Chat_FreeChannelInfo")]
        internal static extern ErrorCode TTV_CSharp_Chat_FreeChannelInfo(IntPtr data);
        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Chat_FreeRawMessageList")]
        internal static extern ErrorCode TTV_CSharp_Chat_FreeRawMessageList(IntPtr data);

        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Chat_FreeUserList_Pointer")]
        internal static extern ErrorCode TTV_CSharp_Chat_FreeUserList_Pointer(IntPtr list);
        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Chat_FreeRawMessageList_Pointer")]
        internal static extern ErrorCode TTV_CSharp_Chat_FreeRawMessageList_Pointer(IntPtr list);
        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Chat_FreeTokenizedMessageList_Pointer")]
        internal static extern ErrorCode TTV_CSharp_Chat_FreeTokenizedMessageList_Pointer(IntPtr list);

        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Chat_SetBroadcasterLanguageChatEnabled")]
        internal static extern ErrorCode TTV_CSharp_Chat_SetBroadcasterLanguageChatEnabled(string userName, string channelName, bool enabled);
        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Chat_OptInToBroadcasterLanguageChat")]
        internal static extern ErrorCode TTV_CSharp_Chat_OptInToBroadcasterLanguageChat(string userName, string channelName, string language);
        [DllImport("__Internal", CallingConvention = CallingConvention.Cdecl, EntryPoint = "TTV_CSharp_Chat_SetLocalLanguage")]
        internal static extern ErrorCode TTV_CSharp_Chat_SetLocalLanguage(string language);
        
        #endregion

        private List<QueuedEventDelegate> m_QueuedEvents = new List<QueuedEventDelegate>();

        internal void Enqueue(QueuedEventDelegate func)
        {
            lock (m_QueuedEvents)
            {
                m_QueuedEvents.Add(func);
            }
        }

        public UnityIosChatApi()
        {
            m_ApiListenerAdapter = new ApiListenerAdapter(this);
        }

        public override ErrorCode Initialize(TTV_ChatTokenizationOption tokenizationOptions, IChatApiListener listener)
        {
            if (m_ApiListenerAdapter.ClientListener != null)
            {
                return ErrorCode.TTV_EC_ALREADY_INITIALIZED;
            }

            m_ApiListenerAdapter.ClientListener = listener;

            ManagedChatAPIListener nativeListener = m_ApiListenerAdapter.ManagedListener;
            return TTV_CSharp_Chat_Initialize(tokenizationOptions, ref nativeListener);
        }

        public override ErrorCode Shutdown()
        {
            return TTV_CSharp_Chat_Shutdown();
        }

        public override ErrorCode LogIn(string userName, string oauthToken)
        {
            return TTV_CSharp_Chat_LogIn(userName, oauthToken);
        }

        public override ErrorCode LogOut(string userName)
        {
            return TTV_CSharp_Chat_LogOut(userName);
        }

        public override ErrorCode Connect(string userName, string channelName, IChatChannelListener listener)
        {
            ChannelListenerAdapter adapter = null;
            if (m_ChannelListenerAdapters.ContainsKey(channelName))
            {
                adapter = m_ChannelListenerAdapters[channelName];
            }
            else
            {
                adapter = new ChannelListenerAdapter(this);
            }

            adapter.ClientListener = listener;
            m_ChannelListenerAdapters[channelName] = adapter;

            ManagedChatChannelListener nativeListener = adapter.ManagedListener;
            ErrorCode ec = TTV_CSharp_Chat_Connect(userName, channelName, ref nativeListener);
            if (Error.Failed(ec))
            {
                m_ChannelListenerAdapters.Remove(channelName);
            }

            return ec;
        }

        public override ErrorCode Disconnect(string userName, string channelName)
        {
            return TTV_CSharp_Chat_Disconnect(userName, channelName);
        }

        public override ErrorCode SendMessage(string userName, string channelName, string message)
        {
            return TTV_CSharp_Chat_SendMessage(userName, channelName, message);
        }

        public override ErrorCode BlockUser(string userName, string blockUserName)
        {
            return TTV_CSharp_Chat_BlockUser(userName, blockUserName);
        }

        public override ErrorCode UnblockUser(string userName, string blockUserName)
        {
            return TTV_CSharp_Chat_UnblockUser(userName, blockUserName);
        }

        public override ErrorCode GetUserBlocked(string userName, string blockUserName, ref bool blocked)
        {
            return TTV_CSharp_Chat_GetUserBlocked(userName, blockUserName, ref blocked);
        }
      
        public override ErrorCode FlushEvents()
        {
            // fire queued up events
            List<QueuedEventDelegate> copy = new List<QueuedEventDelegate>();

            lock (m_QueuedEvents)
            {
                copy.AddRange(m_QueuedEvents);
                m_QueuedEvents.Clear();
            }

            foreach (QueuedEventDelegate e in copy)
            {
                e();
            } 
            
            return TTV_CSharp_Chat_FlushEvents();
        }

        public override ErrorCode ForceUserListUpdate(string userName, string channelName)
        {
            return TTV_CSharp_Chat_ForceUserListUpdate(userName, channelName);
        }

        public override ErrorCode GetBadgeData(string channelName, out ChatBadgeData data)
        {
            data = null;

            IntPtr p = IntPtr.Zero;

            ErrorCode ret = TTV_CSharp_Chat_GetBadgeData(channelName, out p);
            if (Error.Succeeded(ret))
            {
                data = new ChatBadgeData(p);
                ret = TTV_CSharp_Chat_FreeBadgeData(p);
            }

            return ret;
        }

        public override ErrorCode FreeMessageList(ref TTV_ChatMessageList messageList)
        {
            return TTV_CSharp_Chat_FreeMessageList(ref messageList);
        }

        public override ErrorCode FreeUserList(ref TTV_ChatUserList userList)
        {
            return TTV_CSharp_Chat_FreeUserList(ref userList);
        }

        public override int GetMessageFlushInterval()
        {
            return (int)TTV_CSharp_Chat_GetMessageFlushInterval();
        }

        public override ErrorCode SetMessageFlushInterval(int milliseconds)
        {
            return TTV_CSharp_Chat_SetMessageFlushInterval((UInt64)milliseconds);
        }

        public override int GetUserListUpdateInterval()
        {
            return (int)TTV_CSharp_Chat_GetUserListUpdateInterval();
        }

        public override ErrorCode SetUserListUpdateInterval(int milliseconds)
        {
            return TTV_CSharp_Chat_SetUserListUpdateInterval((UInt64)milliseconds);
        }

        public override ErrorCode SetBroadcasterLanguageChatEnabled(string userName, string channelName, bool enabled)
        {
            return TTV_CSharp_Chat_SetBroadcasterLanguageChatEnabled(userName, channelName, enabled);
        }

        public override ErrorCode OptInToBroadcasterLanguageChat(string userName, string channelName, string language)
        {
            return TTV_CSharp_Chat_OptInToBroadcasterLanguageChat(userName, channelName, language);
        }

        public override ErrorCode SetLocalLanguage(string language)
        {
            return TTV_CSharp_Chat_SetLocalLanguage(language);
        }
    }
}
