﻿using System;
using System.Collections.Generic;
using System.Runtime.InteropServices;


namespace Twitch
{
    public class MemoryManager
    {
        protected class PinnedObject
        {
            public int count = 0;
            public GCHandle handle;
        }

        protected static List<PinnedObject> m_List = new List<PinnedObject>();

        protected static Dictionary<object, PinnedObject> m_PinnedObjects = new Dictionary<object, PinnedObject>();
        protected static Dictionary<IntPtr, PinnedObject> m_PinnedPointers = new Dictionary<IntPtr, PinnedObject>();
        protected static Dictionary<IntPtr, FrameBuffer> m_FrameBuffers = new Dictionary<IntPtr, FrameBuffer>();

        public static IntPtr FindPointer(object obj)
        {
            for (int i=0; i<m_List.Count; ++i)
            {
                PinnedObject pin = m_List[i];
            }

            PinnedObject entry = null;
            m_PinnedObjects.TryGetValue(obj, out entry);
            return entry == null ? IntPtr.Zero : entry.handle.AddrOfPinnedObject();
        }

        public static IntPtr PinObject(object obj)
        {
            PinnedObject entry = null;
            
            lock (m_PinnedObjects)
            {
                if (m_PinnedObjects.TryGetValue(obj, out entry))
                {
                    entry.count++;
                    return entry.handle.AddrOfPinnedObject();
                }

                entry = new PinnedObject();
                entry.count = 1;
                entry.handle = GCHandle.Alloc(obj, GCHandleType.Pinned);

                if (entry.handle.AddrOfPinnedObject() == IntPtr.Zero)
                {
                    return IntPtr.Zero;
                }

                m_PinnedObjects[obj] = entry;

                IntPtr ptr = entry.handle.AddrOfPinnedObject();

                lock (m_PinnedPointers)
                {
                    m_PinnedPointers[ptr] = entry;
                }

                return ptr;
            }
        }

        public static bool UnpinObject(object obj)
        {
            PinnedObject entry = null;

            lock (m_PinnedObjects)
            {
                if (m_PinnedObjects.TryGetValue(obj, out entry))
                {
                    entry.count--;
                    if (entry.count == 0)
                    {
                        lock (m_PinnedPointers)
                        {
                            m_PinnedPointers.Remove(entry.handle.AddrOfPinnedObject());
                        }

                        m_PinnedObjects.Remove(obj);
                        entry.handle.Free();

                        return false;
                    }
                    else
                    {
                        return true;
                    }
                }

                return false;
            }
        }

        public static void RegisterFrameBuffer(FrameBuffer buffer)
        {
            m_FrameBuffers[buffer.Pointer] = buffer;
        }

        public static void UnregisterFrameBuffer(FrameBuffer buffer)
        {
            if (buffer.Pointer != IntPtr.Zero)
            {
                if (m_FrameBuffers.ContainsKey(buffer.Pointer))
                {
                    m_FrameBuffers.Remove(buffer.Pointer);
                }
            }
        }

        public static FrameBuffer LookupFrameBuffer(IntPtr p)
        {
            FrameBuffer result = null;
            m_FrameBuffers.TryGetValue(p, out result);
            return result;
        }
    }
}
