//
// Sample code for capturing microphone audio using the old waveIn functions that are available on Windows XP
//

#include <Windows.h>
#include <stdio.h>

static volatile bool gCaptureRunning = false;
static MMIOINFO gMmIoInfo;
static HMMIO gHmmio;
static MMCKINFO gCkRIFF;
static MMCKINFO gCkData;

const int kTotalCaptureDuration = 10;

HRESULT WriteWaveHeader(HMMIO hFile, LPCWAVEFORMATEX pwfx, MMCKINFO *pckRIFF, MMCKINFO *pckData) 
{
    MMRESULT result;
	

    // make a RIFF/WAVE chunk
    pckRIFF->ckid = MAKEFOURCC('R', 'I', 'F', 'F');
    pckRIFF->fccType = MAKEFOURCC('W', 'A', 'V', 'E');

    result = mmioCreateChunk(hFile, pckRIFF, MMIO_CREATERIFF);
    if (MMSYSERR_NOERROR != result) {
        printf("mmioCreateChunk(\"RIFF/WAVE\") failed: MMRESULT = 0x%08x\n", result);
        return E_FAIL;
    }
    
    // make a 'fmt ' chunk (within the RIFF/WAVE chunk)
    MMCKINFO chunk;
    chunk.ckid = MAKEFOURCC('f', 'm', 't', ' ');
    result = mmioCreateChunk(hFile, &chunk, 0);
    if (MMSYSERR_NOERROR != result) {
        printf("mmioCreateChunk(\"fmt \") failed: MMRESULT = 0x%08x\n", result);
        return E_FAIL;
    }

    // write the WAVEFORMATEX data to it
    LONG lBytesInWfx = sizeof(WAVEFORMATEX) + pwfx->cbSize;
    LONG lBytesWritten =
        mmioWrite(
            hFile,
            reinterpret_cast<PCHAR>(const_cast<LPWAVEFORMATEX>(pwfx)),
            lBytesInWfx
        );
    if (lBytesWritten != lBytesInWfx) {
        printf("mmioWrite(fmt data) wrote %u bytes; expected %u bytes\n", lBytesWritten, lBytesInWfx);
        return E_FAIL;
    }

    // ascend from the 'fmt ' chunk
    result = mmioAscend(hFile, &chunk, 0);
    if (MMSYSERR_NOERROR != result) {
        printf("mmioAscend(\"fmt \" failed: MMRESULT = 0x%08x\n", result);
        return E_FAIL;
    }
    
    // make a 'fact' chunk whose data is (DWORD)0
    chunk.ckid = MAKEFOURCC('f', 'a', 'c', 't');
    result = mmioCreateChunk(hFile, &chunk, 0);
    if (MMSYSERR_NOERROR != result) {
        printf("mmioCreateChunk(\"fmt \") failed: MMRESULT = 0x%08x\n", result);
        return E_FAIL;
    }

    // write (DWORD)0 to it
    // this is cleaned up later
    DWORD frames = 0;
    lBytesWritten = mmioWrite(hFile, reinterpret_cast<PCHAR>(&frames), sizeof(frames));
    if (lBytesWritten != sizeof(frames)) {
        printf("mmioWrite(fact data) wrote %u bytes; expected %u bytes\n", lBytesWritten, (UINT32)sizeof(frames));
        return E_FAIL;
    }

    // ascend from the 'fact' chunk
    result = mmioAscend(hFile, &chunk, 0);
    if (MMSYSERR_NOERROR != result) {
        printf("mmioAscend(\"fact\" failed: MMRESULT = 0x%08x\n", result);
        return E_FAIL;
    }

    // make a 'data' chunk and leave the data pointer there
    pckData->ckid = MAKEFOURCC('d', 'a', 't', 'a');
    result = mmioCreateChunk(hFile, pckData, 0);
    if (MMSYSERR_NOERROR != result) {
        printf("mmioCreateChunk(\"data\") failed: MMRESULT = 0x%08x\n", result);
        return E_FAIL;
    }

    return S_OK;
}

HRESULT FinishWaveFile(HMMIO hFile, MMCKINFO *pckRIFF, MMCKINFO *pckData) {
    MMRESULT result;

    result = mmioAscend(hFile, pckData, 0);
    if (MMSYSERR_NOERROR != result) {
        printf("mmioAscend(\"data\" failed: MMRESULT = 0x%08x\n", result);
        return E_FAIL;
    }

    result = mmioAscend(hFile, pckRIFF, 0);
    if (MMSYSERR_NOERROR != result) {
        printf("mmioAscend(\"RIFF/WAVE\" failed: MMRESULT = 0x%08x\n", result);
        return E_FAIL;
    }

    return S_OK;    
}

void CALLBACK waveInProc(HWAVEIN hwi,UINT uMsg,DWORD dwInstance,DWORD dwParam1,DWORD dwParam2)
{
	switch(uMsg)
	{
	case WIM_DATA:
		{
			static int recordedChunks = 0;
			printf("\nReceived capture data...\n");
			WAVEHDR* hdr = (WAVEHDR*)dwParam1;
			if (hdr)
			{		
				// Process the data
				mmioWrite(gHmmio, hdr->lpData, hdr->dwBytesRecorded);

				// Do more capture
				waveInAddBuffer(hwi, hdr, sizeof(WAVEHDR));

				if (++recordedChunks >= kTotalCaptureDuration)
				{
					gCaptureRunning = false;
				}
			}
		}
		break;
	default:
		break;
	}
}

int main()
{
	if (waveInGetNumDevs() == 0)
	{
		return 1;
	}

	const int kSampleRate = 44100;
	const int kBufferDuration = 1;
	const int kChannels = 2;

	short* captureBuffer = new short[kSampleRate * kBufferDuration * kChannels];

	HWAVEIN hwi;
	WAVEFORMATEX wfx;
	wfx.wFormatTag = WAVE_FORMAT_PCM;
	wfx.wBitsPerSample = 16;
	wfx.nChannels = kChannels;
	wfx.nSamplesPerSec = kSampleRate;
	wfx.nAvgBytesPerSec = wfx.nSamplesPerSec * wfx.nChannels * wfx.wBitsPerSample / 8;
	wfx.nBlockAlign = wfx.nChannels * wfx.wBitsPerSample / 8;
	wfx.cbSize = 0;

	ZeroMemory(&gMmIoInfo, sizeof(MMIOINFO));
	gHmmio = mmioOpen(L"output.wav", &gMmIoInfo, MMIO_WRITE | MMIO_CREATE);		
	ZeroMemory(&gCkData,sizeof(MMCKINFO));
	WriteWaveHeader(gHmmio, &wfx, &gCkRIFF, &gCkData);

	MMRESULT ret = waveInOpen(&hwi, WAVE_MAPPER, &wfx, 0, 0, WAVE_FORMAT_QUERY);
	if (ret == MMSYSERR_NOERROR)
	{
		ret = waveInOpen(&hwi, WAVE_MAPPER, &wfx, (DWORD_PTR)waveInProc, 0, CALLBACK_FUNCTION);

		if (ret == MMSYSERR_NOERROR)
		{
			WAVEHDR waveHdr;
			waveHdr.lpData = (LPSTR)captureBuffer;
			waveHdr.dwBufferLength = kBufferDuration * kSampleRate * kChannels * sizeof(short);
			waveHdr.dwUser = 0L;
			waveHdr.dwFlags = 0L;
			waveHdr.dwLoops = 0L;
			waveInPrepareHeader(hwi, &waveHdr, sizeof(WAVEHDR));

			ret = waveInAddBuffer(hwi, &waveHdr, sizeof(WAVEHDR));

			if (ret == MMSYSERR_NOERROR)
			{
				ret = waveInStart(hwi);
				if (ret == MMSYSERR_NOERROR)
				{
					gCaptureRunning = true;
				}
			}

			while (gCaptureRunning)
			{
				Sleep(5);
			}

			waveInStop(hwi);
			waveInUnprepareHeader(hwi, &waveHdr, sizeof(WAVEHDR));
			waveInClose(hwi);


			FinishWaveFile(gHmmio, &gCkData, &gCkRIFF);

			///
			// Capture without using a callback 
			//
			//if (ret == MMSYSERR_NOERROR)
			//{
			//	ret = waveInStart(hwi);
			//	if (ret == MMSYSERR_NOERROR)
			//	{
			//		do
			//		{
			//			printf("\n!!! recording \n");
			//		} while(waveInUnprepareHeader(hwi, &waveHdr, sizeof(WAVEHDR)) == WAVERR_STILLPLAYING);

			//		waveInClose(hwi);
			//		delete [] captureBuffer;
			//	}

			//}

		}
	}

	

	return 0;
}