#include "gamestreamer.h"
#include "twitchsdk.h"
#include <d3d9.h>
#include <d3dx9.h>
#include <dxva2api.h>
#include <assert.h>

const bool kAudioEnabled = false;
const bool kSubmitD3DSurfaces = true;

extern IDirect3DDevice9* WINAPI DXUTGetD3D9Device();
extern IDirect3DDeviceManager9* WINAPI DXUTGetD3D9DeviceManager();
extern IDirect3D9* WINAPI DXUTGetD3D9Object();

namespace JTV_Streamer
{	
	void* AllocCallback(size_t size)
	{
		return malloc(size);
	}

	void FreeCallback(void* ptr)
	{
		free(ptr);
	}

	void FrameUnlockCallback (const uint8_t* buffer, void* userData)
	{
		delete[] buffer;
	}


	//----------------------------------------------------------------------------------
	void Init(const char* localFileName)
	{	
		TTV_MemCallbacks memCallbacks;
		memCallbacks.size = sizeof(TTV_MemCallbacks);
		memCallbacks.allocCallback = AllocCallback;	
		memCallbacks.freeCallback = FreeCallback;
	
		TTV_VideoParams videoParams;
		videoParams.size = sizeof(TTV_VideoParams);	
		videoParams.outputWidth = 1280;
		videoParams.outputHeight = 720;
		videoParams.pixelFormat = TTV_PF_BGRA;

		TTV_AudioParams audioParams;
		audioParams.size = sizeof(TTV_AudioParams);
		audioParams.audioEnabled = kAudioEnabled;
		audioParams.enableMicCapture = true;
		audioParams.enablePlaybackCapture = true;
		audioParams.enablePassthroughAudio = false;

		TTV_AuthParams authParams;
		authParams.size = sizeof(TTV_AudioParams);
		//authParams.clientID = ??
		//authParams.userName = ??
		// authParams.password = ??
					
		TTV_ErrorCode ret = TTV_EC_SUCCESS;
#if D3D_SUPPORT_ENABLED
		if (kSubmitD3DSurfaces)
		{
			ret = TTV_InitWithD3D9(&memCallbacks, DXUTGetD3D9DeviceManager());

		}
		else
#endif
		{
			ret = TTV_Init(&memCallbacks);
		}

		if (ret == TTV_EC_SUCCESS)
		{
			ret = TTV_GetDefaultParams(&videoParams);
			if (ret == TTV_EC_SUCCESS)
			{
				ret = TTV_Start(&videoParams, &audioParams, localFileName);
			}
		}
		 
		assert(ret == TTV_EC_SUCCESS);
	}
			
	//----------------------------------------------------------------------------------
	void Update(IDirect3DSurface9* backBuffer)
	{	
		unsigned char* bgraFrame = NULL;		
		int width = 0;
		int height = 0;
		bool gotFrame = GetBgraFrame(backBuffer, bgraFrame, width, height);
		if (gotFrame)
		{	
			TTV_ErrorCode ret = TTV_SubmitVideoFrame(bgraFrame, FrameUnlockCallback, 0);
			assert(ret == TTV_EC_SUCCESS);			
		}		
	}

	//----------------------------------------------------------------------------------	
	void Shutdown()
	{
		TTV_Shutdown();
	}

	//----------------------------------------------------------------------------------	
	bool GetBgraFrame(IDirect3DSurface9* backBuffer, unsigned char*& bgraFrame, int& srcWidth, int& srcHeight)
	{

		static bool drop = false;
		if (drop)
		{
			drop = false;
			return false;
		}

		drop = true;

		assert(backBuffer);
		if (backBuffer)
		{	
			HRESULT hr = S_OK;

			IDirect3DSurface9* pRenderTarget = backBuffer;

			// *** TODO - Need to make these global so that we can call Release on them in the shutdown() code
			//
			static IDirect3DSurface9* pDestTarget0 = NULL;			
			static IDirect3DSurface9* pDestTarget1 = NULL;			

			static int surfCounter = 0;
			
	
			D3DSURFACE_DESC srcDesc;
			pRenderTarget->GetDesc(&srcDesc);

			srcWidth = srcDesc.Width;
			srcHeight = srcDesc.Height;			

			// create a destination surface and copy the render target to the destination surface
			IDirect3DSurface9** pCurDestSurface = surfCounter % 2 ? &pDestTarget1 : &pDestTarget0;
			hr = DXUTGetD3D9Device()->CreateOffscreenPlainSurface(srcWidth, srcHeight, srcDesc.Format, D3DPOOL_SYSTEMMEM, pCurDestSurface, NULL);			
			hr = DXUTGetD3D9Device()->GetRenderTargetData(pRenderTarget, *pCurDestSurface);
						
			if (surfCounter == 0)
			{
				// On the very first frame we don't have a second surface to lock
				++surfCounter;
				return false;
			}
			
			// save its contents to a bitmap file.
			// hr = D3DXSaveSurfaceToFileA("temp.bmp", D3DXIFF_BMP, pDestTarget, NULL, NULL);

			D3DLOCKED_RECT rect;	

			IDirect3DSurface9** pCurLockTarget = surfCounter % 2 ? &pDestTarget0 : &pDestTarget1;
			hr = (*pCurLockTarget)->LockRect(&rect, 0, 0);			

			// Need to update src width and height since the window may have been resized between the last frame and this one (we need to get the width/heigh of the frame at the time it was copied)
			(*pCurLockTarget)->GetDesc(&srcDesc);
			srcWidth = srcDesc.Width;
			srcHeight = srcDesc.Height;

			const int kPixelSize = 4;
			const int bgraWidthBytes = srcWidth * kPixelSize;
			const int bgraFrameBytes = srcWidth * srcHeight * kPixelSize;
			bgraFrame = new unsigned char[bgraFrameBytes];

			for (int y = 0; y < srcHeight; ++y)
			{				
				memcpy( bgraFrame + y * bgraWidthBytes, 
						(char*)rect.pBits +  y * bgraWidthBytes,
						bgraWidthBytes );
			}

			(*pCurLockTarget)->UnlockRect();
			(*pCurLockTarget)->Release();			
		}

		return true;
 	}

}