////////////////////////////////////////////////////////////////////////////////////
// This file contains the implementation of the chat system.
////////////////////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "chat.h"
#include "twitchsdk.h"
#include "twitchchat.h"
#include "chatrenderer.h"

#include <assert.h>
#include <string>

extern TTV_AuthToken gAuthToken;
extern std::string gUserName;

#define CHAT_STATE(__state__) CS_##__state__

namespace
{
	ChatState gChatState = CHAT_STATE(Uninitialized);
	TTV_ErrorCode gAsyncResult = TTV_EC_SUCCESS;
	TTV_ChatTokenizationOption gEmoticonOptions = TTV_CHAT_TOKENIZATION_OPTION_NONE;

	void InitializationCallback(TTV_ErrorCode result, void* /*userdata*/)
	{
		gAsyncResult = result;

		if (TTV_SUCCEEDED(result))
		{
			gChatState = CHAT_STATE(Disconnected);
		}
		else
		{
			gChatState = CHAT_STATE(Uninitialized);
		}
	}

	void ShutdownCallback(TTV_ErrorCode result, void* /*userdata*/)
	{
		gAsyncResult = result;

		if (TTV_SUCCEEDED(result))
		{
			gChatState = CHAT_STATE(Uninitialized);
		}
		else
		{
			gChatState = CHAT_STATE(Disconnected);
		}
	}

	void ChatChannelStateChangedCallback(TTV_ChatChannelState state, TTV_ErrorCode ec, void* /*userdata*/)
	{
		printf("Received ChatChannelStateChangedCallback: state:%d, error:%d\n", state, ec);

		switch (state)
		{
		case TTV_CHAT_CHANNEL_STATE_DISCONNECTED:
			gChatState = CHAT_STATE(Disconnected);
			break;
		case TTV_CHAT_CHANNEL_STATE_CONNECTING:
			gChatState = CHAT_STATE(Connecting);
			break;
		case TTV_CHAT_CHANNEL_STATE_CONNECTED:
			gChatState = CHAT_STATE(Connected);
			break;
		}
	}

	void ChatChannelInfoChangedCallback(const TTV_ChatChannelInfo* channelInfo, void* /*userdata*/)
	{
		// You can use the localUserRestriction to lock out the UI when it's not TTV_CHAT_RESTRICTION_REASON_NONE
	}

	void ChatLocalUserChangedCallback(const TTV_ChatUserInfo* userInfo, void* /*userdata*/)
	{
	}

	void ChatUserCallback(const TTV_ChatUserList* joinList, const TTV_ChatUserList* leaveList, const TTV_ChatUserList* userInfoList, void* /*userdata*/)
	{
		for (uint i = 0; i < leaveList->userCount; ++i)
		{
			RemoveChatUser(&leaveList->userList[i]);
		}

		for (uint i = 0; i < joinList->userCount; ++i)
		{
			AddChatUser(&joinList->userList[i]);
		}

		for (uint i = 0; i < userInfoList->userCount; ++i)
		{
			UpdateChatUser(&userInfoList->userList[i]);
		}

		//////////////////////////////////////////////////////////////////////////
		// Important to free user lists when we are done with them
		//////////////////////////////////////////////////////////////////////////
		TTV_Chat_FreeUserList(joinList);
		TTV_Chat_FreeUserList(leaveList);
		TTV_Chat_FreeUserList(userInfoList);
	}

	void ChatTokenizedMessageCallback(const TTV_ChatTokenizedMessageList* messageList, void* /*userdata*/)
	{
		assert (messageList);

		AddChatMessages(messageList);
	}

	void ChatClearCallback(const utf8char* username, void* /*userdata*/)
	{
		ClearChatMessages(username);
	}
}


void InitializeChat()
{
	TTV_ErrorCode ret = TTV_Chat_Init(gEmoticonOptions, InitializationCallback, nullptr);
	ASSERT_ON_ERROR(ret);

	if (TTV_SUCCEEDED(ret))
	{
		gChatState = CHAT_STATE(Disconnected);
	}
}


void ConnectChat(const utf8char* channel)
{
	if (gChatState != CHAT_STATE(Disconnected))
	{
		return;
	}

	TTV_ChatCallbacks chatCallbacks;
	memset(&chatCallbacks, 0, sizeof(chatCallbacks));
	chatCallbacks.channelStateChangedCallback = ChatChannelStateChangedCallback;
	chatCallbacks.channelInfoChangedCallback = ChatChannelInfoChangedCallback;
	chatCallbacks.localUserChangedCallback = ChatLocalUserChangedCallback;
	chatCallbacks.userCallback = ChatUserCallback;
	chatCallbacks.messageCallback = nullptr;
	chatCallbacks.tokenizedMessageCallback = ChatTokenizedMessageCallback;
	chatCallbacks.clearCallback = ChatClearCallback;
	chatCallbacks.unsolicitedUserData = nullptr;

	TTV_ErrorCode ret = TTV_Chat_Connect(gUserName.c_str(), gAuthToken.data, &chatCallbacks);
	ASSERT_ON_ERROR(ret);

	if (TTV_SUCCEEDED(ret))
	{
		gChatState = CHAT_STATE(Connecting);
	}
}


void DisconnectChat()
{
	if (gChatState != CHAT_STATE(Connecting) &&
		gChatState != CHAT_STATE(Connected))
	{
		return;
	}

	(void)TTV_Chat_Disconnect();
}


void ShutdownChat()
{
	if (gChatState != CHAT_STATE(Uninitialized) &&
		gChatState != CHAT_STATE(ShuttingDown))
	{
		return;
	}

	gChatState = CHAT_STATE(ShuttingDown);

	TTV_ErrorCode ret = TTV_Chat_Shutdown(ShutdownCallback, nullptr);

	if (TTV_SUCCEEDED(ret))
	{
		while (gChatState != CHAT_STATE(Uninitialized))
		{
			FlushChatEvents();
		}
	}
}


ChatState GetChatState()
{
	return gChatState;
}


void FlushChatEvents()
{
	TTV_Chat_FlushEvents();

	switch (gChatState)
	{
		case CHAT_STATE(Uninitialized):
		{
			break;
		}
		case CHAT_STATE(Initializing):
		{
			break;
		}
		case CHAT_STATE(Connecting):
		{
			break;
		}
		case CHAT_STATE(Connected):
		{
			break;
		}
		case CHAT_STATE(Disconnected):
		case CHAT_STATE(ShuttingDown):
		{
			ClearChatUsers();
			ClearChatMessages(nullptr);
			break;
		}
	}
}
