﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using Twitch;
using ErrorCode = Twitch.ErrorCode;

namespace Twitch.Chat
{
    /// <summary>
    /// The state machine which manages the chat state.  It provides a high level interface to the SDK libraries.  The ChatController (CC) performs many operations
    /// asynchronously and hides the details.  This class can be tweaked if needed but should handle all your chat needs (other than emoticons which may be provided in the future).
    /// 
    /// The typical order of operations a client of CC will take is:
    /// 
    /// - Subscribe for events via delegates on ChatController
    /// - Call CC.Initialize() when starting your game
    /// - Wait for the initialization callback 
    /// - Call CC.Connect()
    /// - Wait for the connection callback 
    /// - Call CC.SendChatMessage() to send messages (if not connected anonymously)
    /// - Receive message callbacks
    /// - Call CC.Disconnect() when done
    /// - Wait for the disconnection callback 
    /// - Call CC.Shutdown() when shutting down your game
    /// 
    /// Events will fired during the call to CC.Update().  When chat messages are received RawMessagesReceived will be fired.
    /// </summary>
    public abstract partial class ChatController
    {
        protected const int MIN_INTERVAL_MS = 200;
        protected const int MAX_INTERVAL_MS = 10000;

        #region Types

        /// <summary>
        /// The possible states the ChatController can be in.
        /// </summary>
        public enum ChatState
        {
            Uninitialized,  //!< The component is not yet initialized.
            Initializing,   //!< The component is initializing.
            Initialized,    //!< The component is initialized.
            ShuttingDown, 	//!< The component is shutting down.
        }

        /// <summary>
        /// The possible states a chat channel can be in.
        /// </summary>
        public enum ChannelState
        {
            Disconnected,
            Connecting,
            Connected,
            Disconnecting,
        }

        /// <summary>
        /// The emoticon parsing mode for chat messages.
        /// </summary>
        public enum EmoticonMode
        {
            None,			//!< Do not parse out emoticons in messages.
            Url, 			//!< Parse out emoticons and return urls only for images.
        }

        /// <summary>
        /// The callback for the event fired when initialization is complete.
        /// </summary>
        public delegate void InitializationCompleteDelegate(ErrorCode result);

        /// <summary>
        /// The callback for the event fired when shutdown is complete.
        /// </summary>
        public delegate void ShutdownCompleteDelegate(ErrorCode result);

        /// <summary>
        /// The callback signature for the event which is fired when the ChatController changes state.
        /// </summary>
        /// <param name="state">The new state.</param>
        /// <param name="result">The error code associated with the transition.</param>
        public delegate void ChatStateChangedDelegate(ChatState state, ErrorCode result);

        /// <summary>
        /// The callback signature for the event fired when a set of messages has been received.
        /// </summary>
        /// <param name="messages">The list of messages</param>
        public delegate void MessagesReceivedDelegate(string channelName, ChatMessage[] messages);

        /// <summary>
        /// The callback signature for the event fired when the properties of the local user change.
        /// </summary>
        public delegate void LocalUserChangedDelegate(string channelName, ChatUserInfo userInfo);

        /// <summary>
        /// The callback signature for the event fired when the emoticon sets have changed for a local user.
        /// </summary>
        public delegate void LocalUserEmoticonSetsChangedDelegate(string username);

        /// <summary>
        /// The callback signature for the event fired when the content of an emoticon set has been updated.
        /// </summary>
        /// <param name="emoticonSet"></param>
        public delegate void EmoticonSetDataAvailableDelegate(ChatEmoticonSetData emoticonSet);

        public delegate void UserBlockChangeDelegate(string username, string blockUsername, bool block, ErrorCode result);

        /// <summary>
        /// The callback signature for the event fired when users join, leave or changes their status in the channel.
        /// </summary>
        /// <param name="joinList">The list of users who have joined the room.</param>
        /// <param name="leaveList">The list of useres who have left the room.</param>
        /// <param name="userInfoList">The list of users who have changed their status.</param>
        public delegate void UsersChangedDelegate(string channelName, ChatUserInfo[] joinList, ChatUserInfo[] leaveList, ChatUserInfo[] userInfoList);

        /// <summary>
        /// The callback signature for the event fired when the channel state has changed.
        /// </summary>
        public delegate void ChannelStateChangedDelegate(string channelName, ChannelState state, ErrorCode result);

        /// <summary>
        /// The callback signature for the event fired when the messages in the room should be cleared.  The UI should be cleared of any previous messages.
        /// If username is null or empty then the entire log was cleared, otherwise only messages for the given user were cleared.
        /// </summary>
        public delegate void ClearMessagesDelegate(string channelName, string username);

        /// <summary>
        /// The callback signature for the event fired when a channel property changes.
        /// </summary>
        public delegate void ChannelInfoChangedDelegate(string channelName, ChatChannelInfo info);

        /// <summary>
        /// The callback signature for the event fired when the badge data has been made available.
        /// </summary>
        public delegate void BadgeDataAvailableDelegate(string channelName);

        /// <summary>
        /// The callback signature for the event fired when host mode changes.
        /// </summary>
        public delegate void HostTargetChangedDelegate(string channelName, string targetChannel, int numViewers);

        /// <summary>
        /// The callback signature for a notice message.
        /// </summary>
        public delegate void NoticeReceivedDelegate(string channelName, string id, Dictionary<string, string> parameters);

        /// <summary>
        /// The Callback for requests to set the broadcaster language.
        /// </summary>
        public delegate void SetBroadcasterLanguageResultReceivedDelegate(string channelName, ErrorCode ec);

        #endregion

        #region Memeber Variables

        public event InitializationCompleteDelegate InitializationComplete;
        public event ShutdownCompleteDelegate ShutdownComplete;
        public event BadgeDataAvailableDelegate BadgeDataAvailable;
        public event ChatStateChangedDelegate ChatStateChanged;
        public event MessagesReceivedDelegate MessagesReceived;
        public event LocalUserChangedDelegate LocalUserChanged;
        public event UserBlockChangeDelegate UserBlockChange;
        public event UsersChangedDelegate UsersChanged;
        public event ChannelStateChangedDelegate ChannelStateChanged;
        public event ClearMessagesDelegate MessagesCleared;
        public event ChannelInfoChangedDelegate ChannelInfoChanged;
        public event LocalUserEmoticonSetsChangedDelegate LocalUserEmoticonSetsChanged;
        public event EmoticonSetDataAvailableDelegate EmoticonSetDataAvailable;
        public event HostTargetChangedDelegate HostTargetChanged;
        public event NoticeReceivedDelegate NoticeReceived;
        public event SetBroadcasterLanguageResultReceivedDelegate SetBroadcasterLanguageResultReceived;

        protected string m_UserName = "";
        protected CoreApi m_CoreApi = null;
        protected ChatApi m_ChatApi = null;

        protected ChatState m_ChatState = ChatState.Uninitialized;
        protected string m_AuthToken = string.Empty;

        protected ChatApiListener m_ChatAPIListener = null;
        protected Dictionary<string, ChatChannelListener> m_Channels = new Dictionary<string, ChatChannelListener>();

        protected uint m_MessageHistorySize = 128;
        protected EmoticonMode m_ActiveEmoticonMode = EmoticonMode.None;

        protected Dictionary<string, ChatUserEmoticonSets> m_UserEmoticonSets = new Dictionary<string, ChatUserEmoticonSets>();
        protected Dictionary<uint, ChatEmoticonSetData> m_EmoticonSets = new Dictionary<uint, ChatEmoticonSetData>();

        #endregion

        protected abstract class ControllerAccess
        {
            protected ChatController m_ChatController;

            protected ControllerAccess(ChatController controller)
            {
                m_ChatController = controller;
            }

            protected EmoticonMode ActiveEmoticonMode
            {
                get { return m_ChatController.m_ActiveEmoticonMode; }
            }

            protected uint MessageHistorySize
            {
                get { return m_ChatController.MessageHistorySize; }
            }

            protected int MessageFlushInterval
            {
                get { return m_ChatController.MessageFlushInterval; }
            }

            protected int UserListUpdateInterval
            {
                get { return m_ChatController.UserListUpdateInterval; }
            }

            protected string UserName
            {
                get { return m_ChatController.UserName; }
            }
            
            protected string AuthToken
            {
                get { return m_ChatController.AuthToken; }
            }

            protected Dictionary<string, ChatChannelListener> Channels
            {
                get { return m_ChatController.m_Channels; }
            }

            protected ChatApi Api
            {
                get { return m_ChatController.m_ChatApi; }
            }

            protected void CheckError(ErrorCode err)
            {
                m_ChatController.CheckError(err);
            }

            protected void ReportError(string err)
            {
                m_ChatController.ReportError(err);
            }

            protected void ReportWarning(string err)
            {
                m_ChatController.ReportWarning(err);
            }

            protected void FireMessagesCleared(string channelName, string username)
            {
                try
                {
                    if (m_ChatController.MessagesCleared != null)
                    {
                        m_ChatController.MessagesCleared(channelName, username);
                    }
                }
                catch (Exception x)
                {
                    ReportError(string.Format("Error clearing chat messages: {0}", x.ToString()));
                }
            }

            protected void FireBadgeDataAvailable(string channelName)
            {
                try
                {
                    if (m_ChatController.BadgeDataAvailable != null)
                    {
                        m_ChatController.BadgeDataAvailable(channelName);
                    }
                }
                catch (Exception x)
                {
                    ReportError(x.ToString());
                }
            }

            protected void FireUsersChanged(string channelName, ChatUserInfo[] joinList, ChatUserInfo[] leaveList, ChatUserInfo[] userInfoList)
            {
                try
                {
                    if (m_ChatController.UsersChanged != null)
                    {
                        m_ChatController.UsersChanged(channelName, joinList, leaveList, userInfoList);
                    }
                }
                catch (Exception x)
                {
                    ReportError(x.ToString());
                }
            }

            protected void FireMessagesReceived(string channelName, ChatMessage[] messages)
            {
                try
                {
                    if (m_ChatController.MessagesReceived != null)
                    {
                        m_ChatController.MessagesReceived(channelName, messages);
                    }
                }
                catch (Exception x)
                {
                    ReportError(x.ToString());
                }
            }

            protected void FireHostTargetChanged(string channelName, string targetChannel, int numViewers)
            {
                try
                {
                    if (m_ChatController.HostTargetChanged != null)
                    {
                        m_ChatController.HostTargetChanged(channelName, targetChannel, numViewers);
                    }
                }
                catch (Exception x)
                {
                    ReportError(x.ToString());
                }
            }

            protected void FireNoticeReceived(string channelName, string id, Dictionary<string, string> parameters)
            {
                try
                {
                    if (m_ChatController.NoticeReceived != null)
                    {
                        m_ChatController.NoticeReceived(channelName, id, parameters);
                    }
                }
                catch (Exception x)
                {
                    ReportError(x.ToString());
                }
            }

            protected void FireSetBroadcasterLanguageResultReceived(string channelName, ErrorCode ec)
            {
                try
                {
                    if (m_ChatController.SetBroadcasterLanguageResultReceived != null)
                    {
                        m_ChatController.SetBroadcasterLanguageResultReceived(channelName, ec);
                    }
                }
                catch (Exception x)
                {
                    ReportError(x.ToString());
                }
            }
        }

        protected class ChatApiListener : ControllerAccess, IChatApiListener
        {
            public ChatApiListener(ChatController controller) : base(controller)
            {
            }

            #region IChatAPIListener Implementation

            void IChatApiListener.ChatInitializationCallback(ErrorCode result)
            {
                if (Error.Succeeded(result))
                {
                    Api.SetMessageFlushInterval(this.MessageFlushInterval);
                    Api.SetUserListUpdateInterval(this.UserListUpdateInterval);

                    m_ChatController.SetChatState(ChatState.Initialized, result);
                }
                else
                {
                    m_ChatController.SetChatState(ChatState.Uninitialized, result);
                }

        	    try
        	    {
		            if (m_ChatController.InitializationComplete != null)
		            {
		        	    m_ChatController.InitializationComplete(result);
		            }
        	    }
        	    catch (Exception x)
        	    {
        		    m_ChatController.ReportError(x.ToString());
        	    }
            }

            void IChatApiListener.ChatShutdownCallback(ErrorCode result)
            {
                if (Error.Succeeded(result))
                {
                    m_ChatController.m_UserEmoticonSets.Clear();

                    m_ChatController.SetChatState(ChatState.Uninitialized, result);
                }
                else
                {
            		// if shutdown fails the state will probably be messed up but this should never happen
                    m_ChatController.SetChatState(ChatState.Initialized, result);
        		
            		ReportError(String.Format("Error shutting down Twith chat: {0}", result));
                }

        	    try
        	    {
		            if (m_ChatController.ShutdownComplete != null)
		            {
		        	    m_ChatController.ShutdownComplete(result);
		            }
        	    }
        	    catch (Exception x)
        	    {
        		    m_ChatController.ReportError(x.ToString());
        	    }
            }

            void IChatApiListener.ChatUserEmoticonSetsChangedCallback(ChatUserEmoticonSets emoticonSets)
            {
                try
                {
                    m_ChatController.m_UserEmoticonSets[emoticonSets.Username] = emoticonSets;

                    if (m_ChatController.LocalUserEmoticonSetsChanged != null)
                    {
                        m_ChatController.LocalUserEmoticonSetsChanged(emoticonSets.Username);
                    }
                }
                catch (Exception x)
                {
                    m_ChatController.ReportError(x.ToString());
                }
            }

            void IChatApiListener.ChatEmoticonSetDataCallback(ChatEmoticonSetData emoticonSet)
            {
                try
                {
                    m_ChatController.m_EmoticonSets[emoticonSet.EmoticonSetId] = emoticonSet;

                    if (m_ChatController.EmoticonSetDataAvailable != null)
                    {
                        m_ChatController.EmoticonSetDataAvailable(emoticonSet);
                    }
                }
                catch (Exception x)
                {
                    m_ChatController.ReportError(x.ToString());
                }
            }
		    
            void IChatApiListener.ChatUserBlockChangeCallback(string userName, string blockUsername, bool block, ErrorCode ec)
            {
                try
                {
                    if (m_ChatController.UserBlockChange != null)
                    {
                        m_ChatController.UserBlockChange(userName, blockUsername, block, ec);
                    }
                }
                catch (Exception x)
                {
                    m_ChatController.ReportError(x.ToString());
                }
            }

            #endregion
        }

        protected class ChatChannelListener : ControllerAccess, IChatChannelListener
        {
            protected string m_ChannelName = null;
            protected bool m_Anonymous = false;
            protected ChannelState m_ChannelState = ChannelState.Disconnected;
            protected ChatChannelInfo m_ChannelInfo = new ChatChannelInfo();
            protected ChatUserInfo m_LocalUserInfo = new ChatUserInfo();

            protected List<ChatUserInfo> m_ChannelUsers = new List<ChatUserInfo>();
            protected LinkedList<ChatMessage> m_Messages = new LinkedList<ChatMessage>();

            protected ChatBadgeData m_BadgeData = null;

            public ChatChannelListener(ChatController controller, string channelName) : 
                base(controller)
            {
                m_ChannelName = channelName;
            }

            #region Properties

            public ChannelState ChannelState
            {
                get { return m_ChannelState; }
            }

            public bool IsConnected
            {
                get { return m_ChannelState == ChannelState.Connected; }
            }

            public ChatChannelInfo ChannelInfo
            {
                get { return m_ChannelInfo; }
            }

            public ChatUserInfo LocalUserInfo
            {
                get { return m_LocalUserInfo; }
            }

            public bool IsAnonymous
            {
                get { return m_Anonymous; }
            }

            public LinkedList<ChatMessage>.Enumerator Messages
            {
                get { return m_Messages.GetEnumerator(); }
            }

            public ChatBadgeData BadgeData
            {
                get { return m_BadgeData; }
            }

            #endregion

            public virtual bool Connect(bool anonymous)
            {
                m_Anonymous = anonymous;

                ErrorCode ret = ErrorCode.TTV_EC_SUCCESS;

                // connect to the channel
                if (anonymous)
                {
                    ret = Api.Connect(m_ChannelName, "", this);
                }
                else
                {
                    ret = Api.Connect(m_ChannelName, UserName, this);
                }

                if (Error.Failed(ret))
                {
                    String err = Error.GetString(ret);
                    ReportError(String.Format("Error connecting: {0}", err));

                    return false;
                }
                else
                {
                    return true;
                }
            }

            public bool Disconnect()
            {
                switch (m_ChannelState)
                {
                    case ChannelState.Connected:
                    case ChannelState.Connecting:
                    {
                        // kick off an async disconnect
                        ErrorCode ret = Api.Disconnect(m_ChannelName);
                        if (Error.Failed(ret))
                        {
                            String err = Error.GetString(ret);
                            ReportError(String.Format("Error disconnecting: {0}", err));

                            return false;
                        }

                        return true;
                    }
                    default:
                    {
                        return false;
                    }
                }
            }

            protected void SetChannelState(ChannelState state, ErrorCode result)
            {
                if (state == m_ChannelState)
                {
                    return;
                }

                m_ChannelState = state;

                FireChannelStateChanged(m_ChannelName, state, result);
            }

            public void ClearMessages()
            {
                ClearMessages(null);
            }

            public void ClearMessages(string username)
            {
                if (string.IsNullOrEmpty(username))
                {
                    m_Messages.Clear();
                }
                else
                {
                    if (m_Messages.Count > 0)
                    {
                        LinkedListNode<ChatMessage> node = m_Messages.First;
                        while (node != null)
                        {
                            if (node.Value.UserName == username)
                            {
                                LinkedListNode<ChatMessage> d = node;
                                node = node.Next;
                                m_Messages.Remove(d);
                            }
                            else
                            {
                                node = node.Next;
                            }
                        }
                    }
                }

                FireMessagesCleared(m_ChannelName, username);
            }

            public bool SendChatMessage(string message)
            {
                if (m_ChannelState != ChannelState.Connected)
                {
                    return false;
                }

                ErrorCode ret = Api.SendMessage(m_ChannelName, message);
                if (Error.Failed(ret))
                {
                    String err = Error.GetString(ret);
                    ReportError(String.Format("Error sending chat message: {0}", err));

                    return false;
                }

                return true;
            }

            public bool SetBroadcasterLanguageChatEnabled(bool enabled)
            {
                ErrorCode ec = Api.SetBroadcasterLanguageChatEnabled(m_ChannelName, enabled);
                return Error.Succeeded(ec);
            }

            public bool OptInToBroadcasterLanguageChat(string language)
            {
                ErrorCode ec = Api.OptInToBroadcasterLanguageChat(m_ChannelName, language);
                return Error.Succeeded(ec);
            }

            #region Badge Handling

            internal virtual void SetupBadgeData()
            {
                if (m_BadgeData != null)
                {
                    return;
                }

                ErrorCode ec = Api.GetBadgeData(m_ChannelName, out m_BadgeData);

                if (Error.Succeeded(ec))
                {
                    FireBadgeDataAvailable(m_ChannelName);
                }
                else
                {
                    ReportError("Error preparing badge data: " + Error.GetString(ec));
                }
            }

            #endregion

            protected void FireChannelStateChanged(String channelName, ChannelState state, ErrorCode result)
            {
                try
                {
                    if (m_ChatController.ChannelStateChanged != null)
                    {
                        m_ChatController.ChannelStateChanged(channelName, state, result);
                    }
                }
                catch (Exception x)
                {
                    ReportError(x.ToString());
                }
            }

            protected void FireChannelInfoChanged(String channelName, ChatChannelInfo info)
            {
                try
                {
                    if (m_ChatController.ChannelInfoChanged != null)
                    {
                        m_ChatController.ChannelInfoChanged(channelName, info);
                    }
                }
                catch (Exception x)
                {
                    ReportError(x.ToString());
                }
            }

            protected void FireChannelLocalUserChanged(String channelName, ChatUserInfo userInfo)
            {
                try
                {
                    if (m_ChatController.LocalUserChanged != null)
                    {
                        m_ChatController.LocalUserChanged(channelName, userInfo);
                    }
                }
                catch (Exception x)
                {
                    ReportError(x.ToString());
                }
            }

            #region IChatChannelListener Implementation

            void IChatChannelListener.ChatChannelStateChangedCallback(string channelName, TTV_ChatChannelState state, ErrorCode result)
            {
                switch (state)
                {
                    case TTV_ChatChannelState.TTV_CHAT_CHANNEL_STATE_DISCONNECTED:
                    {
                        Channels.Remove(m_ChannelName);
                        SetChannelState(ChannelState.Disconnected, result);
                        break;
                    }
                    case TTV_ChatChannelState.TTV_CHAT_CHANNEL_STATE_CONNECTING:
                    {
                        SetChannelState(ChannelState.Connecting, result);
                        break;
                    }
                    case TTV_ChatChannelState.TTV_CHAT_CHANNEL_STATE_CONNECTED:
                    {
                        SetChannelState(ChannelState.Connected, result);
                        break;
                    }
                    case TTV_ChatChannelState.TTV_CHAT_CHANNEL_STATE_DISCONNECTING:
                    {
                        SetChannelState(ChannelState.Disconnecting, result);
                        break;
                    }
                }
            }

            void IChatChannelListener.ChatChannelInfoChangedCallback(string channelName, ChatChannelInfo info)
            {
                m_ChannelInfo = info;

                FireChannelInfoChanged(channelName, info);
            }

            void IChatChannelListener.ChatChannelLocalUserChangedCallback(string channelName, ChatUserInfo userInfo)
            {
                m_LocalUserInfo = userInfo;

                FireChannelLocalUserChanged(channelName, userInfo);
            }

            void IChatChannelListener.ChatChannelUserChangeCallback(string channelName, ChatUserInfo[] joinList, ChatUserInfo[] leaveList, ChatUserInfo[] userInfoList)
            {
                for (int i = 0; i < leaveList.Length; ++i)
                {
                    int index = m_ChannelUsers.IndexOf(leaveList[i]);
                    if (index >= 0)
                    {
                        m_ChannelUsers.RemoveAt(index);
                    }
                }

                for (int i = 0; i < userInfoList.Length; ++i)
                {
                    // this will find the existing user with the same name
                    int index = m_ChannelUsers.IndexOf(userInfoList[i]);
                    if (index >= 0)
                    {
                        m_ChannelUsers.RemoveAt(index);
                    }

                    m_ChannelUsers.Add(userInfoList[i]);
                }

                for (int i = 0; i < joinList.Length; ++i)
                {
                    m_ChannelUsers.Add(joinList[i]);
                }

                FireUsersChanged(m_ChannelName, joinList, leaveList, userInfoList);
            }

            void IChatChannelListener.ChatChannelMessageCallback(string channelName, ChatMessage[] messageList)
            {
                for (int i = 0; i < messageList.Length; ++i)
                {
                    m_Messages.AddLast(messageList[i]);
                }

                FireMessagesReceived(m_ChannelName, messageList);

                // cap the number of messages cached
                while (m_Messages.Count > MessageHistorySize)
                {
                    m_Messages.RemoveFirst();
                }
            }

            void IChatChannelListener.ChatChannelClearCallback(string channelName, string username)
            {
                ClearMessages(username);
            }

            void IChatChannelListener.ChatChannelBadgeDataDownloadCallback(string channelName, ErrorCode error)
            {
                // grab the badge data
                if (Error.Succeeded(error))
                {
                    SetupBadgeData();
                }
            }

            void IChatChannelListener.ChatChannelHostTargetChangedCallback(string channelName, string targetChannel, int numViewers)
            {
                FireHostTargetChanged(channelName, targetChannel, numViewers);
            }

            void IChatChannelListener.ChatChannelNoticeCallback(string channelName, string id, Dictionary<string, string> parameters)
            {
                FireNoticeReceived(channelName, id, parameters);
            }
            
            void IChatChannelListener.ChatChannelSetBroadcasterLanguageCallback(string channelName, ErrorCode ec)
            {
                FireSetBroadcasterLanguageResultReceived(channelName, ec);
            }

            #endregion
        }

        #region Properties

        /// <summary>
        /// Whether or not the controller has been initialized.
        /// </summary>
        public bool IsInitialized
        {
            get { return m_ChatState == ChatState.Initialized; }
        }

        /// <summary>
        /// The username currently logged in.
        /// </summary>
        public string UserName
        {
            get { return m_UserName; }
        }

        /// <summary>
        /// The oauth token.
        /// </summary>
        public string AuthToken
        {
            get { return m_AuthToken; }
        }

        /// <summary>
        /// The Twitch client ID assigned to your application.
        /// </summary>
        public abstract string ClientId
        {
            get;
            set;
        }

        /// <summary>
        /// The secret code gotten from the Twitch site for the client id.
        /// </summary>
        public abstract string ClientSecret
        {
            get;
            set;
        }

        /// <summary>
        /// The emoticon parsing mode for chat messages.  This must be set before initializing.  
        /// </summary>
        public abstract EmoticonMode EmoticonParsingMode
        {
            get;
            set;
        }

        /// <summary>
        /// The number of milliseconds between message events.
        /// </summary>
        public virtual int MessageFlushInterval
        {
            get { return 0; }
            set
            {
                if (m_ChatState == ChatState.Initialized)
                {
                    m_ChatApi.SetMessageFlushInterval(value);
                }
            }
        }

        /// <summary>
        /// The number of milliseconds between events for user joins, leaves and changes in channels.
        /// </summary>
        public virtual int UserListUpdateInterval
        {
            get { return 0; }
            set
            {
                if (m_ChatState == ChatState.Initialized)
                {
                    m_ChatApi.SetUserListUpdateInterval(value);
                }
            }
        }

        /// <summary>
        /// Forces the user list of the named channel to be refreshed and join, leave and change events to be fired.  This does not normally need to be used because user list updates are done periodically.
        /// </summary>
        public void ForceUserListUpdate(string channelName)
        {
            if (m_ChatState == ChatState.Initialized)
            {
                m_ChatApi.ForceUserListUpdate(channelName);
            }
        }

        /// <summary>
        /// The maximum number of messages to be kept in the chat history.
        /// </summary>
        public uint MessageHistorySize
        {
            get { return m_MessageHistorySize; }
            set { m_MessageHistorySize = value; }
        }

        /// <summary>
        /// The current state of the ChatController.
        /// </summary>
        public ChatState CurrentState
        {
            get { return m_ChatState; }
        }

        /// <summary>
        /// Retrieves the channel info for the given channel.
        /// </summary>
        public ChatChannelInfo GetChannelInfo(string channelName)
        {
            if (!m_Channels.ContainsKey(channelName))
            {
                return null;
            }

            ChatChannelListener channel = m_Channels[channelName];
            return channel.ChannelInfo.Copy();
        }

        /// <summary>
        /// Retrieves the local user info for the given channel.
        /// </summary>
        /// <param name="channelName"></param>
        /// <returns></returns>
        public ChatUserInfo GetLocalUserInfo(string channelName)
        {
            if (!m_Channels.ContainsKey(channelName))
            {
                return null;
            }

            ChatChannelListener channel = m_Channels[channelName];
            return channel.LocalUserInfo.Copy();
        }

        /// <summary>
        /// Retrieves the local user emoticon sets.
        /// </summary>
        /// <param name="username"></param>
        /// <returns></returns>
        public ChatUserEmoticonSets GetLocalUserEmoticonSets(string username)
        {
            if (m_UserEmoticonSets.ContainsKey(username))
            {
                return m_UserEmoticonSets[username];
            }
            else
            {
                return null;
            }
        }

        /// <summary>
        /// Whether or not currently connected to the channel.
        /// </summary>
        public bool IsConnected(string channelName)
        {
            channelName = channelName.ToLower();
            if (!m_Channels.ContainsKey(channelName))
            {
                return false;
            }

            ChatChannelListener channel = m_Channels[channelName];
            return channel.IsConnected;
        }

        /// <summary>
        /// Whether or not currently connected to the channel.
        /// </summary>
        public bool IsAnonymous(string channelName)
        {
            channelName = channelName.ToLower();
            if (!m_Channels.ContainsKey(channelName))
            {
                return false;
            }

            ChatChannelListener channel = m_Channels[channelName];
            return channel.IsAnonymous;
        }

        /// <summary>
        /// An iterator for the chat messages from oldest to newest.
        /// </summary>
        public LinkedList<ChatMessage>.Enumerator GetMessages(string channelName)
        {
            channelName = channelName.ToLower();
            if (!m_Channels.ContainsKey(channelName))
            {
                ReportError("Unknown channel: " + channelName);
                return new LinkedList<ChatMessage>().GetEnumerator();
            }

            ChatChannelListener channel = m_Channels[channelName];
            return channel.Messages;
        }

        /// <summary>
        /// Retrieves the badge data that can be used to render icons.
        /// </summary>
        public ChatBadgeData GetBadgeData(string channelName)
        {
            channelName = channelName.ToLower();
            if (!m_Channels.ContainsKey(channelName))
            {
                ReportError("Unknown channel: " + channelName);
                return null;
            }

            ChatChannelListener channel = m_Channels[channelName];
            return channel.BadgeData;
        }

        #endregion

        protected ChatController()
        {
            m_ChatAPIListener = new ChatApiListener(this);
        }

        public virtual bool Initialize()
        {
            if (m_ChatState != ChatState.Uninitialized)
            {
                return false;
            }

            SetChatState(ChatState.Initializing, ErrorCode.TTV_EC_SUCCESS);

            ErrorCode ret;
            if (Library.Instance == null || !Library.Instance.ChatInitialized)
            {
                ret = ErrorCode.TTV_EC_NOT_INITIALIZED;

                SetChatState(ChatState.Uninitialized, ret);

                String err = Error.GetString(ret);
                ReportError(String.Format("Error initializing Twitch sdk: {0}", err));

                return false;
            }

            // initialize chat
            m_ActiveEmoticonMode = this.EmoticonParsingMode;

            TTV_ChatTokenizationOption tokenizationOptions = TTV_ChatTokenizationOption.TTV_CHAT_TOKENIZATION_OPTION_NONE;
            switch (m_ActiveEmoticonMode)
            {
                case EmoticonMode.None:
                    tokenizationOptions = TTV_ChatTokenizationOption.TTV_CHAT_TOKENIZATION_OPTION_NONE;
                    break;
                case EmoticonMode.Url:
                    tokenizationOptions = TTV_ChatTokenizationOption.TTV_CHAT_TOKENIZATION_OPTION_ALL;
                    break;
            }

            // kick off the async init
            ret = m_ChatApi.Initialize(tokenizationOptions, m_ChatAPIListener);
            if (Error.Failed(ret))
            {
                SetChatState(ChatState.Uninitialized, ret);

                String err = Error.GetString(ret);
                ReportError(String.Format("Error initializing Twitch chat: %s", err));

                return false;
            }
            else
            {
                SetChatState(ChatState.Initialized, ErrorCode.TTV_EC_SUCCESS);
                return true;
            }
        }

        public virtual bool LogIn(string userName, string oauthToken)
        {
            if (string.IsNullOrEmpty(userName) ||
                string.IsNullOrEmpty(oauthToken))
            {
                return false;
            }

            string previousUser = m_UserName;

            ErrorCode ec = m_ChatApi.LogIn(userName, oauthToken);
            if (Error.Failed(ec))
            {
                return false;
            }

            m_UserName = userName;
            m_AuthToken = oauthToken;

            if (!string.IsNullOrEmpty(previousUser) && previousUser != userName)
            {
                m_ChatApi.LogOut(previousUser);
            }

            return true;
        }

        public virtual bool LogOut()
        {
            bool ret = true;

            if (!string.IsNullOrEmpty(m_UserName))
            {
                ErrorCode ec = m_ChatApi.LogOut(m_UserName);
                ret = Error.Succeeded(ec);

                if (ret)
                {
                    m_UserName = string.Empty;
                    m_AuthToken = string.Empty;
                }
            }

            return ret;
        }

        /// <summary>
        /// Connects to the given channel.  The actual result of the connection attempt will be returned in the Connected / Disconnected event.
        /// </summary>
        /// <param name="channelName">The name of the channel.</param>
        /// <returns>Whether or not the request was successful.</returns>
        public virtual bool Connect(string channelName)
        {
            return Connect(channelName, false);
        }

        /// <summary>
        /// Connects to the given channel anonymously.  The actual result of the connection attempt will be returned in the Connected / Disconnected event.
        /// </summary>
        /// <param name="channelName">The name of the channel.</param>
        /// <returns>Whether or not the request was valid.</returns>
        public virtual bool ConnectAnonymous(string channelName)
        {
            return Connect(channelName, true);
        }

        protected virtual bool Connect(string channelName, bool anonymous)
        {
            channelName = channelName.ToLower();

            if (m_ChatState != ChatState.Initialized)
            {
                return false;
            }

            if (string.IsNullOrEmpty(channelName))
            {
                return false;
            }

            ChatChannelListener channel = null;

            if (m_Channels.ContainsKey(channelName))
            {
                channel = m_Channels[channelName];
            }
            else
            {
                channel = new ChatChannelListener(this, channelName);
                m_Channels[channelName] = channel;
            }

            bool result = channel.Connect(anonymous);

            if (!result)
            {
                m_Channels.Remove(channelName);
            }

            return result;
        }

        /// <summary>
        /// Disconnects from the channel.  The result of the attempt will be returned in a Disconnected event.
        /// </summary>
        /// <returns>Whether or not the disconnect attempt was valid.</returns>
        public virtual bool Disconnect(string channelName)
        {
            channelName = channelName.ToLower();

            if (m_ChatState != ChatState.Initialized)
            {
                return false;
            }

            if (!m_Channels.ContainsKey(channelName))
            {
                ReportError("Not in channel: " + channelName);
                return false;
            }

            ChatChannelListener channel = m_Channels[channelName];
            return channel.Disconnect();
        }

        public virtual bool Shutdown()
        {
            if (m_ChatState != ChatState.Initialized)
            {
                return false;
            }

            // shutdown asynchronously
            ErrorCode ret = m_ChatApi.Shutdown();
            if (Error.Failed(ret))
            {
                String err = Error.GetString(ret);
                ReportError(String.Format("Error shutting down chat: {0}", err));

                return false;
            }

            SetChatState(ChatState.ShuttingDown, ErrorCode.TTV_EC_SUCCESS);

            return true;
        }

        /// <summary>
        /// Ensures the controller is fully shutdown before returning.  This may fire callbacks to listeners during the shutdown.
        /// </summary>
        public void ForceSyncShutdown()
        {
            // force a low-level shutdown
            if (this.CurrentState != Twitch.Chat.ChatController.ChatState.Uninitialized)
            {
                this.Shutdown();

                // wait for the shutdown to finish
                if (this.CurrentState == Twitch.Chat.ChatController.ChatState.ShuttingDown)
                {
                    while (this.CurrentState != Twitch.Chat.ChatController.ChatState.Uninitialized)
                    {
                        try
                        {
                            System.Threading.Thread.Sleep(200);
                            this.Update();
                        }
                        catch
                        {
                        }
                    }
                }
            }
        }

        /// <summary>
        /// Periodically updates the internal state of the controller.
        /// </summary>
        public virtual void Update()
        {
            // for stress testing to make sure memory is being passed around properly
            //GC.Collect(); 

            if (m_ChatState == ChatState.Uninitialized)
            {
                return;
            }

	        ErrorCode ret = m_ChatApi.FlushEvents();
            if (Error.Failed(ret))
            {
                string err = Error.GetString(ret);
                ReportError(string.Format("Error flushing chat events: {0}, in state {1}", err, m_ChatState.ToString()));
            }
        }

        /// <summary>
        /// Sends a chat message to the channel.
        /// </summary>
        /// <param name="message">The message to send.</param>
        /// <returns>Whether or not the attempt was valid.</returns>
        public virtual bool SendChatMessage(string channelName, string message)
        {
            channelName = channelName.ToLower();

            if (m_ChatState != ChatState.Initialized)
            {
                return false;
            }

            if (!m_Channels.ContainsKey(channelName))
            {
                ReportError("Not in channel: " + channelName);
                return false;
            }

            ChatChannelListener channel = m_Channels[channelName];
            return channel.SendChatMessage(message);
        }

        /// <summary>
        /// Clears the chat message history for all users.
        /// </summary>
        public virtual void ClearMessages(string channelName)
        {
            channelName = channelName.ToLower();

            if (!m_Channels.ContainsKey(channelName))
            {
                ReportError("Not in channel: " + channelName);
                return;
            }

            ChatChannelListener channel = m_Channels[channelName];
            channel.ClearMessages();
        }

        /// <summary>
        /// Clears the chat message history for the given user only.
        /// </summary>
        public virtual void ClearMessages(string channelName, string username)
        {
            channelName = channelName.ToLower();

            if (!m_Channels.ContainsKey(channelName))
            {
                ReportError("Not in channel: " + channelName);
                return;
            }

            ChatChannelListener channel = m_Channels[channelName];
            channel.ClearMessages(username);
        }

        internal virtual void SetChatState(ChatState state, ErrorCode result)
        {
            if (state == m_ChatState)
            {
                return;
            }

            m_ChatState = state;

            try
            {
                if (ChatStateChanged != null)
                {
                    this.ChatStateChanged(state, result);
                }
            }
            catch (Exception x)
            {
                ReportError(x.ToString());
            }
        }

        public bool SetBroadcasterLanguageChatEnabled(string channelName, bool enabled)
        {
            if (m_ChatState != ChatState.Initialized)
            {
                return false;
            }

            if (!m_Channels.ContainsKey(channelName))
            {
                ReportError("Not in channel: " + channelName);
                return false;
            }

            ChatChannelListener channel = m_Channels[channelName];
            return channel.SetBroadcasterLanguageChatEnabled(enabled);
        }

        public bool OptInToBroadcasterLanguageChat(string channelName, string language)
        {
            if (m_ChatState != ChatState.Initialized)
            {
                return false;
            }

            if (!m_Channels.ContainsKey(channelName))
            {
                ReportError("Not in channel: " + channelName);
                return false;
            }

            ChatChannelListener channel = m_Channels[channelName];
            return channel.OptInToBroadcasterLanguageChat(language);
        }

        public bool SetLocalLanguage(string language)
        {
            if (m_ChatState != ChatState.Initialized)
            {
                return false;
            }

            ErrorCode ec = m_ChatApi.SetLocalLanguage(language);
            return Error.Succeeded(ec);
        }


        #region Error Handling

        protected virtual void CheckError(ErrorCode err)
        {
        }

        protected virtual void ReportError(string err)
        {
        }

        protected virtual void ReportWarning(string err)
        {
        }

        #endregion
    }
}
