﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using Twitch;
using UnityEngine;
using ErrorCode = Twitch.ErrorCode;

namespace Twitch.Chat
{
    public class UnityChatController : ChatController
    {
        #region Memeber Variables

        [SerializeField]
        protected string m_ClientId = "";
        [SerializeField]
        protected string m_ClientSecret = "";
        [SerializeField]
        protected EmoticonMode m_EmoticonMode = EmoticonMode.None;
        [SerializeField]
        protected int m_MessageFlushInterval = 500;
        [SerializeField]
        protected int m_UserListUpdateInterval = 10 * 60 * 1000;

        private UnityIosChatApi m_UnityIosChatApi = null;

        #endregion

        #region Properties

        public override string ClientId
        {
            get { return m_ClientId; }
            set { m_ClientId = value; }
        }

        public override string ClientSecret
        {
            get { return m_ClientSecret; }
            set { m_ClientSecret = value; }
        }

        public override EmoticonMode EmoticonParsingMode
        {
            get { return m_EmoticonMode; }
            set { m_EmoticonMode = value; }
        }

        public override int MessageFlushInterval
        {
            get { return m_MessageFlushInterval; }
            set
            {
                value = Math.Min(MAX_INTERVAL_MS, Math.Max(value, MIN_INTERVAL_MS));

                m_MessageFlushInterval = value;
                base.MessageFlushInterval = value;
            }
        }

        public override int UserListUpdateInterval
        {
            get { return m_UserListUpdateInterval; }
            set
            {
                value = Math.Min(MAX_INTERVAL_MS, Math.Max(value, MIN_INTERVAL_MS));

                m_UserListUpdateInterval = value;
                base.UserListUpdateInterval = value;
            }
        }

        #endregion

        #region Unity Overrides

        protected void Awake()
        {
            // force the twitch libraries to be loaded
            Twitch.Broadcast.UnityBroadcastController.LoadTwitchLibraries();

            m_CoreApi = CoreApi.Instance;

            if (Application.platform == RuntimePlatform.WindowsEditor ||
                Application.platform == RuntimePlatform.WindowsPlayer)
            {
                if (m_CoreApi == null)
                {
                    m_CoreApi = new StandardCoreApi();
                }

                m_ChatApi = new StandardChatApi();
            }
            else if (Application.platform == RuntimePlatform.OSXEditor ||
                     Application.platform == RuntimePlatform.OSXPlayer)
            {
                if (m_CoreApi == null)
                {
                    m_CoreApi = new StandardCoreApi();
                }

                m_ChatApi = new StandardChatApi();
            }
            else if (Application.platform == RuntimePlatform.IPhonePlayer)
            {
                if (m_CoreApi == null)
                {
                    m_CoreApi = new UnityIosCoreApi();
                }

                m_UnityIosChatApi = new UnityIosChatApi();
                m_ChatApi = m_UnityIosChatApi;
            }
            else if (Application.platform == RuntimePlatform.Android)
            {
                // TODO: not supported yet
            }
            else
            {
                // TODO: not supported yet
            }
        }

        protected void OnDestroy()
        {
            // we can't do this on iOS because callbacks are not called synchronously from native
            if (Application.platform == RuntimePlatform.WindowsEditor ||
                Application.platform == RuntimePlatform.WindowsPlayer ||
                Application.platform == RuntimePlatform.OSXEditor ||
                Application.platform == RuntimePlatform.OSXPlayer)
            {
                ForceSyncShutdown();
            }
        }

        public override void Update()
        {
            // for awareness that the Unity Update hook is actually being called
            base.Update();
        }

        #endregion

        #region Error Handling

        protected override void CheckError(Twitch.ErrorCode err)
        {
            if (err != Twitch.ErrorCode.TTV_EC_SUCCESS)
            {
                Debug.LogError(err.ToString());
            }
        }

        protected override void ReportError(string err)
        {
            Debug.LogError(err);
        }

        protected override void ReportWarning(string err)
        {
            Debug.LogError(err);
        }

        #endregion

        #region ChatAPI Callback Proxy Handlers

        private void ProxyInitializationCallback(string json)
        {
            m_UnityIosChatApi.ChatInitializationCallback(json);
        }

        private void ProxyShutdownCallback(string json)
        {
            m_UnityIosChatApi.ChatShutdownCallback(json);
        }

        private void ProxyUserEmoticonSetsChangedCallback(string json)
        {
            m_UnityIosChatApi.ChatUserEmoticonSetsChangedCallback(json);
        }

        private void ProxyEmoticonSetDataCallback(string json)
        {
            m_UnityIosChatApi.ChatEmoticonSetDataCallback(json);
        }
        
        public void ProxyChannelStateChangedCallback(string json)
        {
            m_UnityIosChatApi.ChatChannelStateChangedCallback(json);
        }

        public void ProxyChannelInfoChangedCallback(string json)
        {
            m_UnityIosChatApi.ChatChannelInfoChangedCallback(json);
        }

        public void ProxyChannelLocalUserChangedCallback(string json)
        {
            m_UnityIosChatApi.ChatChannelLocalUserChangedCallback(json);
        }

        public void ProxyChannelUserChangeCallback(string json)
        {
            m_UnityIosChatApi.ChatChannelUserChangeCallback(json);
        }

        public void ProxyChannelRawMessageCallback(string json)
        {
            m_UnityIosChatApi.ChatChannelRawMessageCallback(json);
        }

        public void ProxyChannelTokenizedMessageCallback(string json)
        {
            m_UnityIosChatApi.ChatChannelTokenizedMessageCallback(json);
        }

        public void ProxyChannelClearCallback(string json)
        {
            m_UnityIosChatApi.ChatClearCallback(json);
        }

        public void ProxyChannelBadgeDataDownloadCallback(string json)
        {
            m_UnityIosChatApi.ChatBadgeDataDownloadCallback(json);
        }

        public void ProxyChannelHostTargetChangedCallback(string json)
        {
            m_UnityIosChatApi.ChatChannelHostTargetChangedCallback(json);
        }

        public void ProxyChannelNoticeCallback(string json)
        {
            m_UnityIosChatApi.ChatChannelNoticeCallback(json);
        }

        #endregion
    }
}
