#include "chattester.h"
#include "twitchchat.h"
#include <string>
#include <assert.h>
#include <vector>

#ifdef TTV_PLATFORM_WINDOWS
#	include <Windows.h>
#endif

#pragma warning (push)
#pragma warning (disable:4996)

static bool sConnected = false;


void FormatModesAndSubscriptions(const TTV_ChatUserInfo* pUserInfo, char buffer[256])
{
	sprintf(buffer, "%s %s %s %s %s %s %s %s", 
		pUserInfo->modes == TTV_CHAT_USERMODE_VIEWER ? "VIEWER" : "",
		pUserInfo->modes & TTV_CHAT_USERMODE_MODERATOR ? "MODERATOR" : "",
		pUserInfo->modes & TTV_CHAT_USERMODE_BROADCASTER ? "BROADCASTER" : "",
		pUserInfo->modes & TTV_CHAT_USERMODE_ADMINISTRATOR ? "ADMINSTRATOR" : "",
		pUserInfo->modes & TTV_CHAT_USERMODE_STAFF ? "STAFF" : "",
		pUserInfo->modes & TTV_CHAT_USERMODE_BANNED ? "BANNED" : "",

		pUserInfo->subscriptions & TTV_CHAT_USERSUB_SUBSCRIBER ? "SUBSCRIBER" : "",
		pUserInfo->subscriptions & TTV_CHAT_USERSUB_TURBO ? "TURBO" : ""
		);
}

void ChatChannelStateChangedCallback(TTV_ChatChannelState state, TTV_ErrorCode ec, void* /*userdata*/)
{
	printf("Received ChatChannelStateChangedCallback: state:%d, error:%d\n", state, ec);

	sConnected = (state == TTV_CHAT_CHANNEL_STATE_CONNECTED);
}

void ChatChannelInfoChangedCallback(const TTV_ChatChannelInfo* /*channelInfo*/, void* /*userdata*/)
{
	printf("ChatChannelInfoChangedCallback\n");
}

void ChatChannelLocalUserChangedCallback(const TTV_ChatUserInfo* /*userInfo*/, void* /*userdata*/)
{
	printf("ChatChannelLocalUserChangedCallback\n");
}

void ChatChannelUserChangeCallback(const TTV_ChatUserList* joinList, const TTV_ChatUserList* leaveList, const TTV_ChatUserList* userInfoList, void* /*userdata*/)
{
	printf("Received user change callback\n");

	char buffer[256];

	printf("  Joins:\n");
	for (unsigned int i=0; i<joinList->userCount; ++i)
	{
		FormatModesAndSubscriptions(&joinList->userList[i], buffer);
		printf("    %s: %s\n", joinList->userList[i].userName, buffer);
	}

	printf("  Leaves:\n");
	for (unsigned int i=0; i<leaveList->userCount; ++i)
	{
		printf("    %s\n", leaveList->userList[i].userName);
	}

	printf("  Info Changes:\n");
	for (unsigned int i=0; i<userInfoList->userCount; ++i)
	{
		FormatModesAndSubscriptions(&userInfoList->userList[i], buffer);
		printf("    %s: %s\n", userInfoList->userList[i].userName, buffer);
	}

	TTV_Chat_FreeUserList(joinList);
	TTV_Chat_FreeUserList(leaveList);
	TTV_Chat_FreeUserList(userInfoList);
}

void ChatQueryChannelUsersCallback(const TTV_ChatUserList* userList, void* /*userdata*/)
{
	(void)userList;

	printf("Received full user list callback\n");

	for (unsigned int i=0; i<userList->userCount; ++i)
	{
		printf("  %s\n", userList->userList[i].userName);
	}

	TTV_Chat_FreeUserList(userList);
}

void ChatChannelMessageCallback(const TTV_ChatMessageList* messageList, void* /*userdata*/)
{
	assert(messageList);
	assert(messageList->messageCount > 0);
	assert(messageList->messageList);

	printf("Received %d messages\n", messageList->messageCount);

	for (unsigned int i=0; i<messageList->messageCount; ++i)
	{
		// TODO: Print all tokens if desired
		if ((messageList->messageList[i].flags & TTV_CHAT_MESSAGE_FLAG_ACTION) == TTV_CHAT_MESSAGE_FLAG_ACTION)
		{
			printf("  *%s* %s\n", messageList->messageList[i].userName, messageList->messageList[i].tokenList[0].data.text.buffer);
		}
		else
		{
			printf("  [%s]: %s\n", messageList->messageList[i].userName, messageList->messageList[i].tokenList[0].data.text.buffer);
		}
	}

	TTV_Chat_FreeMessageList(messageList);
}

void ChatClearCallback(const utf8char* /*username*/, void* /*userdata*/)
{
	printf("Clear the channel\n");
}


void ChatEmoticonDataDownloadCallback(TTV_ErrorCode error, void* /*userdata*/)
{
	printf("TTV_Chat_DownloadEmoticonData callback: %s\n", TTV_ErrorToString(error));
}


void ChatBadgeDataDownloadCallback(TTV_ErrorCode error, void* /*userdata*/)
{
	printf("TTV_Chat_DownloadBadgeData callback: %s\n", TTV_ErrorToString(error));
}


bool IsChatConnected()
{
	return sConnected;
}


void TestChatInit()
{
	TTV_ErrorCode err = TTV_Chat_Init(TTV_CHAT_TOKENIZATION_OPTION_NONE, nullptr, nullptr);
	if (err != TTV_EC_SUCCESS)
	{
		ASSERT_ON_ERROR(err);
	}
}


void TestChatShutdown()
{
	TTV_ErrorCode err = TTV_Chat_Shutdown(nullptr, nullptr);
	if (err != TTV_EC_SUCCESS)
	{
		ASSERT_ON_ERROR(err);
	}
}


void TestChatConnect(const char* username, const TTV_AuthToken& authToken)
{
	TTV_ChatCallbacks chatCallbacks;
	memset(&chatCallbacks, 0, sizeof(chatCallbacks));
	chatCallbacks.channelStateChangedCallback = &ChatChannelStateChangedCallback;
	chatCallbacks.channelInfoChangedCallback = &ChatChannelInfoChangedCallback;
	chatCallbacks.localUserChangedCallback = &ChatChannelLocalUserChangedCallback;
	chatCallbacks.userCallback = &ChatChannelUserChangeCallback;
	chatCallbacks.messageCallback = &ChatChannelMessageCallback;
	chatCallbacks.clearCallback = &ChatClearCallback;

	TTV_ErrorCode err = TTV_Chat_Connect(username, authToken.data, &chatCallbacks);
	if (err != TTV_EC_SUCCESS)
	{
		ASSERT_ON_ERROR(err);
	}
}


void TestChatDisconnect()
{
	TTV_ErrorCode err = TTV_Chat_Disconnect();
	if (err != TTV_EC_SUCCESS)
	{
		ASSERT_ON_ERROR(err);
	}
}


void TestChatSpeak(const char* msg)
{
	TTV_ErrorCode err = TTV_Chat_SendMessage(msg);

	if (err < TTV_EC_SUCCESS)
	{
		printf("WARNING: %s\n", TTV_ErrorToString(err));
	}
	else if (TTV_FAILED(err))
	{
		printf("ERROR: %s\n", TTV_ErrorToString(err));
	}
}


void TestChatFlush()
{
	TTV_ErrorCode err = TTV_Chat_FlushEvents();
	if (err != TTV_EC_SUCCESS)
	{
		ASSERT_ON_ERROR(err);
	}
}

#pragma warning (pop)
