#ifdef TTV_PLATFORM_WINDOWS
#	define _CRT_SECURE_NO_WARNINGS 1

#	define ENABLE_MEMORY_LEAK_DEBUGGING 1
#	if ENABLE_MEMORY_LEAK_DEBUGGING
#		define _CRTDBG_MAP_ALLOC
#		include <stdlib.h>
#		include <crtdbg.h>
#	endif

#endif

#include "main.h"
#include "twitchsdk.h"
#include "sdktester.h"
#include "chattester.h"

#if !TTV_PLATFORM_IOS && !TTV_PLATFORM_MAC
#   define SUPPORT_X264 1
#endif

#if SUPPORT_X264
#   include "../twitchsdk/samples/encoderplugin/x264plugin.h"
#endif

#include <stdlib.h>
#include <stdio.h>
#include <time.h>

#include <fstream>
#include <algorithm>
#include <iostream>
#include <functional>
#include <memory>
#include <unordered_map>
#include <cctype>
#include <assert.h>
#include <list>
#include <cstring>

#ifdef TTV_PLATFORM_WINDOWS
#define NOMINMAX
#include <Windows.h>
#include <conio.h>
#else
#endif

#include <twitchwebcam.h>

#if TTV_PLATFORM_IOS
#import <UIKit/UIKit.h>
#endif

extern "C" TTVSDK_API void TTV_SetEncoders(TTV_VideoEncoder vidEncoder, TTV_AudioEncoder audEncoder);

static std::unordered_map<std::string, std::string> sVariableMap;

static TTV_AuthToken sAuthToken;
static TTV_IngestList sIngestList;
static TTV_GameInfoList sGameInfoList;
static int sSelectedIngestIndex = -1;
static TTV_ChannelInfo sChannelInfo;
static bool sLoggedIn = false;
static TTV_UserInfo sUserInfo;
static TTV_StreamInfo sStreamInfo;
static TTV_ArchivingState sArchivingState;
static bool sHasAuthToken = false;
static bool sUsePasswordGrant = true;
static TTV_StreamInfoForSetting sStreamInfoToSet;
static TTV_LiveGameStreamList sLiveGameStreams;

static std::string sUserName;
static const char* kInputFileName = "inputFrames.bin";

static bool sWebcamsInitialized = false;
static int sSelectedWebcam = -1;
static int sWebcamCapabilityIndex = -1;
static bool sWebcamShutdownComplete = false;
static std::string sWebcamUniqueIdentifier;
static uint32_t sWebcamFrameRate = 0;
static unsigned int sWebcamHeight = 0;
static unsigned int sWebcamWidth = 0;
static bool sWebcamTopToBottom = false;
static bool sWaitingForStartStopReply = false;
static bool sStarted = false;
static bool sSubmitTheFrames = true;
static bool sSubmitAudioSamples = false;

#if SUPPORT_X264
static X264Plugin* sX264Plugin = nullptr;
#endif

static int sNumSubmittedBuffers = 0;


void StreamInfoDoneCallback(TTV_ErrorCode result, void* userData);

#if !TTV_PLATFORM_WINDOWS
	#define _aligned_free(ptr) free(ptr)
	#define _aligned_malloc(size, alignment) malloc(size)
#endif

void FrameUnlockCallback (const uint8_t* buffer, void* /*userData*/)
{
	//delete[] buffer;
	_aligned_free(const_cast<void*>(reinterpret_cast<const void*>(buffer)));
	--sNumSubmittedBuffers;
}

void StartCallback(TTV_ErrorCode result, void* /*userData*/)
{
	sStarted = TTV_SUCCEEDED(result) ? true : sStarted;
	assert( sStarted );

	sWaitingForStartStopReply = false;
}

void StopCallback(TTV_ErrorCode result, void* /*userData*/)
{
	sStarted = TTV_SUCCEEDED(result) ? false : sStarted;
	assert( !sStarted );

	sWaitingForStartStopReply = false;
}

void SetStreamInfoCallback(TTV_ErrorCode result, void* /*userData*/)
{
	assert( TTV_SUCCEEDED(result) || result == TTV_EC_API_REQUEST_FAILED );

	if (TTV_SUCCEEDED(result))
	{
		printf ("SetStreamInfo succeeded\n");
	}
	else
	{
		printf ("SetStreamInfo failed\n");
	}
}


#pragma region Config Settings

void ParseConfigFile(const std::string& workingDir, const std::string& filename)
{
	std::string path(filename);
	if (workingDir.size() > 0)
	{
		path = workingDir + "/" + path;
	}
	std::ifstream file(path);

	// file doesn't exist
	if (!file.is_open())
	{
		return;
	}

	while (!file.eof())
	{
		std::string line;
		std::getline(file, line);

		// trim leading and trailing whitespace
		line.erase(line.begin(), std::find_if(line.begin(), line.end(), std::not1(std::ptr_fun<int, int>(std::isspace))));
		line.erase(std::find_if(line.rbegin(), line.rend(), std::not1(std::ptr_fun<int, int>(std::isspace))).base(), line.end());

		if (line.size() == 0 || line[0] == '#')
		{
			continue;
		}

		// On Mac getline seems to keep \r at the end of the line
		if (line[line.size()-1] == '\r')
		{
			line.resize(line.size()-1);
		}

		std::string key;
		std::string val;

		size_t index = line.find('=');
		if (index == std::string::npos)
		{
			key = line;
			val = "";
		}
		else
		{
			key = line.substr(0, index);
			val = line.substr(index+1, line.size()-index-1);
		}

		sVariableMap[key] = val;
	}

	file.close();
}

bool ContainsValue(const std::string& key)
{
	return sVariableMap.find(key) != sVariableMap.end();
}

std::string LookupString(const std::string& key)
{
	return sVariableMap[key];
}

int LookupInt(const std::string& key)
{
	std::string val = sVariableMap[key];
	int i = atoi(val.c_str());
	return i;
}

float LookupFloat(const std::string& key)
{
	std::string val = sVariableMap[key];
	float i = static_cast<float>(atof(val.c_str()));
	return i;
}

bool LookupBool(const std::string& key)
{
	std::string val = sVariableMap[key];
	return val == "true" || val == "1";
}


void PrintUsage()
{
	std::cout <<
		"help                 produce help message\n"
		"username             user name\n"
		"password             password\n"
		"password_grant		  enable or disable password grant"
		"bits_per_pixel		  bits per pixel used to calculate output resolution [0.1]"
		"client_id            client ID\n"
		"client_secret        client secret\n"
		"output_file          output filename\n"
		"enable_audio         enable audio\n"
		"frame_pusher         number of frames for frame pusher (0=disabled)\n"
		"output_width         output width (uses bits_per_pixel if not set)\n"
		"output_height        output height (uses bits_per_pixel if not set)\n"
		"output_bitrate       output bitrate\n"
		"output_fps           output framereate\n"
		"motion_factor		  motion factor (1.0 for average; as high as 2.0 for fast moving scense like first person shooters)"
		"volume_test          Test audio volume controls\n"
		"tracing_file         tracing filename\n"
		"trace_level          tracing level\n"
		"async_start          test waiting on an async start\n"
		"async_stop           test waiting on an async stop\n"
		"video_encoder        video encoder to use - intel or x264\n"
		"audio_encoder		  audio encoder to use - only mp3 on pc, aac on apple\n"
		"ingest_server_index  Index of ingest server to use\n"
		"chat_test            Run the chat test as well.  Values: none, spam, listen\n"
		"print_webcam_identities		Print the unique identifiers of the available webcams\n"
		"broadcast_webcam		Broadcast a webcam image instead of a desktop image\n"
		"webcam_identifier	The unique identifier of which webcam to use. A list of unique identifiers can be printed out if the print_webcam_identities option is set to true\n"
		;
}

#pragma endregion


void RunFramePusher(const TTV_VideoParams& videoParams, const TTV_AudioParams& audioParams, int frameCount)
{
	FILE* inputFile = fopen(kInputFileName, "rb");
	if (!inputFile)
	{
		printf ("FramePusher: No input file, generating one\n");
		// No frames input file exists. Generate one.
		//
		inputFile = fopen(kInputFileName, "ab");
		assert (inputFile);
		for (int i = 0; i < frameCount; ++i)
		{
			//uint8_t* frameData = new uint8_t[videoParams.outputWidth * videoParams.outputHeight * 4];
			uint8_t* frameData = reinterpret_cast<uint8_t*> (_aligned_malloc(videoParams.outputWidth * videoParams.outputHeight * 4, 16));
			assert (frameData);
			GenerateBGRAFrame(frameData, videoParams.outputWidth, videoParams.outputHeight);
			fwrite(frameData, videoParams.outputWidth * videoParams.outputHeight * 4, 1, inputFile);
			//delete [] frameData;
			_aligned_free(frameData);

			if (i % 13)
			{
				printf ("FramePusher: %d\r", i);
			}
		}
		printf ("FramePusher: %d\n", frameCount);


		fclose(inputFile);
		inputFile = nullptr;
	}

	if (!inputFile)
	{
		inputFile = fopen(kInputFileName, "rb");
	}
	assert(inputFile);

	// Load all the frames from the file
	//
	if (inputFile)
	{
		// Push all the frames through the encoder
		printf ("FramePusher: Pushing all frames\n");

		TTV_ErrorCode ret = TTV_EC_SUCCESS;
		TTV_IngestServer* ingest = sSelectedIngestIndex >= 0 ? &sIngestList.ingestList[sSelectedIngestIndex] : nullptr;

		if (LookupBool("async_start"))
		{
			sWaitingForStartStopReply = true;
			ret = TTV_Start(&videoParams, &audioParams, ingest, 0, StartCallback, nullptr);
			ASSERT_ON_ERROR(ret);

			while (sWaitingForStartStopReply)
			{
				TTV_PollTasks();
			}
			assert(sStarted);
		}
		else
		{
			ret = TTV_Start(&videoParams, &audioParams, ingest, 0, nullptr, nullptr);
			ASSERT_ON_ERROR(ret);
		}

		TTV_FreeIngestList(&sIngestList);

		uint64_t ta = GetSystemClockTime();

		for (int i = 0; i < frameCount; ++i)
		{
			//uint8_t* frameData = new uint8_t[videoParams.outputWidth * videoParams.outputHeight * 4];
			uint8_t* frameData = reinterpret_cast<uint8_t*> (_aligned_malloc(videoParams.outputWidth * videoParams.outputHeight * 4, 16));

			// TODO: Handle rejected frames due to frame limiting
			if (TTV_FAILED(ret))
			{
				break;
			}

			fread(frameData, videoParams.outputWidth * videoParams.outputHeight * 4, 1, inputFile);

			assert (sNumSubmittedBuffers <= 2);
			ret = TTV_SubmitVideoFrame(frameData, FrameUnlockCallback, 0);
			sNumSubmittedBuffers++;

			if (TTV_FAILED(ret))
			{
				break;
			}
			if (i % 13)
			{
				printf ("FramePusher: %d\r", i);
			}
			//delete[] frameData;
			_aligned_free(frameData);
		}
		printf ("FramePusher: %d\n", frameCount);

		if (TTV_FAILED(ret))
		{
			printf("\n!! Error code = %d. Shutting down...\n\n", ret);
			assert(false);
		}

		printf("\nShutting down...\n\n");
		TTV_Shutdown();

		uint64_t tb = GetSystemClockTime();
		uint64_t ms = SystemTimeToMs(tb - ta);

		printf("\n\n===> total encode time = %lld\n", ms);

	}
}


void FindAvailableWebcamFormat(const TTV_WebCamDeviceCapabilityList& webcamCapabilities)
{
	// Find the first set of MediaParameters that supports RGB32 and has a frame rate above 20
	// Also, make sure height and width are evenly divisible by 16 so they are useful to the encoder
	bool supportedFormatFound = false;
	for (size_t i = 0; i < webcamCapabilities.count; ++i)
	{
		if (webcamCapabilities.list[i].format != TTV_WEBCAM_FORMAT_XRGB32 && // The encoder currently ignores the X or alpha channel.
			webcamCapabilities.list[i].format != TTV_WEBCAM_FORMAT_ARGB32)
		{
			continue;
		}

		if (    webcamCapabilities.list[i].frameRate > 20
			&& (webcamCapabilities.list[i].resolution.height % 16) == 0
			&& (webcamCapabilities.list[i].resolution.width % 16) == 0 )
		{
			supportedFormatFound = true;
			sWebcamCapabilityIndex = webcamCapabilities.list[i].capabilityIndex;
			sWebcamTopToBottom = webcamCapabilities.list[i].isTopToBottom;
			sWebcamFrameRate = webcamCapabilities.list[i].frameRate;
			sWebcamHeight = webcamCapabilities.list[i].resolution.height;
			sWebcamWidth = webcamCapabilities.list[i].resolution.width;
			break;
		}
	}

	if (supportedFormatFound == false)
	{
		printf("No webcam format found that matches 24bpp and >20fps\n");
		exit(-1);
	}
}

void FlipVertically(uint8_t* const imageBuffer, const size_t bitsPerPixel, const size_t width, const size_t height)
{
	std::unique_ptr<uint8_t[]> temporaryRow(new uint8_t[(bitsPerPixel / 8) * width]);

	const size_t halfHeight = (height / 2);
	const size_t bytesPerRow = (bitsPerPixel / 8) * width;

	uint8_t* topRowToCopy = imageBuffer;
	uint8_t* bottomRowToCopy = imageBuffer + (bitsPerPixel / 8) * width * (height - 1);

	for (size_t currentRow = 0; currentRow < halfHeight; currentRow++)
	{
		memcpy(temporaryRow.get(), topRowToCopy, bytesPerRow);
		memcpy(topRowToCopy, bottomRowToCopy, bytesPerRow);
		memcpy(bottomRowToCopy, temporaryRow.get(), bytesPerRow);

		topRowToCopy += bytesPerRow;
		bottomRowToCopy -= bytesPerRow;
	}
}

void WebCamDeviceChangeCallback(TTV_WebCamDeviceChange change, const TTV_WebCamDevice* device, TTV_ErrorCode error, void* /*userdata*/)
{
	if (TTV_SUCCEEDED(error))
	{
		switch (change)
		{
			case TTV_WEBCAM_DEVICE_FOUND:
			{
				bool printWebcamIdentities = false;
				if (ContainsValue("print_webcam_identities"))
				{
					printWebcamIdentities = LookupBool("print_webcam_identities");
				}
				if (printWebcamIdentities)
				{
					printf("Webcam unique identifier: %s\n", device->uniqueId);
				}
			}

			if (sSelectedWebcam == -1) // We haven't set a webcam yet
			{
				if (sWebcamUniqueIdentifier.empty())
				{
					// Use the first available webcam
					sSelectedWebcam = device->deviceIndex;
					FindAvailableWebcamFormat(device->capabilityList);
				}
				else
				{
					// Try to match the specific unique device id.
					// We are only going to test if the first device matches that id and fail if it isn't.
					// This means if there are two devices, and the other one gets initialized first, we will fail incorrectly.
					// This is just a temporary hack.

					if (sWebcamUniqueIdentifier == device->uniqueId)
					{
						sSelectedWebcam = device->deviceIndex;
						FindAvailableWebcamFormat(device->capabilityList);
					}
					else
					{
						printf("Could not find matching webcam device.\n");
						exit(-1);
					}
				}
			}

			sWebcamsInitialized = true;

			break;
		case TTV_WEBCAM_DEVICE_LOST:
			// Not really going to handle this in SDK tester
			break;
		}
	}
}

void UpdateWebcamFrame()
{
	if (sWebcamCapabilityIndex < 0)
	{
		return;
	}

	bool available = false;
	(void)TTV_WebCam_IsFrameAvailable(sSelectedWebcam, &available);

	if (!available)
	{
		return;
	}

	// If I can just pass the raw frame data, I might not need to keep my own buffer in this example
	const unsigned int width = sWebcamWidth;
	const unsigned int height = sWebcamHeight;

	uint8_t* frameData = new uint8_t[width * height * 4];
	assert (frameData);

	// get the next frame if available
	auto ret = TTV_WebCam_GetFrame(sSelectedWebcam, frameData, width*4);

	if (!sWebcamTopToBottom)
	{
		FlipVertically(frameData, 32, width, height);
	}

#if TTV_PLATFORM_IOS
	// TODO: We can rotate the image here if we're feeling up to it
	UIInterfaceOrientation orient = [[UIApplication sharedApplication] statusBarOrientation];
	switch(orient)
	{
		// No rotation required
		case UIDeviceOrientationPortrait:
		default:
			break;
		// Rotation required
		case UIDeviceOrientationPortraitUpsideDown:
			break;
		case UIDeviceOrientationLandscapeLeft:
			break;
		case UIDeviceOrientationLandscapeRight:
			break;
	}
#endif

	assert (sNumSubmittedBuffers <= 2);

	ret = TTV_SubmitVideoFrame(frameData, FrameUnlockCallback, 0);
	sNumSubmittedBuffers++;

	assert(TTV_SUCCEEDED(TTV_EC_SUCCESS));
	(void)ret;
}


void WebcamShutdown(TTV_ErrorCode /*error*/, void* /*userdata*/)
{
	sWebcamShutdownComplete = true;
}


void InitializeWebcam()
{
	bool printWebcamIdentities = false;
	if (ContainsValue("print_webcam_identities"))
	{
		printWebcamIdentities = LookupBool("print_webcam_identities");
	}
	bool broadcastWebcam = false;
	std::string selectedWebcamIdentifier;
	if (ContainsValue("broadcast_webcam"))
	{
		broadcastWebcam = LookupBool("broadcast_webcam");
		if (ContainsValue("webcam_identifier"))
		{
			// If a specific webcam identifier is provided, try to find that webcam.
			// Otherwise, just use the first available webcam
			sWebcamUniqueIdentifier = LookupString("webcam_identifier");
		}
	}

	const bool shouldInitializeWebcams = printWebcamIdentities || broadcastWebcam;
	if (shouldInitializeWebcams)
	{
		TTV_WebCamCallbacks webcamCallbacks;
		webcamCallbacks.deviceChangeCallback = WebCamDeviceChangeCallback;
		webcamCallbacks.deviceChangeUserData = nullptr;

		if (TTV_FAILED(TTV_WebCam_Init(&webcamCallbacks, nullptr, nullptr)))
		{
			printf("Could not initialize TwitchSDK webcams\n");
			exit(-1);
		}
		else
		{
			// Spin until the webcams are initialized
			while (sWebcamsInitialized == false)
			{
				if (TTV_FAILED(TTV_WebCam_FlushEvents()))
				{
					printf("Could not flush webcam events\n");
					exit(-1);
				}
			}
		}
	}
}

void GetLiveStreamsCallback(TTV_ErrorCode result, void* /*userData*/)
{
	if (TTV_FAILED(result))
	{
		printf ("GetLiveStreamsCallback got failure\n");
	}
	else if (sLiveGameStreams.count > 0)
	{
		printf("\nThere are %d streams for the game:\n", sLiveGameStreams.count);
		for (uint i = 0; i < sLiveGameStreams.count; ++i)
		{
			printf("%s - %s\n%s\n%s\n%d Viewers\n----------\n", sLiveGameStreams.list[i].channelDisplayName, sLiveGameStreams.list[i].streamTitle,
																sLiveGameStreams.list[i].channelUrl,
																sLiveGameStreams.list[i].previewUrlTemplate,
																sLiveGameStreams.list[i].viewerCount);
		}
	}
	TTV_FreeGameLiveStreamList(&sLiveGameStreams);
}

void RunCommercialCallback(TTV_ErrorCode result, void* /*userData*/)
{
	if (TTV_FAILED(result))
	{
		printf("RunCommercialCallback got failure\n");
	}
	else
	{
		printf("RunCommercialCallback got success\n");
	}
}


void StartStream()
{
	TTV_VideoParams videoParams = {sizeof(TTV_VideoParams)};
	videoParams.maxKbps = LookupInt("output_bitrate");
	videoParams.targetFps = 30;
	videoParams.verticalFlip = false;

	TTV_AudioParams audioParams;
	memset(&audioParams, 0, sizeof(TTV_AudioParams));
	audioParams.size = sizeof(TTV_AudioParams);
	audioParams.audioEnabled = LookupBool("enable_audio");
	audioParams.enableMicCapture = true;
#if TTV_PLATFORM_IOS
	if (ContainsValue("openal_test"))
	{
		sSubmitAudioSamples = true;
	}
	audioParams.enablePassthroughAudio = sSubmitAudioSamples;
#else
	audioParams.enablePlaybackCapture = true;
#endif

#if TTV_PLATFORM_IOS
	if (sSubmitAudioSamples)
	{
		StartAudioPlayback();
		ConfigureAudioCapture();
	}
#endif

	if (ContainsValue("output_file"))
	{
		std::string outputFile = LookupString("output_file");
		SetOutputFileName(outputFile);
	}

	if (ContainsValue("broadcast_webcam"))
	{
		if (TTV_FAILED(TTV_WebCam_Start(sSelectedWebcam, sWebcamCapabilityIndex, nullptr, nullptr)))
		{
			printf("Could not start webcam capture\n");
			exit(-1);
		}

		videoParams.outputWidth = sWebcamWidth;
		videoParams.outputHeight = sWebcamHeight;
		videoParams.targetFps = sWebcamFrameRate;
		videoParams.maxKbps = 1500;
		videoParams.pixelFormat = TTV_PF_BGRA;
	}
	else
	{
		if (ContainsValue("output_fps"))
		{
			videoParams.targetFps = LookupInt("output_fps");
		}
		videoParams.encodingCpuUsage = TTV_ECU_MEDIUM;
		videoParams.pixelFormat = TTV_PF_BGRA;
#if SCREEN_CAPTURE
		videoParams.pixelFormat = kScreenCapturePixelFormat;
#endif

		float bitsPerPixel = LookupFloat("bits_per_pixel");

		TTV_ErrorCode ret = TTV_GetMaxResolution(videoParams.maxKbps,
			videoParams.targetFps,
			bitsPerPixel,
			16.f/9.f,
			&videoParams.outputWidth,
			&videoParams.outputHeight);

		if (ContainsValue("output_width"))
		{
			videoParams.outputWidth = LookupInt("output_width");
		}

		if (ContainsValue("output_height"))
		{
			videoParams.outputHeight = LookupInt("output_height");
		}

		if (TTV_FAILED(ret))
		{
			ASSERT_ON_ERROR(ret);
		}
	}

#if SUPPORT_X264
	videoParams.encoderPlugin = sX264Plugin;
#endif

	bool hasAuthParams = ContainsValue("username") && ContainsValue("password") && ContainsValue("client_secret");
	sUsePasswordGrant = LookupBool("password_grant");

	// Read frames from a file and then push them through the encoder at full speed
	//

	int frameCount = LookupInt("frame_pusher");

	if (frameCount > 0)
	{
		RunFramePusher(videoParams, audioParams, frameCount);
	}
	else
	{
		TTV_ErrorCode ret = TTV_EC_SUCCESS;

		TTV_IngestServer* ingest = sSelectedIngestIndex >= 0 ? &sIngestList.ingestList[sSelectedIngestIndex] : nullptr;
		
		if (LookupBool("async_start"))
		{
			sWaitingForStartStopReply = true;
			ret = TTV_Start(&videoParams, &audioParams, ingest, 0, StartCallback, nullptr);
			ASSERT_ON_ERROR(ret);

			while (sWaitingForStartStopReply)
			{
				TTV_PollTasks();
			}
			assert(sStarted);
		}
		else
		{
			ret = TTV_Start(&videoParams, &audioParams, ingest, 0, nullptr, nullptr);
			ASSERT_ON_ERROR(ret);
		}

		TTV_FreeIngestList(&sIngestList);

		float volume = 1.f;
		float volumeDir = -(1/30.f);
		bool volume_test = LookupBool("volume_test");
		std::string chat_test = LookupString("chat_test");

		int frameCount = 0;

#if TTV_PLATFORM_IOS
		if (ContainsValue("openal_test"))
		{
			StartAudioCapture();
		}
#endif

		while (TTV_SUCCEEDED(ret))
		{
			if (ContainsValue("broadcast_webcam"))
			{
				if (TTV_FAILED(TTV_WebCam_FlushEvents()))
				{
					printf("Could not flush webcam events\n");
					exit(-1);
				}

				// get the latest webcam frame
				UpdateWebcamFrame();
			}
			else
			{
				//uint8_t* frameData = new uint8_t[videoParams.outputWidth * videoParams.outputHeight * 4];
				uint8_t* frameData = reinterpret_cast<uint8_t*> (_aligned_malloc(videoParams.outputWidth * videoParams.outputHeight * 4, 16));

				memset(frameData, 0, videoParams.outputWidth * videoParams.outputHeight * 4);
				assert (frameData);

				GenerateBGRAFrame(frameData, videoParams.outputWidth, videoParams.outputHeight);

				if (sSubmitTheFrames)
				{
					ret = TTV_SubmitVideoFrameWithTimeStamp(frameData, FrameUnlockCallback, TTV_GetCurrentTimeStamp(), 0);

					if (ret == TTV_EC_FRAME_QUEUE_TOO_LONG)
					{
						ret = TTV_EC_SUCCESS;
						static bool b = true;
						if (b)
						{
							printf ("Queue is long!!\n");
							b=false;
						}
					}
				}
				else
				{
					//delete[] frameData;
					_aligned_free(frameData);
				}
				assert(TTV_SUCCEEDED(ret));

#if TTV_PLATFORM_IOS
				if (sSubmitAudioSamples)
				{
					SubmitAudioSamples();
				}
#endif
			}

			frameCount++;

			if (volume_test)
			{
				volume -= volumeDir;
				if (volume > 1.f || volume < -1.f)
				{
					volume = std::max (std::min (volume, 1.f), -1.f);
					volumeDir *= -1;
				}

#if !TTV_PLATFORM_IOS
				TTV_SetVolume(TTV_PLAYBACK_DEVICE, volume);
				TTV_SetVolume(TTV_RECORDER_DEVICE  , volume);
				printf ("\rVolume: %f                ", volume);
#endif
			}

			// Send some test metadata every 100 frames (if not writing to a local file)
			if (hasAuthParams)
			{
				if (frameCount % 100 == 0)
				{
					std::string desc(std::to_string(frameCount) + " Frames Sent!");

					uint64_t streamTime = 0;
					TTV_GetStreamTime(&streamTime);

					ret = TTV_SendActionMetaData(&sAuthToken, "marker", streamTime, desc.c_str(), "{\"title\":\"sdktester\"}", nullptr, nullptr);
					if (ret == TTV_EC_METADATA_CACHE_FULL)
					{
						//printf("\n!! Metadata cache is full\n");
					}
					else
					{
						ASSERT_ON_ERROR(ret);
					}

					unsigned long sid = 0;
					ret = TTV_SendStartSpanMetaData(&sAuthToken, "begin", streamTime, &sid, desc.c_str(), "{\"title\":\"sdktester\"}", nullptr, nullptr);

					if (ret == TTV_EC_INVALID_JSON)
					{
						printf("\n!! Invalid JSON passed for metadata\n");
					}
					else if (ret == TTV_EC_METADATA_CACHE_FULL)
					{
						//printf("\n!! Metadata cache is full\n");
					}
					else if (TTV_FAILED(ret))
					{
						ASSERT_ON_ERROR(ret);
					}
					else
					{
						streamTime++;

						ret = TTV_SendEndSpanMetaData(&sAuthToken, "end", streamTime, sid, desc.c_str(), "{\"title\":\"sdktester\"}", nullptr, nullptr);
						ASSERT_ON_ERROR(ret);
					}

					ret = TTV_GetStreamInfo(&sAuthToken, StreamInfoDoneCallback, nullptr, sUserName.c_str(), &sStreamInfo);
					ASSERT_ON_ERROR(ret);
				}
			}

			// chat stuff
			if (chat_test != "none")
			{
				// send messages
				if (chat_test == "spam")
				{
					// if you send messages too quickly (20 in 30 seconds) you will have your ip banned for 8 hours and it'll drop the connection
					if (frameCount % 200 == 0)
					{
						for (int i=0; i<1; ++i)
						{
							char buffer[64];
							snprintf(buffer, sizeof(buffer), "%d-Chat on frame %d-%d", i, frameCount, i);
							TestChatSpeak(buffer);
						}
					}
				}

				TestChatFlush();
			}

			ret = TTV_PollTasks();

			if (_kbhit())
			{
#ifdef TTV_PLATFORM_WINDOWS
				// pause the video
				int ch = _getch();
				if (ch == 'p')
				{
					printf("Pausing video...");
					TTV_PauseVideo();
					Sleep(3000);
					printf("Done\n");
				}
				else if (ch == 's')
				{
					printf("Stalling system...");
					Sleep(3000);
					printf("Done\n");
				}
				// connect/disconnect from chat
				else if (ch == 'c')
				{
					if (IsChatConnected())
					{
						printf("Attempting disconnect from chat...\n");
						TestChatDisconnect();
					}
					else
					{
						printf("Attempting connection to chat...\n");
						TestChatConnect(sUserName.c_str(), sAuthToken);
					}
				}
				else if (ch == 'a')
				{
					printf("Triggering a commercial...\n");
					TTV_RunCommercial(&sAuthToken, RunCommercialCallback, nullptr);
				}
				else if (ch == 'm')
				{
					TestChatSpeak("message");
				}
				else if (ch == 'l')
				{
					static uint offset = 0;
					const uint kLimit = 30;
					memset(&sLiveGameStreams, 0, sizeof(sLiveGameStreams));
					TTV_GetGameLiveStreams("Smite", kLimit, offset, GetLiveStreamsCallback, nullptr, &sLiveGameStreams);
					offset += kLimit;
				}
				else if (ch == 's')
				{
					sSubmitTheFrames = !sSubmitTheFrames;
					printf ("Now I %s submitting frames\n", sSubmitTheFrames ? "am" : "am not");
				}
				else if (ch == '-')
				{
					TTV_SetVolume(TTV_RECORDER_DEVICE, 0.f);
				}
				else if (ch == '=')
				{
					TTV_SetVolume(TTV_RECORDER_DEVICE, 1.f);
				}
				else
#endif
				{
					break;
				}
			}
		}

		if (TTV_FAILED(ret))
		{
			printf("\n!! Error code = %d. Shutting down...\n\n", ret);
			ASSERT_ON_ERROR(ret);
		}

#ifdef TTV_PLATFORM_WINDOWS
		while (_kbhit())
		{
#pragma warning(suppress: 6031)
			_getch();
		}
#endif

		if (LookupBool("async_stop"))
		{
			sWaitingForStartStopReply = true;
			ret = TTV_Stop(StopCallback, nullptr);
			assert(TTV_SUCCEEDED(ret));

			while (sWaitingForStartStopReply)
			{
				TTV_PollTasks();
			}
			assert(!sStarted);
		}
		else
		{
			ret = TTV_Stop(nullptr, nullptr);
			assert(TTV_SUCCEEDED(ret));
		}

#if SUPPORT_X264
		delete sX264Plugin;
#endif

		if (ContainsValue("broadcast_webcam"))
		{
			if (TTV_FAILED(TTV_WebCam_Stop(sSelectedWebcam, nullptr, nullptr)))
			{
				printf("Could not stop the webcam\n");
				exit(-1);
			}
			TTV_WebCam_Shutdown(WebcamShutdown, nullptr);

			while (sWebcamShutdownComplete == false)
			{
				if (TTV_FAILED(TTV_WebCam_FlushEvents()))
				{
					printf("Could not flush webcam events\n");
					exit(-1);
				}
			}
		}

		// shutdown chat
		if (chat_test != "none")
		{
			TestChatDisconnect();
			TestChatShutdown();
		}

		TTV_Shutdown();
	}
}

void AuthDoneCallback(TTV_ErrorCode result, void* /*userData*/)
{
	if (TTV_SUCCEEDED(result))
	{
		sHasAuthToken = true;

		// Save the auth token
		FILE* file = fopen("authtoken.cfg", "wb");
		if (file != nullptr)
		{
			fwrite(&sAuthToken, sizeof(sAuthToken), 1, file);
			fclose(file);
		}
	}
	else
	{
		printf ("AuthDoneCallback got failure\n");
		exit(-1);
	}
}

void LoginCallback(TTV_ErrorCode result, void* userData)
{
	if (TTV_SUCCEEDED(result))
	{
		printf("Login successfull - Broadcasting on channel %s (%s)\n", sChannelInfo.displayName, sChannelInfo.channelUrl);
		sLoggedIn = true;

		sUserName = sChannelInfo.name;
	}
	else
	{
		printf ("LoginCallback got failure - Auth token is invalid\n");

		if (userData)
		{
			// Invalidate the current auth token and get a new one
			sHasAuthToken = false;
			TTV_AuthParams* authParams = reinterpret_cast<TTV_AuthParams*>(userData);
			TTV_ErrorCode ret = TTV_RequestAuthToken(authParams, (TTV_RequestAuthToken_Broadcast | TTV_RequestAuthToken_Chat), AuthDoneCallback, nullptr, &sAuthToken);
			while (!sHasAuthToken)
			{
				TTV_PollTasks();
				ThreadSleep(5);
			}
			sChannelInfo.size = sizeof(sChannelInfo);
			ret = TTV_Login(&sAuthToken, LoginCallback, nullptr, &sChannelInfo);
			ASSERT_ON_ERROR(ret);
		}
		else
		{
			exit(-1);
		}
	}
}

void IngestListCallback(TTV_ErrorCode result, void* /*userData*/)
{
	if (TTV_SUCCEEDED(result))
	{
		uint serverIndex = 0;
		if (sVariableMap.count("ingest_server_index"))
		{
			serverIndex = LookupInt("ingest_server_index") % sIngestList.ingestCount;
		}
		else for (uint i = 0; i < sIngestList.ingestCount; ++i)
		{
			if (sIngestList.ingestList[i].defaultServer)
			{
				serverIndex = i;
				break;
			}
		}

		printf ("Using ingest server %s [%s] (%d/%d)\n",
		sIngestList.ingestList[serverIndex].serverName,
		sIngestList.ingestList[serverIndex].serverUrl,
		serverIndex+1, sIngestList.ingestCount);

		sSelectedIngestIndex = serverIndex;
	}
	else
	{
		printf ("IngestListCallback got failure\n");
		TTV_FreeIngestList(&sIngestList);
		exit(-1);
	}
}

void GameNameListCallback(TTV_ErrorCode result, void* /*userData*/)
{
	if (TTV_SUCCEEDED(result))
	{
		printf ("GameNameListCallback succeeded\n");

		for (unsigned int i=0; i<sGameInfoList.count; ++i)
		{
			printf("  Name: %s Id: %d Popularity: %d\n", sGameInfoList.list[i].name, sGameInfoList.list[i].id, sGameInfoList.list[i].popularity);
		}

		TTV_FreeGameNameList(&sGameInfoList);
	}
	else
	{
		printf ("GameNameListCallback got failure\n");
		TTV_FreeGameNameList(&sGameInfoList);
	}
}

void UserInfoDoneCallback(TTV_ErrorCode result, void* /*userData*/)
{
	if (TTV_FAILED(result))
	{
		printf ("UserInfoDoneCallback got failure\n");
	}
}

void ArchivingStateCallback(TTV_ErrorCode result, void* /*userData*/)
{
	if (TTV_SUCCEEDED(result))
	{
		if (sArchivingState.recordingEnabled)
		{
			printf("Archiving is enabled!\n");
		}
		else
		{
			printf("Archiving is disabled! Go here to fix it %s\n", sArchivingState.cureUrl);
		}
	}
	else
	{
		printf("ArchivingStateCallback got failure\n");
	}
}

void StreamInfoDoneCallback(TTV_ErrorCode result, void* /*userData*/)
{
	if (TTV_FAILED(result))
	{
		if (result == TTV_EC_WEBAPI_RESULT_NO_STREAMINFO)
		{
			printf("\n==>No Stream info is available\n");
		}
		else
		{
			printf ("StreamInfoDoneCallback got failure\n");
		}
	}
	else
	{
		printf("Got stream info, current viewer count: %d\n", sStreamInfo.viewers);
	}
}

int main(int argc, char** argv)
{
	(void)argc;
	(void)argv;

#if TTV_PLATFORM_WINDOWS && ENABLE_MEMORY_LEAK_DEBUGGING
	_CrtSetDbgFlag ( _CRTDBG_ALLOC_MEM_DF | _CRTDBG_LEAK_CHECK_DF );
	_CrtSetReportMode( _CRT_ERROR, _CRTDBG_MODE_DEBUG );
#endif


	srand(static_cast<uint>(time(nullptr)));
	setlocale(LC_ALL, "utf8");

	// Set default values
	sVariableMap["enable_audio"] = "true";
	sVariableMap["volume_test"] = "false";
	sVariableMap["frame_pusher"] = "0";
	sVariableMap["bits_per_pixel"] = "0.1";
	sVariableMap["trace_level"] = "TTV_ML_NONE";
	sVariableMap["chat_test"] = "none";
	sVariableMap["chat_channel"] = "";
	sVariableMap["async_start"] = "false";
	sVariableMap["async_stop"] = "false";
	
#if TTV_PLATFORM_WINDOWS
	sVariableMap["password_grant"] = "false";
#else
	sVariableMap["password_grant"] = "true";
#endif

	std::string workingDir;
#if TTV_PLATFORM_IOS
	workingDir = std::string(argv[0]);
	workingDir = workingDir.substr(0, workingDir.rfind("/"));
#endif
	ParseConfigFile(workingDir, "sdktester.cfg");

	// Init stream info
	memset(&sStreamInfo, 0, sizeof(sStreamInfo));
	sStreamInfo.size = sizeof(sStreamInfo);

	// try and read the auth token
	FILE* authFile = fopen("authtoken.cfg", "rb");
	sHasAuthToken = authFile != nullptr;
	if (authFile)
	{
		fread(&sAuthToken, sizeof(sAuthToken), 1, authFile);
		fclose(authFile);
	}

	if (ContainsValue("help"))
	{
		PrintUsage();
		return 1;
	}

	if (ContainsValue("client_id") == 0)
	{
		std::cout << "ERROR: You have to supply a client id'\n\n";
		PrintUsage();
		return 1;
	}

	bool hasAuthParams = sVariableMap.count("username") && sVariableMap.count("password") && sVariableMap.count("client_secret");
	bool writeToLocalFile = (sVariableMap.count("output_file") > 0);
	if ( !hasAuthParams && !sHasAuthToken && !writeToLocalFile)
	{
		std::cout << "ERROR: Need to supply atleast one of 'username/password/client_id/client_secret' or 'auth_token' or 'output file'\n\n";
		PrintUsage();
		return 1;
	}

	sUserName = ContainsValue("username") ? LookupString("username") : "";
	std::string password = ContainsValue("password") ? LookupString("password") : "";
	std::string clientId = ContainsValue("client_id") ? LookupString("client_id") : "";
	std::string clientSecret = ContainsValue("client_secret") ? LookupString("client_secret") : "";

	TTV_MemCallbacks memCallbacks;
	memCallbacks.size = sizeof(TTV_MemCallbacks);
	memCallbacks.allocCallback = AllocCallback;
	memCallbacks.freeCallback = FreeCallback;

	TTV_AuthParams authParams;
	if (hasAuthParams)
	{
		authParams.size = sizeof(TTV_AuthParams);
		authParams.userName = sUserName.c_str();
		authParams.password = password.c_str();
		authParams.clientSecret = clientSecret.c_str();
	}

	std::wstring dllLoadPath = GetIntelDllPath();

	TTV_ErrorCode ret = TTV_Init(&memCallbacks, clientId.c_str(), dllLoadPath.c_str());
	ASSERT_ON_ERROR(ret);
	if (TTV_FAILED(ret))
	{
		std::cout << "Failed to initialize the SDK, error code " << ret << " (" << TTV_ErrorToString(ret) << ")" << std::endl;
		exit(-1);
	}

	if (ContainsValue("tracing_file"))
	{
		std::string path = LookupString("tracing_file");
		std::wstring tracingFile (path.begin(), path.end());
		TTV_SetTraceOutput(tracingFile.c_str());
	}
	if (ContainsValue("trace_level"))
	{
		std::string val = LookupString("trace_level");
		TTV_MessageLevel level = TTV_ML_NONE; // Default to none when no other value is found.

		if (val == "TTV_ML_DEBUG") level = TTV_ML_DEBUG;
		else if (val == "TTV_ML_INFO") level = TTV_ML_INFO;
		else if (val == "TTV_ML_WARNING") level = TTV_ML_WARNING;
		else if (val == "TTV_ML_ERROR") level = TTV_ML_ERROR;

		TTV_SetTraceLevel (level);
	}

	TTV_VideoEncoder vidEncoder = TTV_VID_ENC_DEFAULT;
	if (ContainsValue("video_encoder"))
	{
		if (LookupString("video_encoder") == "intel")
		{
			vidEncoder = TTV_VID_ENC_INTEL;
		}
#if SUPPORT_X264
		else if (LookupString("video_encoder") == "plugin-x264")
		{
			sX264Plugin = new X264Plugin;
		}
#endif
		else
		{
			std::cout << "Unknown video encoder " << LookupString("video_encoder") << std::endl;
			exit(-1);
		}
	}

	TTV_AudioEncoder audEncoder = TTV_AUD_ENC_DEFAULT;
	if (ContainsValue("audio_encoder"))
	{
		if (LookupString("audio_encoder").find("mp3") != std::string::npos)
		{
			audEncoder = TTV_AUD_ENC_LAMEMP3;
		}
		else if (LookupString("audio_encoder").find("aac") != std::string::npos)
		{
			audEncoder = TTV_AUD_ENC_APPLEAAC;
		}
		else
		{
			std::cout << "Unknown audio encoder " << LookupString("audio_encoder") << std::endl;
			exit(-1);
		}
	}
	TTV_SetEncoders(vidEncoder, audEncoder);

	//TTV_SetTraceChannelLevel("FrameQueue", TTV_ML_DEBUG);

	if (hasAuthParams)
	{
		if (!sHasAuthToken)
		{
			if (sUsePasswordGrant)
			{
				ret = TTV_RequestAuthToken(&authParams, (TTV_RequestAuthToken_Broadcast | TTV_RequestAuthToken_Chat), AuthDoneCallback, nullptr, &sAuthToken);
			}
			else
			{
				ret = TTV_ImplicitGrantAuthToken(clientId.c_str(), "55542", "http://twitch.tv", "http://darthno.ytmnd.com/", (TTV_RequestAuthToken_Broadcast | TTV_RequestAuthToken_Chat), AuthDoneCallback, nullptr, &sAuthToken);
			}
			ASSERT_ON_ERROR(ret);
		}

		while (!sHasAuthToken)
		{
			TTV_PollTasks();
			ThreadSleep(5);
		}

		// Send the authParams as userData so that the LoginCallback can request a new auth token
		// if login fails
		sChannelInfo.size = sizeof(sChannelInfo);
		ret = TTV_Login(&sAuthToken, LoginCallback, &authParams, &sChannelInfo);
		ASSERT_ON_ERROR(ret);
		while (!sLoggedIn)
		{
			TTV_PollTasks();
			ThreadSleep(5);
		}

		ret = TTV_GetIngestServers(&sAuthToken, IngestListCallback, nullptr, &sIngestList);
		ASSERT_ON_ERROR(ret);
		while (sSelectedIngestIndex < 0)
		{
			TTV_PollTasks();
			ThreadSleep(5);
		}

		sStreamInfo.size = sizeof(sStreamInfo);
		ret = TTV_GetStreamInfo(&sAuthToken, StreamInfoDoneCallback, nullptr, sUserName.c_str(), &sStreamInfo);
		ASSERT_ON_ERROR(ret);

		sUserInfo.size = sizeof(sUserInfo);
		ret = TTV_GetUserInfo(&sAuthToken, UserInfoDoneCallback, nullptr, &sUserInfo);
		ASSERT_ON_ERROR(ret);

		sStreamInfoToSet.size = sizeof(sStreamInfoToSet);
		strcpy(sStreamInfoToSet.streamTitle, "Playing my awesome game with some friends.");
		strcpy(sStreamInfoToSet.gameName, "My awesome game!");
		ret = TTV_SetStreamInfo(&sAuthToken, sUserName.c_str(), &sStreamInfoToSet, SetStreamInfoCallback, nullptr);
		ASSERT_ON_ERROR(ret);

		///////
		// TODO - commented out until the backend is implemented
		// sArchivingState.size = sizeof(sArchivingState);
		// ret = TTV_GetArchivingState(sAuthToken, ArchivingStateCallback, nullptr, &sArchivingState);
		// ASSERT_ON_ERROR(ret);
		/////
	}

	if (sHasAuthToken)
	{
		TTV_GetGameNameList("Star", GameNameListCallback, &sGameInfoList, nullptr);
		TTV_GetGameNameList("a", GameNameListCallback, &sGameInfoList, nullptr);
		TTV_GetGameNameList("b", GameNameListCallback, &sGameInfoList, nullptr);
		TTV_GetGameNameList("c", GameNameListCallback, &sGameInfoList, nullptr);
		TTV_GetGameNameList("d", GameNameListCallback, &sGameInfoList, nullptr);
	}

	std::string chat_test = LookupString("chat_test");

	// setup chat
	if (chat_test != "none")
	{
		TestChatInit();
		TestChatConnect(sUserName.c_str(), sAuthToken);
	}

#if SCREEN_CAPTURE && !TTV_PLATFORM_WINDOWS
	//   TODO - doing screen capture on mac is different
	//
	//  // Start ObjC runloop and capture frames
	//    NSAutoreleasePool *pool = [[NSAutoreleasePool alloc] init];
	//
	//  capture = [[TTVAVCapture alloc] init];
	//
	//  [capture start];
	//
	//  while (YES) {
	//    [[NSRunLoop currentRunLoop] runMode:NSDefaultRunLoopMode beforeDate:[NSDate distantFuture]];
	//  }
	//
	//  [pool drain];
#endif

#if TTV_PLATFORM_IOS
	@autoreleasepool {
		return UIApplicationMain(argc, argv, NULL, @"AppDelegate");
	}
#else
	InitializeWebcam();
	StartStream();

	return 0;
#endif
}
