#include "../sdktester.h"
#include <string>
#include <assert.h>
#include <windows.h>

extern "C" void TTV_SetOutputFileName(const wchar_t*);

void SetOutputFileName(const std::string& filename)
{
	const std::wstring outputFilename (filename.begin(), filename.end());
	TTV_SetOutputFileName(outputFilename.c_str());
}

std::wstring GetIntelDllPath()
{
	const int kFullPathLen = 512;
	wchar_t curDir[kFullPathLen];
	GetFullPathNameW(L".", kFullPathLen, curDir, 0);
#ifdef _M_X64
	return std::wstring(std::wstring(curDir) + L"\\..\\twitchsdk\\intel\\bin\\x64");
#else
	return std::wstring(std::wstring(curDir) + L"\\..\\twitchsdk\\intel\\bin\\win32");
#endif
}

void GenerateBGRAFrame(uint8_t* buffer, int width, int height)
{
	assert(buffer);
	assert(width > 0);
	assert(height > 0);
	
#if SCREEN_CAPTURE
	HWND hwnd = GetDesktopWindow();
	RECT rcClient;
	GetClientRect(hwnd, &rcClient);
	POINT topLeft = {rcClient.left, rcClient.top};
	POINT bottomRight = {rcClient.right, rcClient.bottom};
	ClientToScreen(hwnd, &topLeft);
	ClientToScreen(hwnd, &bottomRight);
	
	int srcHeight = bottomRight.y - topLeft.y;
	int srcWidth = bottomRight.x - topLeft.x;
	
	int bitmapBufferSize = width * height * 4;
	
	HDC hDC = GetDC(hwnd);
	assert(hDC);

	HDC hCaptureDC = CreateCompatibleDC(hDC);
	assert(hCaptureDC);

	HBITMAP hCaptureBitmap = CreateCompatibleBitmap(hDC, width, height);
	assert(hCaptureBitmap);

	HBITMAP hOldBitmap = (HBITMAP)SelectObject(hCaptureDC,hCaptureBitmap);
	assert(hOldBitmap);
	
	SetStretchBltMode(hCaptureDC, HALFTONE);
	
	StretchBlt(hCaptureDC, 0, 0, width, height, hDC, 0, 0, srcWidth, srcHeight, SRCCOPY);
	
	GetBitmapBits(hCaptureBitmap, bitmapBufferSize, buffer);
	
	SelectObject(hCaptureDC, hOldBitmap);
	DeleteObject(hCaptureBitmap);
	DeleteDC(hCaptureDC);
	ReleaseDC(hwnd, hDC);

#else
	static int i = 0;
	
	const int kPixelBytes = 4;
	
	// Every 10 frames switch between red and green
	unsigned char R = (i % 30 < 10) ? 0xFF : 0x00;
	unsigned char G = (i % 30 >= 10 && i % 30 < 20) ? 0xFF : 0x00;
	unsigned char B = (i % 30 > 20) ? 0xFF : 0x00;
	unsigned char A = 0x00;
	
	for (int y = 0; y < height; ++y)
	{
		for (int x = 0; x < width; ++x)
		{
			buffer[y*width*kPixelBytes + x*kPixelBytes + 0] = B;
			buffer[y*width*kPixelBytes + x*kPixelBytes + 1] = G;
			buffer[y*width*kPixelBytes + x*kPixelBytes + 2] = R;
			buffer[y*width*kPixelBytes + x*kPixelBytes + 3] = A;
		}
	}
	++i;

#endif
	
}

void ThreadSleep(uint32_t ms)
{
	Sleep(ms);
}

void* AllocCallback(size_t size, size_t alignment)
{
#if ENABLE_MEMORY_LEAK_DEBUGGING
	//_CrtSetBreakAlloc(1668);		// put the allocation number here from the report spit out on the console
#endif
	return _aligned_malloc(size, alignment);
}

void FreeCallback(void* ptr)
{
	_aligned_free(ptr);
}

uint64_t GetSystemClockFrequency()
{
	static uint64_t frequency = 0;
	if (frequency == 0)
	{
		QueryPerformanceFrequency( reinterpret_cast<LARGE_INTEGER*>(&frequency) );
	}
	
	return frequency;
}

uint64_t GetSystemClockTime()
{
	uint64_t counter;
	QueryPerformanceCounter( reinterpret_cast<LARGE_INTEGER*>(&counter) );
	return counter;
}

uint64_t SystemTimeToMs(uint64_t sysTime)
{
	return sysTime * 1000 / GetSystemClockFrequency();
}