/********************************************************************************************
* Twitch Broadcasting SDK
*
* This software is supplied under the terms of a license agreement with Twitch Interactive, Inc. and
* may not be copied or used except in accordance with the terms of that agreement
* Copyright (c) 2012-2016 Twitch Interactive, Inc.
*********************************************************************************************/
#include "gtest/gtest.h"
#include "targa.h"


struct TargaHeader
{
	unsigned char identsize;          // size of ID field that follows 18 byte header (0 usually)
	unsigned char colourmaptype;      // type of colour map 0=none, 1=has palette
	unsigned char imagetype;          // type of image 0=none,1=indexed,2=rgb,3=grey,+8=rle packed

	unsigned short colourmapstart;     // first colour map entry in palette
	unsigned short colourmaplength;    // number of colours in palette
	unsigned char colourmapbits;      // number of bits per palette entry 15,16,24,32

	unsigned short xstart;             // image x origin
	unsigned short ystart;             // image y origin
	unsigned short width;              // image width in pixels
	unsigned short height;             // image height in pixels
	unsigned char bpp;               // image bits per pixel 8,16,24,32
	unsigned char descriptor;         // image descriptor bits (vh flip bits)
    
	// pixel data follows header
    
};


void ReadTarga(const std::string& path, int& width, int& height, uint8_t*& buffer)
{
	FILE* file = nullptr;
	fopen_s(&file, path.c_str(), "rb");

	if (file == nullptr)
	{
		width = 0;
		height = 0;
		buffer = nullptr;

		return;
	}

	TargaHeader header;

	fread(&header.identsize, sizeof(header.identsize), 1, file);
	fread(&header.colourmaptype, sizeof(header.colourmaptype), 1, file);
	fread(&header.imagetype, sizeof(header.imagetype), 1, file);
	fread(&header.colourmapstart, sizeof(header.colourmapstart), 1, file);
	fread(&header.colourmaplength, sizeof(header.colourmaplength), 1, file);
	fread(&header.colourmapbits, sizeof(header.colourmapbits), 1, file);
	fread(&header.xstart, sizeof(header.xstart), 1, file);
	fread(&header.ystart, sizeof(header.ystart), 1, file);
	fread(&header.width, sizeof(header.width), 1, file);
	fread(&header.height, sizeof(header.height), 1, file);
	fread(&header.bpp, sizeof(header.bpp), 1, file);
	fread(&header.descriptor, sizeof(header.descriptor), 1, file);

	ASSERT_EQ(header.imagetype, 2);
	ASSERT_EQ(header.bpp, 32);

	width = header.width;
	height = header.height;

	int bufferSize = width * height * 4;
	buffer = new uint8_t[bufferSize];

	for (int y=0; y<height; ++y)
	{
		fread(&buffer[(height-y-1)*width*4], width*4, 1, file);
	}

	fclose(file);
}

void WriteTarga(const std::string& path, int width, int height, const uint8_t* buffer)
{
	FILE* file = nullptr;
	fopen_s(&file, path.c_str(), "wb");

	if (file == nullptr)
	{
		return;
	}

	TargaHeader header;

	header.identsize = 0;
	header.colourmaptype = 0;
	header.imagetype = 2; // RGB uncompressed
	header.colourmapstart = 0;
	header.colourmaplength = 0;
	header.colourmapbits = 0;
	header.xstart = 0;
	header.ystart = 0;
	header.width = width;
	header.height = height;
	header.bpp = 32;
	header.descriptor = 0;

	fwrite(&header.identsize, sizeof(header.identsize), 1, file);
	fwrite(&header.colourmaptype, sizeof(header.colourmaptype), 1, file);
	fwrite(&header.imagetype, sizeof(header.imagetype), 1, file);
	fwrite(&header.colourmapstart, sizeof(header.colourmapstart), 1, file);
	fwrite(&header.colourmaplength, sizeof(header.colourmaplength), 1, file);
	fwrite(&header.colourmapbits, sizeof(header.colourmapbits), 1, file);
	fwrite(&header.xstart, sizeof(header.xstart), 1, file);
	fwrite(&header.ystart, sizeof(header.ystart), 1, file);
	fwrite(&header.width, sizeof(header.width), 1, file);
	fwrite(&header.height, sizeof(header.height), 1, file);
	fwrite(&header.bpp, sizeof(header.bpp), 1, file);
	fwrite(&header.descriptor, sizeof(header.descriptor), 1, file);

	for (int y=0; y<height; ++y)
	{
		fwrite(&buffer[(height-y-1)*width*4], width*4, 1, file);
	}

	fclose(file);
}
