/********************************************************************************************
* Twitch Broadcasting SDK
*
* This software is supplied under the terms of a license agreement with Twitch Interactive, Inc. and
* may not be copied or used except in accordance with the terms of that agreement
* Copyright (c) 2012-2016 Twitch Interactive, Inc.
*********************************************************************************************/

#include "gtest/gtest.h"
#include "twitchsdk.h"
#include "twitch_asserts.h"
#include "fixtures/encoder.h"

namespace
{
	void GenerateBGRAFrame(uint8_t* buffer, int width, int height)
	{
		static int i = 0;
	
		const int kPixelBytes = 4;
	
		// Every 10 frames switch between red and green
		unsigned char R = (i % 30 < 10) ? 0xFF : 0x00;
		unsigned char G = (i % 30 >= 10 && i % 30 < 20) ? 0xFF : 0x00;
		unsigned char B = (i % 30 > 20) ? 0xFF : 0x00;
		unsigned char A = 0x00;
	
		for (int y = 0; y < height; ++y)
		{
			for (int x = 0; x < width; ++x)
			{
				buffer[y*width*kPixelBytes + x*kPixelBytes + 0] = B;
				buffer[y*width*kPixelBytes + x*kPixelBytes + 1] = G;
				buffer[y*width*kPixelBytes + x*kPixelBytes + 2] = R;
				buffer[y*width*kPixelBytes + x*kPixelBytes + 3] = A;
			}
		}
		++i;
	}

	void FrameUnlockCallback (const uint8_t* buffer, void* /*userData*/)
	{
		delete[] buffer;
	}
}

extern "C" void TTV_SetOutputFileName(const wchar_t*);
extern "C" void TTV_SetEncoders(TTV_VideoEncoder vidEncoder, TTV_AudioEncoder audEncoder);

TEST_P(EncoderTest, StreamsToFileOk)
{
	TTV_VideoEncoder encoder = GetParam();
	TTV_SetEncoders(encoder, TTV_AUD_ENC_DEFAULT);

	TTV_VideoParams videoParams = {sizeof(TTV_VideoParams)};
	videoParams.outputWidth = 1024;
	videoParams.outputHeight = 768;
	videoParams.targetFps = 30;
	TTV_GetDefaultParams(&videoParams);
	videoParams.pixelFormat = TTV_PF_BGRA;
	
	TTV_AudioParams audioParams;
	audioParams.size = sizeof(TTV_AudioParams);
	audioParams.audioEnabled = false;
	audioParams.enableMicCapture = false;
	audioParams.enablePlaybackCapture = false;
	audioParams.enablePassthroughAudio = false;

	TTV_SetOutputFileName(L"sdktest.flv");
	
	res = TTV_Start(&videoParams, &audioParams, nullptr, 0, nullptr, nullptr);
	ASSERT_TTV_SUCCEEDED(res);

	for (time_t start = time(NULL); time(NULL) < start + 5;)
	{
		uint8_t* frameData = new uint8_t[videoParams.outputWidth * videoParams.outputHeight * 4];
		GenerateBGRAFrame(frameData, videoParams.outputWidth, videoParams.outputHeight);
		res = TTV_SubmitVideoFrame(frameData, FrameUnlockCallback, 0);
		if (TTV_FAILED(res))
			break;
	}
	ASSERT_TTV_SUCCEEDED(res); // no need to assert 1k times, only assert on failure.

	res = TTV_Stop(nullptr, nullptr);
	ASSERT_TTV_SUCCEEDED(res);

	// now see that the file is an appropriate size. As yet we have no real way of knowing if
	// the data is correct, we just want to see if it went somewhere.
	// Just less than 5s of video seems to reliably produce about 1000kB of data, so we give a
	// nice wide berth around that to allow for specific timings.
	struct stat stats = {0};
	stat("sdktest.flv", &stats);
	ASSERT_GT(stats.st_size, 750*1024);
	ASSERT_LT(stats.st_size, 1250*1024);

	unlink("sdktest.flv");
}

#if SUPPORT_INTEL_ENCODER
INSTANTIATE_TEST_CASE_P(IntelEncoder, EncoderTest, testing::Values(TTV_VID_ENC_INTEL));
#endif
#if SUPPORT_X264_ENCODER
INSTANTIATE_TEST_CASE_P(X264Encoder, EncoderTest, testing::Values(TTV_VID_ENC_X264));
#endif
#if SUPPORT_APPLE_ENCODER
INSTANTIATE_TEST_CASE_P(AppleEncoder, EncoderTest, testing::Values(TV_VID_ENC_APPLE));
#endif
