/********************************************************************************************
* Twitch Broadcasting SDK
*
* This software is supplied under the terms of a license agreement with Twitch Interactive, Inc. and
* may not be copied or used except in accordance with the terms of that agreement
* Copyright (c) 2012-2016 Twitch Interactive, Inc.
*********************************************************************************************/

#include "twitchsdk/chat/internal/pch.h"
#include "twitchsdk/chat/internal/bindings/csharp/csharp_chat.h"
#include "twitchsdk/chat/internal/chathelpers.h"
#include "twitchsdk/core/stringutilities.h"
#include <functional>

namespace
{
	using namespace ttv;
	using namespace ttv::chat;

	class NativeChatAPIListener;
	class NativeChatChannelListener;

	struct LocalContext
	{
		std::unique_ptr<ttv::chat::ChatAPI> chatApi;
		std::shared_ptr<NativeChatAPIListener> chatAPIListener;
		std::shared_ptr< std::map<std::string, std::shared_ptr<NativeChatChannelListener>> > chatChannelListeners;

		LocalContext()
		{
			chatChannelListeners = std::make_shared<std::map<std::string, std::shared_ptr<NativeChatChannelListener>>>();
		}
	};

	std::shared_ptr<LocalContext> gLocalContext; // The container of all objects required in this module.


	template<typename T>
	struct TTV_CallbackInfo
	{
		T callback;
		void* userdata;
	};

	class NativeChatAPIListener : public IChatAPIListener
	{
	public:
		NativeChatAPIListener(const ManagedChatAPIListener* apiListener)
		:	mWaitingForInitialization(false)
		,	mWaitingForShutdown(false)
		{
			assert(apiListener != nullptr);

			mManagedApiListener = *apiListener;
		}

		virtual void ChatInitializationCallback(TTV_ErrorCode result)
		{
			assert(mWaitingForInitialization);

			ClearInitializationCallback();

			// notify the client
			mManagedApiListener.FireInitializationCallback(result);

			// if init fails we need to clear the globals
			if (TTV_FAILED(result))
			{
				gLocalContext.reset();
			}
		}

		virtual void ChatShutdownCallback(TTV_ErrorCode result)
		{
			assert(mWaitingForShutdown);

			ClearShutdownCallback();

			// notify the client
			mManagedApiListener.FireShutdownCallback(result);

			// if the shutdown succeeds then clear globals
			if (TTV_SUCCEEDED(result))
			{
				gLocalContext.reset();
			}
		}

		virtual void ChatUserEmoticonSetsChangedCallback(const TTV_ChatUserEmoticonSets* emoticonSets)
		{
			mManagedApiListener.FireUserEmoticonSetsChangedCallback(emoticonSets);
		}

		virtual void ChatEmoticonSetDataCallback(const TTV_ChatEmoticonSetData* emoticonSet)
		{
			mManagedApiListener.FireEmoticonSetDataCallback(emoticonSet);
		}

		virtual void ChatFetchBitsConfigurationCallback(TTV_ErrorCode /*ec*/) override
		{
			// TODO: Add support if we want it
		}

		virtual void ChatUserBlockChangeCallback(const std::string& userName, const std::string& blockUsername, bool block, TTV_ErrorCode ec)
		{
			mManagedApiListener.FireUserBlockChangeCallback(userName, blockUsername, block, ec);
		}
		
		virtual void ChatUserIdLookupCallback(const std::string& /*userName*/, UserId /*userId*/, TTV_ErrorCode /*ec*/)
		{
			// TODO: Add support if we want it
		}

		virtual void ChatGenerateThreadIdCallback(const std::string& /*userName*/, const std::string& /*otherUserName*/, const std::string& /*threadId*/, TTV_ErrorCode /*ec*/)
		{
			// TODO: Add support if we want it
		}

		void RegisterInitializationCallback()
		{
			assert(!mWaitingForInitialization);
		
			mWaitingForInitialization = true;
		}
		void ClearInitializationCallback()
		{
			mWaitingForInitialization = false;
		}

		void RegisterShutdownCallback()
		{
			assert(!mWaitingForShutdown);
		
			mWaitingForShutdown = true;
		}
		void ClearShutdownCallback()
		{
			mWaitingForShutdown = false;
		}

		bool IsWaitingForInitializationCallback() const { return mWaitingForInitialization; }
		bool IsWaitingForShutdownCallback() const { return mWaitingForShutdown; }

	private:
		ManagedChatAPIListener mManagedApiListener;

		bool mWaitingForInitialization;
		bool mWaitingForShutdown;
	};

	class NativeChatChannelListener : public IChatChannelListener
	{
	public:
		NativeChatChannelListener(const ManagedChatChannelListener* channelListener)
		{
			mManagedChannelListener = *channelListener;
		}

		~NativeChatChannelListener()
		{
		}

		void ReleaseListener(const std::string& channelName)
		{
			auto iter = gLocalContext->chatChannelListeners->find(channelName);
			if (iter != gLocalContext->chatChannelListeners->end())
			{
				gLocalContext->chatChannelListeners->erase(iter);
			}
		}

		virtual void ChatChannelStateChangedCallback(const std::string& /*userName*/, const std::string& channelName, TTV_ChatChannelState state, TTV_ErrorCode ec)
		{
			mManagedChannelListener.FireChannelStateChangedCallback(channelName, state, ec);

			switch (state)
			{
			case TTV_CHAT_CHANNEL_STATE_DISCONNECTED:
				ReleaseListener(channelName);
				break;
			default:
				break;
			}
		}
		
		virtual void ChatChannelLocalUserChangedCallback(const std::string& /*userName*/, const std::string& channelName, const TTV_ChatUserInfo& userInfo)
		{
			mManagedChannelListener.FireChannelLocalUserChangedCallback(channelName, userInfo);
		}

		virtual void ChatChannelUserChangeCallback(const std::string& /*userName*/, const std::string& channelName, const TTV_ChatUserList* joinList, const TTV_ChatUserList* leaveList, const TTV_ChatUserList* infoChangeList)
		{
			mManagedChannelListener.FireChannelUserChangeCallback(channelName, joinList, leaveList, infoChangeList);

			// It is assumed that the managed wrapper will free the user lists when it's done with them
		}

		virtual void ChatChannelInfoChangedCallback(const std::string& /*userName*/, const std::string& channelName, const TTV_ChatChannelInfo& info)
		{
			mManagedChannelListener.FireChannelInfoChangedCallback(channelName, info);
		}

		virtual void ChatChannelMessageCallback(const std::string& /*userName*/, const std::string& channelName, const TTV_ChatMessageList* messageList)
		{
			mManagedChannelListener.FireChannelMessageCallback(channelName, messageList);
			
			// It is assumed that the managed wrapper will free the message list when it's done with it
		}

		virtual void ChatChannelClearCallback(const std::string& /*userName*/, const std::string& channelName, const std::string& clearUserName)
		{
			mManagedChannelListener.FireChannelClearCallback(channelName, clearUserName);
		}

		virtual void ChatChannelHostTargetChangedCallback(const std::string& /*userName*/, const std::string& channelName, const std::string& targetChannel, uint32_t numViewers)
		{
			mManagedChannelListener.FireChannelHostTargetChangedCallback(channelName, targetChannel, numViewers);
		}

		virtual void ChatChannelNoticeCallback(const std::string& /*userName*/, const std::string& channelName, const std::string& id, const std::map<std::string, std::string>& params)
		{
			mManagedChannelListener.FireChannelNoticeCallback(channelName, id, params);
		}

		virtual void ChatChannelSetBroadcasterLanguageCallback(const std::string& /*userName*/, const std::string& channelName, TTV_ErrorCode ec)
		{
			mManagedChannelListener.FireChannelSetBroadcasterLanguageCallback(channelName, ec);
		}

		virtual void ChatChannelBadgeDataDownloadCallback(const std::string& channelName, TTV_ErrorCode result)
		{
			mManagedChannelListener.FireChannelBadgeDataDownloadCallback(channelName, result);
		}

	private:
		ManagedChatChannelListener mManagedChannelListener;
	};
}


extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Chat_Initialize(TTV_ChatTokenizationOption tokenizationOptions, const ManagedChatAPIListener* chatAPIListener)
{
	TTV_RETURN_ON_NOT_NULL(gLocalContext, TTV_EC_ALREADY_INITIALIZED);
	TTV_RETURN_ON_NULL(chatAPIListener, TTV_EC_INVALID_ARG);
	
	TTV_ErrorCode ec = TTV_EC_UNKNOWN_ERROR;

	// create the container for the chat module globals
	gLocalContext = std::make_shared<LocalContext>();

	gLocalContext->chatApi = std::make_unique<ttv::chat::ChatAPI>();
	assert(gLocalContext->chatApi);

	if (gLocalContext->chatApi == nullptr)
	{
		ec = TTV_EC_MEMORY;
	}
	else
	{
		gLocalContext->chatAPIListener = std::make_shared<NativeChatAPIListener>(chatAPIListener);
		gLocalContext->chatAPIListener->RegisterInitializationCallback();

		// TODO: This is broken, fix the C# bindings someday

		ec = gLocalContext->chatApi->Initialize(nullptr, tokenizationOptions, gLocalContext->chatAPIListener);
	}

	if (TTV_FAILED(ec))
	{
		gLocalContext.reset();
	}

	return ec;
}


extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Chat_Shutdown()
{
	TTV_RETURN_ON_NULL(gLocalContext, TTV_EC_NOT_INITIALIZED);
	TTV_RETURN_ON_NULL(gLocalContext->chatApi, TTV_EC_NOT_INITIALIZED);
	TTV_RETURN_ON_NULL(gLocalContext->chatAPIListener, TTV_EC_NOT_INITIALIZED);

	if (gLocalContext->chatAPIListener->IsWaitingForInitializationCallback())
	{
		return TTV_EC_NOT_INITIALIZED;
	}
	else if (gLocalContext->chatAPIListener->IsWaitingForShutdownCallback())
	{
		return TTV_EC_SHUTTING_DOWN;
	}

	gLocalContext->chatAPIListener->RegisterShutdownCallback();

	TTV_ErrorCode ec = gLocalContext->chatApi->Shutdown();
	if (TTV_FAILED(ec))
	{
		gLocalContext->chatAPIListener->ClearShutdownCallback();
	}

	return ec;
}

//
//extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Chat_LogIn(const utf8char* userName, const utf8char* oauthToken)
//{
//	TTV_RETURN_ON_NULL(gLocalContext, TTV_EC_NOT_INITIALIZED);
//	TTV_RETURN_ON_NULL(gLocalContext->chatApi, TTV_EC_NOT_INITIALIZED);
//
//	TTV_ErrorCode ec = gLocalContext->chatApi->LogIn(userName, oauthToken);
//	return ec;
//}
//
//
//extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Chat_LogOut(const utf8char* userName)
//{
//	TTV_RETURN_ON_NULL(gLocalContext, TTV_EC_NOT_INITIALIZED);
//	TTV_RETURN_ON_NULL(gLocalContext->chatApi, TTV_EC_NOT_INITIALIZED);
//
//	TTV_ErrorCode ec = gLocalContext->chatApi->LogOut(userName);
//	return ec;
//}


extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Chat_Connect(const utf8char* channelName, const utf8char* userName, const ManagedChatChannelListener* chatChannelListener)
{
	TTV_RETURN_ON_NULL(gLocalContext, TTV_EC_NOT_INITIALIZED);
	TTV_RETURN_ON_NULL(gLocalContext->chatApi, TTV_EC_NOT_INITIALIZED);
	TTV_RETURN_ON_NULL(gLocalContext->chatAPIListener, TTV_EC_NOT_INITIALIZED);
	TTV_RETURN_ON_DIFFERENT(IsValidUserName(userName), true, TTV_EC_INVALID_ARG);
	TTV_RETURN_ON_DIFFERENT(IsValidChannelName(channelName), true, TTV_EC_INVALID_ARG);
	TTV_RETURN_ON_NULL(chatChannelListener, TTV_EC_INVALID_ARG);

	std::shared_ptr<NativeChatChannelListener> nativeChannelListener;

	auto iter = gLocalContext->chatChannelListeners->find(channelName);
	if (iter != gLocalContext->chatChannelListeners->end())
	{
		nativeChannelListener = iter->second;
	}
	else
	{
		nativeChannelListener = std::make_shared<NativeChatChannelListener>(chatChannelListener);
	}

	TTV_ErrorCode ec = gLocalContext->chatApi->Connect(channelName, userName, nativeChannelListener);
	if (TTV_SUCCEEDED(ec))
	{
		if (iter == gLocalContext->chatChannelListeners->end())
		{
			gLocalContext->chatChannelListeners->insert( std::pair<std::string, std::shared_ptr<NativeChatChannelListener>>(channelName, nativeChannelListener) );
		}
	}

	return ec;
}


extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Chat_Disconnect(const utf8char* userName, const utf8char* channelName)
{
	TTV_RETURN_ON_NULL(gLocalContext, TTV_EC_NOT_INITIALIZED);
	TTV_RETURN_ON_NULL(gLocalContext->chatApi, TTV_EC_NOT_INITIALIZED);
	TTV_RETURN_ON_NULL(gLocalContext->chatAPIListener, TTV_EC_NOT_INITIALIZED);

	// NOTE: gLocalContext->chatChannelListener will be cleared by the listener during a disconnection event

	TTV_ErrorCode ec = gLocalContext->chatApi->Disconnect(userName, channelName);
	return ec;
}


extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Chat_SendMessage(const utf8char* userName, const utf8char* channelName, const utf8char* message)
{
	TTV_RETURN_ON_NULL(gLocalContext, TTV_EC_NOT_INITIALIZED);
	TTV_RETURN_ON_NULL(gLocalContext->chatApi, TTV_EC_NOT_INITIALIZED);
	TTV_RETURN_ON_NULL(gLocalContext->chatAPIListener, TTV_EC_NOT_INITIALIZED);

	TTV_ErrorCode ec = gLocalContext->chatApi->SendChatMessage(userName, channelName, message);
	return ec;
}


extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Chat_BlockUser(const utf8char* userName, const utf8char* blockUserName, const utf8char* reason, bool whisper)
{
	TTV_RETURN_ON_NULL(gLocalContext, TTV_EC_NOT_INITIALIZED);
	TTV_RETURN_ON_NULL(gLocalContext->chatApi, TTV_EC_NOT_INITIALIZED);
	TTV_RETURN_ON_NULL(gLocalContext->chatAPIListener, TTV_EC_NOT_INITIALIZED);

	TTV_ErrorCode ec = gLocalContext->chatApi->BlockUser(userName, blockUserName, reason, whisper);
	return ec;
}


extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Chat_UnblockUser(const utf8char* userName, const utf8char* blockUserName)
{
	TTV_RETURN_ON_NULL(gLocalContext, TTV_EC_NOT_INITIALIZED);
	TTV_RETURN_ON_NULL(gLocalContext->chatApi, TTV_EC_NOT_INITIALIZED);
	TTV_RETURN_ON_NULL(gLocalContext->chatAPIListener, TTV_EC_NOT_INITIALIZED);

	TTV_ErrorCode ec = gLocalContext->chatApi->UnblockUser(userName, blockUserName);
	return ec;
}


extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Chat_GetUserBlocked(const utf8char* userName, const utf8char* blockUserName, bool* blocked)
{
	TTV_RETURN_ON_NULL(gLocalContext, TTV_EC_NOT_INITIALIZED);
	TTV_RETURN_ON_NULL(gLocalContext->chatApi, TTV_EC_NOT_INITIALIZED);
	TTV_RETURN_ON_NULL(gLocalContext->chatAPIListener, TTV_EC_NOT_INITIALIZED);
	TTV_RETURN_ON_NULL(blocked, TTV_EC_INVALID_ARG);

	TTV_ErrorCode ec = gLocalContext->chatApi->GetUserBlocked(userName, blockUserName, *blocked);
	return ec;
}


extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Chat_FlushEvents()
{
	TTV_RETURN_ON_NULL(gLocalContext, TTV_EC_NOT_INITIALIZED);
	TTV_RETURN_ON_NULL(gLocalContext->chatApi, TTV_EC_NOT_INITIALIZED);

	// keep a reference to the local context until all callbacks are fired in case a callback wants to shutdown
	auto keepContextAlive = gLocalContext;

	// keep a reference to the local listeners until all callbacks are fired in case a callback wants to disconnect
	auto keepChannelListenersAlive = gLocalContext->chatChannelListeners;

	TTV_ErrorCode ec = gLocalContext->chatApi->Update();
	return ec;
}


extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Chat_ForceUserListUpdate(const utf8char* userName, const utf8char* channelName)
{
	TTV_RETURN_ON_NULL(gLocalContext, TTV_EC_NOT_INITIALIZED);
	TTV_RETURN_ON_NULL(gLocalContext->chatApi, TTV_EC_NOT_INITIALIZED);
	TTV_RETURN_ON_NULL(gLocalContext->chatAPIListener, TTV_EC_NOT_INITIALIZED);

	TTV_ErrorCode ec = gLocalContext->chatApi->ForceUserListUpdate(userName, channelName);
	return ec;
}


extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Chat_GetBadgeData(const utf8char* channelName, const TTV_ChatBadgeData** badgeData)
{
	TTV_RETURN_ON_NULL(gLocalContext, TTV_EC_NOT_INITIALIZED);
	TTV_RETURN_ON_NULL(gLocalContext->chatApi, TTV_EC_NOT_INITIALIZED);
	TTV_RETURN_ON_NULL(channelName, TTV_EC_INVALID_ARG);
	TTV_RETURN_ON_NULL(badgeData, TTV_EC_INVALID_ARG);

	auto iter = gLocalContext->chatChannelListeners->find(channelName);
	if (iter == gLocalContext->chatChannelListeners->end())
	{
		return TTV_EC_CHAT_NOT_IN_CHANNEL;
	}

	*badgeData = nullptr;
	TTV_ErrorCode ec = gLocalContext->chatApi->GetBadgeData(channelName, badgeData);
	return ec;
}


extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Chat_FreeBadgeData(TTV_ChatBadgeData* badgeData)
{
	TTV_RETURN_ON_NULL(gLocalContext, TTV_EC_NOT_INITIALIZED);
	TTV_RETURN_ON_NULL(gLocalContext->chatApi, TTV_EC_NOT_INITIALIZED);
	
	TTV_ErrorCode ec = gLocalContext->chatApi->FreeBadgeData(badgeData);
	return ec;
}


extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Chat_FreeUserList(TTV_ChatUserList* list)
{
	TTV_RETURN_ON_NULL(gLocalContext, TTV_EC_NOT_INITIALIZED);
	TTV_RETURN_ON_NULL(gLocalContext->chatApi, TTV_EC_NOT_INITIALIZED);
	
	TTV_ErrorCode ec = gLocalContext->chatApi->FreeUserList(list);
	return ec;
}
extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Chat_FreeUserList_Pointer(void* list)
{
	return TTV_CSharp_Chat_FreeUserList(reinterpret_cast<TTV_ChatUserList*>(list));
}


extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Chat_FreeMessageList(TTV_ChatMessageList* list)
{
	TTV_RETURN_ON_NULL(gLocalContext, TTV_EC_NOT_INITIALIZED);
	TTV_RETURN_ON_NULL(gLocalContext->chatApi, TTV_EC_NOT_INITIALIZED);
	
	TTV_ErrorCode ec = gLocalContext->chatApi->FreeMessageList(list);
	return ec;
}
extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Chat_FreeMessageList_Pointer(void* list)
{
	return TTV_CSharp_Chat_FreeMessageList(reinterpret_cast<TTV_ChatMessageList*>(list));
}


extern "C" EXPORT_API uint64_t TTV_CSharp_Chat_GetMessageFlushInterval()
{
	if (gLocalContext == nullptr || gLocalContext->chatApi == nullptr)
	{
		return 0;
	}

	return gLocalContext->chatApi->GetMessageFlushInterval();
}


extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Chat_SetMessageFlushInterval(uint64_t milliseconds)
{
	TTV_RETURN_ON_NULL(gLocalContext, TTV_EC_NOT_INITIALIZED);
	TTV_RETURN_ON_NULL(gLocalContext->chatApi, TTV_EC_NOT_INITIALIZED);

	gLocalContext->chatApi->SetMessageFlushInterval(milliseconds);

	return TTV_EC_SUCCESS;
}


extern "C" EXPORT_API uint64_t TTV_CSharp_Chat_GetUserListUpdateInterval()
{
	if (gLocalContext == nullptr || gLocalContext->chatApi == nullptr)
	{
		return 0;
	}

	return gLocalContext->chatApi->GetUserListUpdateInterval();
}


extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Chat_SetUserListUpdateInterval(uint64_t milliseconds)
{
	TTV_RETURN_ON_NULL(gLocalContext, TTV_EC_NOT_INITIALIZED);
	TTV_RETURN_ON_NULL(gLocalContext->chatApi, TTV_EC_NOT_INITIALIZED);

	gLocalContext->chatApi->SetUserListUpdateInterval(milliseconds);

	return TTV_EC_SUCCESS;
}


extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Chat_SetBroadcasterLanguageChatEnabled(const utf8char* userName, const utf8char* channelName, bool enabled)
{
	TTV_RETURN_ON_NULL(gLocalContext, TTV_EC_NOT_INITIALIZED);
	TTV_RETURN_ON_NULL(gLocalContext->chatApi, TTV_EC_NOT_INITIALIZED);

	return gLocalContext->chatApi->SetBroadcasterLanguageChatEnabled(userName, channelName, enabled);
}


extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Chat_OptInToBroadcasterLanguageChat(const utf8char* userName, const utf8char* channelName, const utf8char* language)
{
	TTV_RETURN_ON_NULL(gLocalContext, TTV_EC_NOT_INITIALIZED);
	TTV_RETURN_ON_NULL(gLocalContext->chatApi, TTV_EC_NOT_INITIALIZED);

	return gLocalContext->chatApi->OptInToBroadcasterLanguageChat(userName, channelName, language);
}


extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Chat_SetLocalLanguage(const utf8char* language)
{
	TTV_RETURN_ON_NULL(gLocalContext, TTV_EC_NOT_INITIALIZED);
	TTV_RETURN_ON_NULL(gLocalContext->chatApi, TTV_EC_NOT_INITIALIZED);
	
	return gLocalContext->chatApi->SetLocalLanguage(language);
}
