/********************************************************************************************
* Twitch Broadcasting SDK
*
* This software is supplied under the terms of a license agreement with Twitch Interactive, Inc. and
* may not be copied or used except in accordance with the terms of that agreement
* Copyright (c) 2012-2016 Twitch Interactive, Inc.
*********************************************************************************************/

#include "twitchsdk/chat/internal/pch.h"
#include "twitchsdk/chat/internal/bindings/csharp/csharp_chat.h"
#include "twitchsdk/core/json/value.h"
#include "twitchsdk/core/json/writer.h"

#if TTV_PLATFORM_UNITYIOS

using namespace ttv::Json;

#ifdef __cplusplus
extern "C" 
{
#endif
	void UnitySendMessage(const char* obj, const char* method, const char* msg);
#ifdef __cplusplus
}
#endif


namespace
{
	const char* kUnityCallbackObjectName = "Chat";
}


// Callbacks in Unity-iOS work very differently than in regular .Net and Unity.
// Unity on iOS does not have full support for callbacks from native to managed.  However, it does allow you to send a 
// message to a specifically named game object which is what we will use to pass callback data.
//
// Here we're passing a pointer address to a struct containing the callback data as a string.  In managed you then
// marshal the pointer value to the expected result type and pull out values that way.  The problem is that the message
// won't be executed in managed synchronously so there will need to be a function to free the callback result.
// Maybe also encode in json for simple types


void ttv::chat::ManagedChatAPIListener::FireInitializationCallback(TTV_ErrorCode ec)
{
	Value root;
	root["ec"] = Value(static_cast<int64_t>(ec));
			
	FastWriter writer;
	std::string json = writer.write(root);
	
	UnitySendMessage(kUnityCallbackObjectName, "ProxyInitializationCallback", json.c_str());
}

void ttv::chat::ManagedChatAPIListener::FireShutdownCallback(TTV_ErrorCode ec)
{
	Value root;
	root["ec"] = Value(static_cast<int64_t>(ec));
			
	FastWriter writer;
	std::string json = writer.write(root);

	UnitySendMessage(kUnityCallbackObjectName, "ProxyShutdownCallback", json.c_str());
}

void ttv::chat::ManagedChatAPIListener::FireUserEmoticonSetsChangedCallback(const TTV_ChatUserEmoticonSets* emoticonSets)
{
	Value root;
	root["emoticonSets"] = reinterpret_cast<int64_t>(emoticonSets);

	FastWriter writer;
	std::string json = writer.write(root);

	UnitySendMessage(kUnityCallbackObjectName, "ProxyUserEmoticonSetsChangedCallback", json.c_str());
}

void ttv::chat::ManagedChatAPIListener::FireEmoticonSetDataCallback(const TTV_ChatEmoticonSetData* emoticonSet)
{
	Value root;
	root["emoticonSet"] = reinterpret_cast<int64_t>(emoticonSet);

	FastWriter writer;
	std::string json = writer.write(root);

	UnitySendMessage(kUnityCallbackObjectName, "ProxyEmoticonSetDataCallback", json.c_str());
}


void ttv::chat::ManagedChatAPIListener::FireUserBlockChangeCallback(const std::string& userName, const std::string& blockUsername, bool block, TTV_ErrorCode ec)
{
	// NOTE: Unity no longer supported
}

void ttv::chat::ManagedChatChannelListener::FireChannelStateChangedCallback(const std::string& channelName, TTV_ChatChannelState state, TTV_ErrorCode ec)
{
	Value root;
	root["channelName"] = channelName;
	root["state"] = static_cast<int64_t>(state);
	root["ec"] = static_cast<int64_t>(ec);

	FastWriter writer;
	std::string json = writer.write(root);

	UnitySendMessage(kUnityCallbackObjectName, "ProxyChannelStateChangedCallback", json.c_str());
}

void ttv::chat::ManagedChatChannelListener::FireChannelInfoChangedCallback(const std::string& channelName, const TTV_ChatChannelInfo& info)
{
	Value root;
	root["channelName"] = channelName;

	TTV_ChatChannelInfo* copy = new TTV_ChatChannelInfo;
	*copy = info;
	root["channelInfo"] = reinterpret_cast<int64_t>(copy);
			
	// NOTE: Unity will call TTV_CSharp_Chat_FreeChannelInfo to free the channelInfo

	FastWriter writer;
	std::string json = writer.write(root);

	UnitySendMessage(kUnityCallbackObjectName, "ProxyChannelInfoChangedCallback", json.c_str());
}

void ttv::chat::ManagedChatChannelListener::FireChannelLocalUserChangedCallback(const std::string& channelName, const TTV_ChatUserInfo& userInfo)
{
	Value root;
	root["channelName"] = channelName;

	TTV_ChatUserInfo* copy = new TTV_ChatUserInfo;
	*copy = userInfo;
	root["userInfo"] = reinterpret_cast<int64_t>(copy);

	// NOTE: Unity will call TTV_CSharp_Chat_FreeUserInfo to free the userInfo
	
	FastWriter writer;
	std::string json = writer.write(root);

	UnitySendMessage(kUnityCallbackObjectName, "ProxyChannelLocalUserChangedCallback", json.c_str());
}

void ttv::chat::ManagedChatChannelListener::FireChannelUserChangeCallback(const std::string& channelName, const TTV_ChatUserList* joinList, const TTV_ChatUserList* leaveList, const TTV_ChatUserList* infoChangeList)
{
	Value root;
	root["channelName"] = channelName;
	root["joinList"] = reinterpret_cast<int64_t>(joinList);
	root["leaveList"] = reinterpret_cast<int64_t>(leaveList);
	root["infoChangeList"] = reinterpret_cast<int64_t>(infoChangeList);

	// NOTE: the lists will be freed manually from Unity
	
	FastWriter writer;
	std::string json = writer.write(root);

	UnitySendMessage(kUnityCallbackObjectName, "ProxyChannelUserChangeCallback", json.c_str());
}

void ttv::chat::ManagedChatChannelListener::FireChannelMessageCallback(const std::string& channelName, const TTV_ChatMessageList* messageList)
{
	Value root;
	root["channelName"] = channelName;
	root["messageList"] = reinterpret_cast<int64_t>(messageList);

	// NOTE: the list will be freed manually from Unity
	
	FastWriter writer;
	std::string json = writer.write(root);

	UnitySendMessage(kUnityCallbackObjectName, "ProxyChannelMessageCallback", json.c_str());
}

void ttv::chat::ManagedChatChannelListener::FireChannelClearCallback(const std::string& channelName, const std::string& username)
{
	Value root;
	root["channelName"] = channelName;
	root["username"] = username;
	
	FastWriter writer;
	std::string json = writer.write(root);
		
	UnitySendMessage(kUnityCallbackObjectName, "ProxyChannelClearCallback", json.c_str());
}

void ttv::chat::ManagedChatChannelListener::FireChannelBadgeDataDownloadCallback(const std::string& channelName, TTV_ErrorCode ec)
{
	Value root;
	root["channelName"] = channelName;
	root["result"] = Value(static_cast<int64_t>(ec));
	
	FastWriter writer;
	std::string json = writer.write(root);
			
	UnitySendMessage(kUnityCallbackObjectName, "ProxyChannelBadgeDataDownloadCallback", json.c_str());
}

void ttv::chat::ManagedChatChannelListener::FireChannelHostTargetChangedCallback(const std::string& channelName, const std::string& targetChannel, uint32_t numViewers)
{
	Value root;
	root["channelName"] = channelName;
	root["targetChannel"] = targetChannel;
	root["numViewers"] = Value(static_cast<int64_t>(numViewers));

	FastWriter writer;
	std::string json = writer.write(root);

	UnitySendMessage(kUnityCallbackObjectName, "ProxyChannelHostTargetChangedCallback", json.c_str());
}

void ttv::chat::ManagedChatChannelListener::FireChannelNoticeCallback(const std::string& channelName, const std::string& id, const std::map<std::string, std::string>& params)
{
	Value root;
	root["channelName"] = channelName;
	root["id"] = id;

	Json::Value arr(Json::objectValue);

	for (auto kvp : params)
	{
		arr[kvp.first] = kvp.second;
	}

	root["params"] = arr;

	FastWriter writer;
	std::string json = writer.write(root);

	UnitySendMessage(kUnityCallbackObjectName, "ProxyChannelNoticeCallback", json.c_str());
}

void ttv::chat::ManagedChatChannelListener::FireChannelSetBroadcasterLanguageCallback(const std::string& channelName, TTV_ErrorCode ec)
{
	Value root;
	root["channelName"] = channelName;
	root["ec"] = Value(static_cast<int64_t>(ec));

	FastWriter writer;
	std::string json = writer.write(root);

	UnitySendMessage(kUnityCallbackObjectName, "ProxyChannelSetBroadcasterLanguageCallback", json.c_str());
}

extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Chat_FreeChannelInfo(TTV_ChatChannelInfo* channelInfo)
{
	TTV_RETURN_ON_NULL(channelInfo, TTV_EC_INVALID_ARG);
	
	delete channelInfo;

	return TTV_EC_SUCCESS;
}


extern "C" EXPORT_API TTV_ErrorCode TTV_CSharp_Chat_FreeUserInfo(TTV_ChatUserInfo* userInfo)
{
	TTV_RETURN_ON_NULL(userInfo, TTV_EC_INVALID_ARG);
	
	delete userInfo;

	return TTV_EC_SUCCESS;
}

#endif
