#pragma once

#include "twitchsdk/core/types/coretypes.h"
#include "twitchsdk/core/types/errortypes.h"
#include <collection.h>
#include <codecvt>
#include <memory>
#include <string>
#include <vector>
#include <map>

#include "twitch/ErrorCode.h"


namespace ttv
{
	namespace binding
	{
		namespace cx
		{
			void ToBinding(const char* const * val, Platform::String^* result);
			void ToBinding(const char* const * val, size_t length, Platform::String^* result);
			void ToBinding(const ::std::string* val, Platform::String^* result);
			void ToBinding(const ::std::string* val, Platform::String^** result);

			void ToNative(Platform::String^* val, ::std::string* result);
			inline void ToNative(Platform::String^ val, ::std::string* result) { ToNative(&val, result); }

			//template <typename N, typename B> inline bool ToNative(::Windows::Foundation::Collections::IVector<B>^& val, ::std::vector<N>& result)
			//{
			//	// TODO: Write this when needed

			//	return true;
			//}

			/**
			* The generic marshalling of array types.
			*/
			template <typename N, typename B> inline void ToBinding(const N* val, uint32_t arrayLength, ::Platform::Array<B>^* result)
			{
				if (result == nullptr)
				{
					return;
				}

				if (val == nullptr)
				{
					*result = nullptr;
				}
				else
				{
					*result = ref new Platform::Array<B>(static_cast<unsigned int>(arrayLength));

					for (uint32_t i = 0; i < arrayLength; ++i)
					{
						B bval;
						ToBinding(&(*val)[i], &bval);
						(*result)[static_cast<unsigned int>(i)] = bval;
					}
				}
			}

			/**
			 * The generic marshalling of vector types.
			 */
			template <typename N, typename B> inline void ToBinding(const ::std::vector<N>* val, ::Windows::Foundation::Collections::IVector<B>^* result)
			{
				if (result == nullptr)
				{
					return;
				}

				if (val == nullptr)
				{
					*result = nullptr;
				}
				else
				{
					*result = ref new ::Platform::Collections::Vector<B>(static_cast<unsigned int>(val->size()));

					for (size_t i = 0; i < val->size(); ++i)
					{
						B bval = nullptr;
						ToBinding(&(*val)[i], &bval);
						(*result)->SetAt(static_cast<unsigned int>(i), bval);
					}
				}
			}

			template <typename N, typename B> inline void ToBinding(const ::std::vector<N>* val, ::Platform::Array<B>^* result)
			{
				if (result == nullptr)
				{
					return;
				}

				if (val == nullptr)
				{
					*result = nullptr;
				}
				else
				{
					*result = ref new ::Platform::Array<B>(static_cast<unsigned int>(val->size()));

					const auto& v = *val;
					for (size_t i = 0; i < val->size(); ++i)
					{
						B bval = nullptr;
						ToBinding(&v[i], &bval);
						(*result)[static_cast<int>(i)] = bval;
					}
				}
			}

			/**
			 * The generic marshalling of map types.
			 */
			template <typename NK, typename NV, typename BK, typename BV> inline void ToBinding(const ::std::map<NK, NV>* val, ::Windows::Foundation::Collections::IMapView<BK, BV>^* result)
			{
				if (result == nullptr)
				{
					return;
				}

				if (val == nullptr)
				{
					*result = nullptr;
				}
				else
				{
					::Platform::Collections::Map<BK, BV>^ map = ref new ::Platform::Collections::Map<BK, BV>();

					for (const auto& kvp : (*val))
					{
						BK bkey;
						BV bval;
						ToBinding(&kvp.first, &bkey);
						ToBinding(&kvp.second, &bval);
						map->Insert(bkey, bval);
					}

					*result = map->GetView();
				}
			}

			template <typename BK, typename BV, typename NK, typename NV> inline void ToNative(::Windows::Foundation::Collections::IMapView<BK, BV>^* val, ::std::map<NK, NV>* result)
			{
				if (result == nullptr)
				{
					return;
				}

				if (val != nullptr && (*val) != nullptr)
				{
					Windows::Foundation::Collections::IIterator<Windows::Foundation::Collections::IKeyValuePair<BK, BK>^>^ iter = (*val)->First();
					while (iter->MoveNext())
					{
						Windows::Foundation::Collections::IKeyValuePair<BK, BK>^ kvp = iter->Current;
						
						BK bkey = kvp->Key;
						BK bval = kvp->Value;
						NK nkey;
						NV nval;
						ToNative(&bkey, &nkey);
						ToNative(&bval, &nval);
						(*result)[nkey] = nval;
					}
				}
			}

			/**
			 * Default implementation that simply attempts to static cast the values.
			 */
			template<typename A, typename B>
			inline void ToNative(const A* val, B* result) { if (val != nullptr && result != nullptr) *result = static_cast<B>(*val); }

			/**
			 * Default implementation that simply attempts to static cast the values.
			 */
			template<typename A, typename B>
			inline void ToBinding(const A* val, B* result) { if (val != nullptr && result != nullptr) *result = static_cast<B>(*val); }
		}
	}
}
