#include "twitchsdk/core/internal/pch.h"
#include "twitchsdk/core/cx_coreutil.h"
#include "twitchsdk/core/cx_httprequest.h"

using namespace ttv;
using namespace ttv::binding::cx;


::ttv::binding::cx::CxHttpRequest::CxHttpRequest(::Twitch::IHttpRequestProvider^ bindingImpl)
:	mBindingImpl(bindingImpl)
{
}


TTV_ErrorCode ttv::binding::cx::CxHttpRequest::SendHttpRequest(
	const std::string& url,
	const std::vector<HttpParam>& requestHeaders,
	const std::string& requestBody,
	HttpRequestType httpReqType,
	uint timeOutInSecs,
	HttpRequestHeadersCallback headersCallback,
	HttpRequestCallback responseCallback,
	void* userData)
{
	if (url.size() == 0)
	{
		return TTV_EC_INVALID_HTTP_REQUEST_PARAMS;
	}

	if (!responseCallback)
	{
		return TTV_EC_INVALID_HTTP_REQUEST_PARAMS;
	}

	if (mBindingImpl == nullptr)
	{
		return TTV_EC_NOT_INITIALIZED;
	}

	TTV_ErrorCode ec = TTV_EC_API_REQUEST_FAILED;

	// Convert native parameters into CX objects
	::Platform::String^ bindingUrl;
	ToBinding(&url, &bindingUrl);
	::Platform::String^ bindingRequestBody;
	ToBinding(&requestBody, &bindingRequestBody);
	::Platform::Array<::Twitch::HttpParam^>^ bindingHeaders;
	ToBinding(&requestHeaders, &bindingHeaders);

	::Twitch::HttpRequestType bindingRequestType;
	switch (httpReqType)
	{
		case HTTP_PUT_REQUEST:
		{
			bindingRequestType = ::Twitch::HttpRequestType::HTTP_PUT_REQUEST;
			break;
		}
		case HTTP_POST_REQUEST:
		{
			bindingRequestType = ::Twitch::HttpRequestType::HTTP_POST_REQUEST;
			break;
		}
		case HTTP_DELETE_REQUEST:
		{
			bindingRequestType = ::Twitch::HttpRequestType::HTTP_DELETE_REQUEST;
			break;
		}
		case HTTP_GET_REQUEST:
		default:
		{
			bindingRequestType = ::Twitch::HttpRequestType::HTTP_GET_REQUEST;
			break;
		}
	}

	::Twitch::HttpRequestResult^ bindingResult = nullptr;

	// Call into CX to make the request synchronously
	ttv::trace::Message("bindings", TTV_ML_DEBUG, "Calling into CX IHttpRequestProvider implementation...");
	::Twitch::ErrorCode bindingErrorCode = mBindingImpl->SendHttpRequest(bindingUrl, bindingHeaders, bindingRequestBody, bindingRequestType, timeOutInSecs, &bindingResult);
	ttv::trace::Message("bindings", TTV_ML_DEBUG, "CX IHttpRequestProvider implementation returned");

	// Convert the result into native types
	ToNative(&bindingErrorCode, &ec);

	if (TTV_SUCCEEDED(ec))
	{
		if (bindingResult == nullptr)
		{
			ec = TTV_EC_API_REQUEST_FAILED;
		}
	}

	// If the call succeeds then fire callbacks
	if (TTV_SUCCEEDED(ec))
	{
		std::map<std::string, std::string> resultHeaders;

		if (bindingResult->headers != nullptr)
		{
			::Platform::Array<::Twitch::HttpParam^>^ bindingHeaders = bindingResult->headers;
			for (unsigned int i = 0; i < bindingHeaders->Length; ++i)
			{
				const auto& header = bindingHeaders[i];
				if (header != nullptr &&
					header->paramName != nullptr &&
					header->paramValue != nullptr)
				{
					::Platform::String^ bindingName = header->paramName;
					std::string nativeName;
					ToNative(&bindingName, &nativeName);

					::Platform::String^ bindingValue = header->paramValue;
					std::string nativeValue;
					ToNative(&bindingValue, &nativeValue);

					resultHeaders[nativeName] = nativeValue;
				}
			}
		}

		// Notify the client
		bool notify = true;
		if (headersCallback != nullptr)
		{
			notify = headersCallback(bindingResult->statusCode, resultHeaders, userData);
		}

		if (notify)
		{
			// Convert the response
			std::vector<char> response;
			if (bindingResult->body != nullptr)
			{
				::Platform::String^ bindingBody = bindingResult->body;
				std::string nativeBody;
				ToNative(&bindingBody, &nativeBody);

				response.reserve(nativeBody.size());
				response.insert(response.end(), nativeBody.data(), nativeBody.data() + nativeBody.size());
			}

			responseCallback(bindingResult->statusCode, response, userData);
		}
	}

	ttv::trace::Message("bindings", TTV_ML_DEBUG, "Done processing HTTP response from CX");

	return ec;
}
