#include "twitchsdk/core/internal/pch.h"
#include "twitchsdk/core/cx_coreutil.h"
#include "twitchsdk/core/cx_socket.h"

using namespace ttv::binding::cx;


ttv::binding::cx::CxSocketBase::CxSocketBase()
:	mSendBuffer(nullptr)
{
}


void ttv::binding::cx::CxSocketBase::EnsureSendBufferSize(size_t size)
{
	unsigned int usize = static_cast<unsigned int>(size);

	if (mSendBuffer == nullptr || mSendBuffer->Length < usize)
	{
		mSendBuffer = ref new ::Platform::Array<uint8>(usize);
	}
}


ttv::binding::cx::CxSocket::CxSocket(::Twitch::ISocket^ socket)
: mBindingImpl(socket)
{
	assert(socket != nullptr);
}


TTV_ErrorCode ttv::binding::cx::CxSocket::Connect()
{
	::Twitch::ErrorCode bindingErrorCode = mBindingImpl->Connect();
	TTV_ErrorCode nativeErrorCode;
	ToNative(&bindingErrorCode, &nativeErrorCode);
	return nativeErrorCode;
}


TTV_ErrorCode ttv::binding::cx::CxSocket::Disconnect()
{
	::Twitch::ErrorCode bindingErrorCode = mBindingImpl->Disconnect();
	TTV_ErrorCode nativeErrorCode;
	ToNative(&bindingErrorCode, &nativeErrorCode);
	return nativeErrorCode;
}


TTV_ErrorCode ttv::binding::cx::CxSocket::Send(const uint8_t* buffer, size_t length, size_t& sent)
{
	TTV_ErrorCode ec = TTV_EC_SOCKET_ENOTCONN;

	sent = 0;

	// Make sure the cached byte array is large enough
	EnsureSendBufferSize(length);

	// Fill the buffer
	memcpy(mSendBuffer->Data, buffer, length);

	// Pass it to CX
	uint bindingSent = 0;
	::Twitch::ErrorCode bindingErrorCode = mBindingImpl->Send(mSendBuffer, static_cast<uint>(length), &bindingSent);

	TTV_ErrorCode nativeErrorCode;
	ToNative(&bindingErrorCode, &nativeErrorCode);

	if (TTV_SUCCEEDED(ec))
	{
		ToNative(&bindingSent, &sent);
	}

	return ec;
}


TTV_ErrorCode ttv::binding::cx::CxSocket::Recv(uint8_t* buffer, size_t length, size_t& received)
{
	received = 0;

	// Get some bytes from CX
	::Platform::Array<uint8>^ bindingBuffer = nullptr;
	uint bindingLength = 0;
	uint bindingReceived = 0;
	::Twitch::ErrorCode bindingErrorCode = mBindingImpl->Recv(&bindingBuffer, bindingLength, &bindingReceived);

	TTV_ErrorCode nativeErrorCode;
	ToNative(&bindingErrorCode, &nativeErrorCode);

	if (TTV_SUCCEEDED(nativeErrorCode))
	{
		assert(bindingBuffer != nullptr);

		if (bindingBuffer != nullptr)
		{
			received = static_cast<size_t>(bindingReceived);
			assert(received <= length);

			// Copy the bytes to native
			memcpy(buffer, bindingBuffer->Data, received);
		}
	}

	return nativeErrorCode;
}


uint64_t ttv::binding::cx::CxSocket::TotalSent()
{
	uint64 total = mBindingImpl->TotalSent();
	return static_cast<uint64_t>(total);
}


uint64_t ttv::binding::cx::CxSocket::TotalReceived()
{
	uint64 total = mBindingImpl->TotalReceived();
	return static_cast<uint64_t>(total);
}


bool ttv::binding::cx::CxSocket::Connected()
{
	return mBindingImpl->Connected();
}


ttv::binding::cx::CxSocketFactory::CxSocketFactory(::Twitch::ISocketFactory^ factory)
:	mBindingImpl(factory)
{
	assert(factory != nullptr);
}


bool ttv::binding::cx::CxSocketFactory::IsProtocolSupported(const std::string& protocol)
{
	::Platform::String^ bindingProtocol;
	ToBinding(&protocol, &bindingProtocol);
	return mBindingImpl->IsProtocolSupported(bindingProtocol);
}


TTV_ErrorCode ttv::binding::cx::CxSocketFactory::CreateSocket(const std::string& uri, std::shared_ptr<ISocket>& result)
{
	result.reset();

	::Platform::String^ bindingUri;
	ToBinding(&uri, &bindingUri);

	::Twitch::ISocket^ socket = nullptr;
	::Twitch::ErrorCode bindingErrorCode = mBindingImpl->CreateSocket(bindingUri, &socket);

	TTV_ErrorCode ec;
	ToNative(&bindingErrorCode, &ec);

	if (TTV_SUCCEEDED(ec))
	{
		if (socket != nullptr)
		{
			result = std::make_shared<CxSocket>(socket);
			ec = TTV_EC_SUCCESS;
		}
		else
		{
			ec = TTV_EC_UNIMPLEMENTED;
		}
	}

	return ec;
}


ttv::binding::cx::CxWebSocket::CxWebSocket(::Twitch::IWebSocket^ socket)
: mBindingImpl(socket)
{
	assert(socket != nullptr);
}


TTV_ErrorCode ttv::binding::cx::CxWebSocket::Connect()
{
	::Twitch::ErrorCode bindingErrorCode = mBindingImpl->Connect();
	TTV_ErrorCode nativeErrorCode;
	ToNative(&bindingErrorCode, &nativeErrorCode);
	return nativeErrorCode;
}


TTV_ErrorCode ttv::binding::cx::CxWebSocket::Disconnect()
{
	::Twitch::ErrorCode bindingErrorCode = mBindingImpl->Disconnect();
	TTV_ErrorCode nativeErrorCode;
	ToNative(&bindingErrorCode, &nativeErrorCode);
	return nativeErrorCode;
}


TTV_ErrorCode ttv::binding::cx::CxWebSocket::Send(MessageType type, const uint8_t* buffer, size_t length)
{
	TTV_ErrorCode ec = TTV_EC_SOCKET_ENOTCONN;

	// Make sure the cached byte array is large enough
	EnsureSendBufferSize(length);

	// Fill the buffer
	memcpy(mSendBuffer->Data, buffer, length);

	::Twitch::WebSocketMessageType bindingType;
	ToBinding(&type, &bindingType);

	// Pass it to CX
	::Twitch::ErrorCode bindingErrorCode = mBindingImpl->Send(bindingType, mSendBuffer, static_cast<uint>(length));

	TTV_ErrorCode nativeErrorCode;
	ToNative(&bindingErrorCode, &nativeErrorCode);

	return ec;
}


TTV_ErrorCode ttv::binding::cx::CxWebSocket::Recv(MessageType& type, uint8_t* buffer, size_t length, size_t& received)
{
	received = 0;
	type = MessageType::Unknown;

	// Get some bytes from CX
	::Platform::Array<uint8>^ bindingBuffer = nullptr;
	::Twitch::WebSocketMessageType bindingType = ::Twitch::WebSocketMessageType::None;
	uint bindingReceived = 0;
	::Twitch::ErrorCode bindingErrorCode = mBindingImpl->Recv(&bindingType, &bindingBuffer, static_cast<uint>(length), &bindingReceived);

	TTV_ErrorCode nativeErrorCode;
	ToNative(&bindingErrorCode, &nativeErrorCode);

	ToNative(&bindingType, &type);
	ToNative(&bindingReceived, &received);

	if (TTV_SUCCEEDED(nativeErrorCode))
	{
		assert(bindingBuffer != nullptr);

		if (bindingBuffer != nullptr)
		{
			received = static_cast<size_t>(bindingReceived);
			assert(received <= length);

			// Copy the bytes to native
			memcpy(buffer, bindingBuffer->Data, received);
		}
	}

	return nativeErrorCode;
}


TTV_ErrorCode ttv::binding::cx::CxWebSocket::Peek(MessageType& type, size_t& length)
{
	// Get the available message type and size from CX
	::Twitch::WebSocketMessageType bindingType = ::Twitch::WebSocketMessageType::None;
	uint bindingLength = 0;
	::Twitch::ErrorCode bindingErrorCode = mBindingImpl->Peek(&bindingType, &bindingLength);

	TTV_ErrorCode nativeErrorCode;
	ToNative(&bindingErrorCode, &nativeErrorCode);

	ToNative(&bindingType, &type);
	ToNative(&bindingLength, &length);

	return nativeErrorCode;
}


bool ttv::binding::cx::CxWebSocket::Connected()
{
	return mBindingImpl->Connected();
}


ttv::binding::cx::CxWebSocketFactory::CxWebSocketFactory(::Twitch::IWebSocketFactory^ factory)
: mBindingImpl(factory)
{
	assert(factory != nullptr);
}


bool ttv::binding::cx::CxWebSocketFactory::IsProtocolSupported(const std::string& protocol)
{
	::Platform::String^ bindingProtocol;
	ToBinding(&protocol, &bindingProtocol);
	return mBindingImpl->IsProtocolSupported(bindingProtocol);
}


TTV_ErrorCode ttv::binding::cx::CxWebSocketFactory::CreateWebSocket(const std::string& uri, std::shared_ptr<IWebSocket>& result)
{
	result.reset();

	::Platform::String^ bindingUri;
	ToBinding(&uri, &bindingUri);

	::Twitch::IWebSocket^ socket = nullptr;
	::Twitch::ErrorCode bindingErrorCode = mBindingImpl->CreateWebSocket(bindingUri, &socket);

	TTV_ErrorCode ec;
	ToNative(&bindingErrorCode, &ec);

	if (TTV_SUCCEEDED(ec))
	{
		if (socket != nullptr)
		{
			result = std::make_shared<CxWebSocket>(socket);
			ec = TTV_EC_SUCCESS;
		}
		else
		{
			ec = TTV_EC_UNIMPLEMENTED;
		}
	}

	return ec;
}
