/********************************************************************************************
* Twitch Broadcasting SDK
*
* This software is supplied under the terms of a license agreement with Twitch Interactive, Inc. and
* may not be copied or used except in accordance with the terms of that agreement
* Copyright (c) 2012-2016 Twitch Interactive, Inc.
*********************************************************************************************/

#ifndef TTVSDK_WEBCAM_SYSTEMMESSAGES_H
#define TTVSDK_WEBCAM_SYSTEMMESSAGES_H

#include "twitchwebcam.h"

namespace ttv
{
	namespace cam
	{
		class VideoCaptureDevice;

		/**
		 * Messages from the system thread to the main thread for dispatch to the SDK client.
		 */
		namespace ClientMessageType
		{
			enum Enum
			{
				SystemInitialized,			//!< The result of a call to intialize the system.
				SystemShutdown,				//!< The result of a call to shutdown the system.
	
				DeviceChange,				//!< Device found or lost.
				DeviceStatus				//!< Change in device capture status.
			};
		}

		struct ClientMessage
		{
			ClientMessageType::Enum		type;		//!< The type of message which determines what type to cast it to.
			TTV_ErrorCode				err;		//!< The error code associated with the message, if any.
			void*						userdata;	//!< Custom user data.

			virtual ~ClientMessage();
		};

		struct SystemInitializedClientMessage : public ClientMessage
		{
			TTV_WebCamInitializationCallback callback;

			SystemInitializedClientMessage(TTV_WebCamInitializationCallback cb, TTV_ErrorCode e, void* ud);
		};

		struct SystemShutdownClientMessage : public ClientMessage
		{
			TTV_WebCamShutdownCallback callback;

			SystemShutdownClientMessage(TTV_WebCamShutdownCallback cb, void* ud);
		};

		struct DeviceStatusClientMessage : public ClientMessage
		{
			unsigned int deviceIndex;
			unsigned int capabilityIndex;
			TTV_WebCamDeviceStatus status;
			TTV_WebCamDeviceStatusCallback callback;
			TTV_WebCamDeviceCapability capability;

			DeviceStatusClientMessage(unsigned int d, TTV_WebCamDeviceStatus s, const TTV_WebCamDeviceCapability& c, TTV_WebCamDeviceStatusCallback cb, void* ud);
			DeviceStatusClientMessage(unsigned int d, TTV_WebCamDeviceStatus s, unsigned int ci, TTV_ErrorCode e, TTV_WebCamDeviceStatusCallback cb, void* ud);
		};

		struct DeviceChangeClientMessage : public ClientMessage
		{
			std::shared_ptr<VideoCaptureDevice> device;
			TTV_WebCamDeviceChange change;
			TTV_WebCamDeviceChangeCallback callback;

			DeviceChangeClientMessage(std::shared_ptr<VideoCaptureDevice> d, TTV_WebCamDeviceChange ch, TTV_WebCamDeviceChangeCallback cb, void* ud);
		};



		/**
		 * Messages from the main thread to the system thread.
		 */
		namespace SystemMessageType
		{
			enum Enum
			{
				Invalid = -1,			//!< Invalid message.

				Initialize,				//!< Initialize the system.
				Shutdown,				//!< Shutdown the system.

				DeviceInitialized,		//!< Message from a device to the system that the device has finished initializing.
				DeviceShutdown,			//!< Message from a device to the system that the device has finished shutting down.

				StartDevice,			//!< Specifies that the device should be started.
				StopDevice,				//!< Specifies that the device should be stopped.

				SystemCustomMessages	//!< The beginning of the range to be used for custom messages.  These can overlap for different systems without problem.
			};
		}

		struct SystemMessage
		{
			SystemMessageType::Enum		type;		//!< The type of message which determines what type to cast it to.
			TTV_ErrorCode				err;		//!< The error code associated with the message, if any.
			void*						userdata;	//!< Custom user data.

			SystemMessage();
			virtual ~SystemMessage();
		};

		struct InitializeSystemMessage : public SystemMessage
		{
			TTV_WebCamInitializationCallback callback;

			InitializeSystemMessage(TTV_WebCamInitializationCallback cb, void* ud);
		};

		struct ShutdownSystemMessage : public SystemMessage
		{
			TTV_WebCamShutdownCallback callback;

			ShutdownSystemMessage(TTV_WebCamShutdownCallback cb, void* ud);
		};

		struct DeviceInitializedSystemMessage : public SystemMessage
		{
			unsigned int deviceIndex;

			DeviceInitializedSystemMessage(int index, TTV_ErrorCode error);
		};

		struct DeviceShutdownSystemMessage : public SystemMessage
		{
			unsigned int deviceIndex;
			bool initError;

			DeviceShutdownSystemMessage(int index, bool initError);
		};

		struct StartDeviceSystemMessage : public SystemMessage
		{
			unsigned int deviceIndex;
			unsigned int capabilityIndex;
			TTV_WebCamDeviceStatusCallback callback;
			void* userdata;

			StartDeviceSystemMessage(unsigned int index, unsigned int ci, TTV_WebCamDeviceStatusCallback cb, void* ud);
		};

		struct StopDeviceSystemMessage : public SystemMessage
		{
			unsigned int deviceIndex;
			TTV_WebCamDeviceStatusCallback callback;
			void* userdata;

			StopDeviceSystemMessage(int index, TTV_WebCamDeviceStatusCallback cb, void* ud);
		};

	} // namespace cam
} // namespace ttv

#endif	/* TTVSDK_WEBCAM_SYSTEMMESSAGES_H */
