/********************************************************************************************
* Twitch Broadcasting SDK
*
* This software is supplied under the terms of a license agreement with Twitch Interactive, Inc. and
* may not be copied or used except in accordance with the terms of that agreement
* Copyright (c) 2012-2016 Twitch Interactive, Inc.
*********************************************************************************************/

#ifndef TTV_CAM_VIDEOCAPTUREDEVICE_H
#define TTV_CAM_VIDEOCAPTUREDEVICE_H

#include <string>
#include <vector>
#include "twitchwebcam.h"
#include <concurrent_queue.h>
#include "twitchcore/mutex.h"
#include "twitchcore/thread.h"

#include "internal/webcam/devicemessages.h"


namespace ttv
{
	namespace cam
	{
		class VideoCaptureDevice;
		class VideoCaptureSystem;

		struct ClientMessage;
		struct SystemMessage;
	}
}



class ttv::cam::VideoCaptureDevice
{
public:

	VideoCaptureDevice(VideoCaptureSystem* parentSystem);
	virtual ~VideoCaptureDevice();

	/**
	 * Initialize the internals of the device.
	 */
	TTV_ErrorCode Initialize(unsigned int deviceIndex);

	/**
	 * Cleanup the internals of the device and prepare it for removal from the system.
	 */
	TTV_ErrorCode Shutdown();

	/**
	 * Retrieves the current device info.  This is only safe to be called in the client thread.  The system implementation is free to 
	 * write to this to update the status.
	 */
	TTV_WebCamDevice& GetClientDeviceInfo() { return mClientDeviceInfo; }

	/**
	 * Requests that the device begin capturing video.  This must be called on the client thread.
	 */
	TTV_ErrorCode Start(int capabilityIndex, TTV_WebCamDeviceStatusCallback callback, void* userdata);

	/**
	 * Requests that the device stop capturing video.  This must be called on the client thread.
	 */
	TTV_ErrorCode Stop(TTV_WebCamDeviceStatusCallback callback, void* userdata);

	/**
	 * Processes all events from the underlying device and process requests from the system and client.  This must be called on the device thread.
	 */
	void ProcessDeviceMessages();

	/**
	 * Determines whether or not a new frame is available via a call to LockFrame().
	 */
	bool IsFrameAvailable() const { return mReadFrame >= 0; }

	/**
	 * Fills the given buffer with the next available frame if one exists.
	 */
	TTV_ErrorCode GetFrame(void* buffer, unsigned int pitch);

	/**
	 * Attempts to retrieve the latest available frame from the device.  If a new one is available, the returned frame will be non-null and 
	 * access to the frame is locked.
	 * The caller must call UnlockFrame when done with it.
	 */
	const TTV_WebCamFrame* LockFrame();

	/**
	 * Unlocks the frame after it has been locked by LockFrame.
	 */
	void UnlockFrame();

	/**
	 * Returns when the underlying device thread has terminated.
	 */
	void WaitForThreadShutdown();
	
	// methods that the derived instance needs to implement
protected:

	/**
	 * The device implementation should begin the capturing of frames.  When start has succeeded or failed, a TTV_WEBCAM_DEVICE_STARTED 
	 * DeviceStatus event should be sent to the client.
	 */
	virtual TTV_ErrorCode StartDevice(const StartDeviceMessage* pMessage) = 0;

	/**
	 * The device implementation should stop the capturing of frames.  When stop has succeeded or failed, a TTV_WEBCAM_DEVICE_STOPPED 
	 * DeviceStatus event should be sent to the client.
	 */
	virtual TTV_ErrorCode StopDevice(const StopDeviceMessage* pMessage) = 0;

	/**
	 * The device implementation should setup the device.
	 */
	virtual TTV_ErrorCode InitializeDevice(const InitializeDeviceMessage* pMessage) = 0;

	/**
	 * The device implementation should stop the device thread and prepare for removal of the device.
	 */
	virtual TTV_ErrorCode ShutdownDevice(const ShutdownDeviceMessage* pMessage) = 0;

	/**
	 *
	 */
	virtual TTV_ErrorCode HandleCustomDeviceMessage(std::shared_ptr<cam::DeviceMessage> msg);

	/**
	 * An optional hook which is called every time through the message processing loop.  This can be used for any custom processing required.
	 */
	virtual void Update() = 0;


	// helper methods
protected:
	/**
	 * Helper function for the device to send a message to the client.
	 */
	void SendEventToClient(std::shared_ptr<ClientMessage> msg);

	/**
	 * Helper function for the device to send a message to the system.
	 */
	void SendEventToSystem(std::shared_ptr<SystemMessage> msg);

	/**
	 * This should be called by the device implementation after it is done filling in the capture frame and wants to make it available to the client.
	 */
	void SwapFrames();

	/**
	 * Marks frames as unavailable so the client does not receive any more updates.
	 */
	void ClearFrames();

	/**
	 * Determines which non-native formats can be added and appends them to the given capabilities list.
	 */
	void ExtendCapabilities(std::vector<TTV_WebCamDeviceCapability>& list);

	/**
	 * Finds the index in the mClientDeviceInfo capability list with the entry matching the given index.
	 */
	int FindCapability(unsigned int capabilityIndex) const;

	/**
	 * Find the native capability for the one given.  If it is native then the same one is returned.
	 */
	const TTV_WebCamDeviceCapability& GetParentCapability(const TTV_WebCamDeviceCapability& capability);


	// internal methods
private:
	void ThreadProc();
	void StartDeviceThread(ttv::ThreadProc proc, const char* name);


	// member variables available to the derived instance
protected:
	VideoCaptureSystem* mSystem; //!< The owning system.
	TTV_WebCamDevice mClientDeviceInfo; //!< The cached device info, accessible only in the client thread once the device reports itself as found.
	volatile bool mDeviceThreadRunning;
	volatile TTV_WebCamDeviceStatus mStatus; //!< Whether or not currently capturing.
	std::array<TTV_WebCamFrame, 2> mFrames; //!< The frames to use for captured data.
	volatile int mCaptureFrame; //!< The index into mFrames the device writes to.
	volatile int mReadFrame; //!< The index into mFrames to allow the client to read from.  This may be -1 if a new frame isn't available.
	std::shared_ptr<IMutex> mMutex; //!< The mutex to use to manage the current buffer to write to.


	// private variables
private:
	Concurrency::concurrent_queue< std::shared_ptr<DeviceMessage> > mToDeviceQ; //!< The queue for messages from the main thread to device thread.
	std::shared_ptr<IThread> mDeviceThread; //!< The thread managing the device.

	bool mClientFrameLocked; //!< Whether or not the client has locked the read frame.
};

#endif // defined TTV_CAM_VIDEOCAPTUREDEVICE_H
