/********************************************************************************************
* Twitch Broadcasting SDK
*
* This software is supplied under the terms of a license agreement with Twitch Interactive, Inc. and
* may not be copied or used except in accordance with the terms of that agreement
* Copyright (c) 2012-2016 Twitch Interactive, Inc.
*********************************************************************************************/

#ifndef TTV_CAM_VIDEOCAPTURESYSTEM_H
#define TTV_CAM_VIDEOCAPTURESYSTEM_H

#include "twitchwebcam.h"
#include "twitchcore/thread.h"
#include "internal/webcam/systemmessages.h"
#include "twitchcore/mutex.h"
#include <concurrent_queue.h>


namespace ttv
{
	namespace cam
	{
		class VideoCaptureSystem;
		class VideoCaptureDevice;
	}
}


class ttv::cam::VideoCaptureSystem
{
public:

	VideoCaptureSystem();
	virtual ~VideoCaptureSystem();

	TTV_ErrorCode Initialize(const TTV_WebCamCallbacks& interruptCallbacks, TTV_WebCamInitializationCallback initCallback, void* userdata);
	TTV_ErrorCode Shutdown(TTV_WebCamShutdownCallback callback, void* userdata);

	TTV_ErrorCode Start(unsigned int deviceIndex, unsigned int capabilityIndex, TTV_WebCamDeviceStatusCallback callback, void* userdata);
	TTV_ErrorCode Stop(unsigned int deviceIndex, TTV_WebCamDeviceStatusCallback callback, void* userdata);

	TTV_ErrorCode IsFrameAvailable(int deviceIndex, bool* available);
	TTV_ErrorCode GetFrame(int deviceIndex, void* buffer, unsigned int pitch);

	/**
	 * Processes all requests from the client.  This must be called on the system thread.
	 */
	void ProcessSystemMessages();

	/**
	 * Flushes all callbacks to the client thread.  This must be called on the client thread.
	 */
	void FlushClientEvents();

	/**
	 * Retrieves the message queue used to send messages to the client.
	 */
	Concurrency::concurrent_queue< std::shared_ptr<ClientMessage> >& GetClientQueue() { return mToClientQ; }

	/**
	 * Retrieves the message queue used to send messages to the system.
	 */
	Concurrency::concurrent_queue< std::shared_ptr<SystemMessage> >& GetSystemQueue() { return mToSystemQ; }

	/**
	 * Helper function for the device to send a message to the client.
	 */
	void SendEventToClient(std::shared_ptr<ClientMessage> msg);

	/**
	 * Helper function for the device to send a message to the system.
	 */
	void SendEventToSystem(std::shared_ptr<SystemMessage> msg);

	/**
	 * Retrieves the callback data used for firing unsolicited callbacks.
	 */
	const TTV_WebCamCallbacks& GetInterruptCallbacks() const { return mInterruptCallbacks; }

	/**
	 * Detemines whether or not the system is initialized.
	 */
	bool IsInitialized() const { return mInitialized; }


	// methods that the derived instance needs to implement
protected:

	/**
	 * The system implementation should obtain the list of devices, initialize each of them and determine their capabilities.  Once initialization is 
	 * complete the system should send the client a SystemInitialized event.  This will be called on the system thread.
	 */
	virtual TTV_ErrorCode InitializeSystem(const InitializeSystemMessage* msg) = 0;

	/**
	 * The system implementation should shut down all devices and release any references to them.  Once shutdown is 
	 * complete the system should send the client a SystemShutdown event.  This will be called on the system thread.
	 */
	virtual TTV_ErrorCode ShutdownSystem(const ShutdownSystemMessage* msg) = 0;

	/**
	 * An optional custom message handler that derived instances can implement in case their implementation generates custom messages.
	 */
	virtual TTV_ErrorCode HandleCustomSystemMessage(std::shared_ptr<cam::SystemMessage> msg);

	/**
	 * Update the internal state of the system if custom processing is needed.
	 */
	virtual void Update() = 0;


	// helper methods
protected:
	std::vector< std::shared_ptr<VideoCaptureDevice> >::iterator FindDeviceInstance(std::vector< std::shared_ptr<VideoCaptureDevice> >& list, unsigned int deviceIndex);
	std::vector< std::shared_ptr<VideoCaptureDevice> >::iterator FindDeviceInstance(std::vector< std::shared_ptr<VideoCaptureDevice> >& list, const utf8char* uniqueId);


	// internal methods
private:
	TTV_ErrorCode DeviceInitialized(const DeviceInitializedSystemMessage* msg);
	TTV_ErrorCode DeviceShutdown(const DeviceShutdownSystemMessage* msg);

	void ThreadProc();
	void StartSystemThread(ttv::ThreadProc proc, const char* name);


	// members available to all threads
protected:	
	TTV_WebCamCallbacks mInterruptCallbacks; // The client interrupt callbacks.

	Concurrency::concurrent_queue< std::shared_ptr<ClientMessage> > mToClientQ; //!< The queue for messages from the system thread to main thread.
	Concurrency::concurrent_queue< std::shared_ptr<SystemMessage> > mToSystemQ; //!< The queue for messages to the system thread.

	std::shared_ptr<IThread> mSystemThread; //!< The thread managing the system.

	// the members accessible from the client thread
protected:	
	std::vector< std::shared_ptr<VideoCaptureDevice> > mClientDevices; //!< The list of devices in the system readable by the client thread.
	std::shared_ptr<IMutex> mClientDeviceListMutex; //!< The lock on the client list to ensure that multiple client threads can access the client device list.
	bool mInitialized; //!< Whether or not the system has been initialized completely.


	// the members accessible from the system thread
protected:
	std::vector< std::shared_ptr<VideoCaptureDevice> > mSystemDevices; //!< The list of devices in the system readable by the system thread.
	unsigned int mNextDeviceIndex; //!< The next available device index.
	bool mSystemThreadRunning;
};

#endif // defined TTV_CAM_VIDEOCAPTURESYSTEM_H
