/********************************************************************************************
* Twitch Broadcasting SDK
*
* This software is supplied under the terms of a license agreement with Twitch Interactive, Inc. and
* may not be copied or used except in accordance with the terms of that agreement
* Copyright (c) 2012-2016 Twitch Interactive, Inc.
*********************************************************************************************/

#include "internal/pch.h"
#include "internal/webcam/systemmessages.h"
#include "internal/webcam/webcamformat.h"
#include <AVFoundation/AVCaptureSession.h>

#import "internal/webcam/ios/iosvideocapturedevice.h"




@implementation CaptureDeviceProxy

- (void) initializeProxy:(ttv::cam::IosVideoCaptureDevice*)owner
{
	mOwner = owner;
}


/**
 * AVCaptureVideoDataOutputSampleBufferDelegate implementation
 * The callback from AVCaptureVideoDataOutput to supply video frames.
 */
- (void)captureOutput:(AVCaptureOutput *)captureOutput didOutputSampleBuffer:(CMSampleBufferRef)sampleBuffer fromConnection:(AVCaptureConnection *)connection
{
	assert(mOwner);
	
	CVImageBufferRef pixelBuffer = CMSampleBufferGetImageBuffer(sampleBuffer);	
	mOwner->OnReceivePixelBuffer(pixelBuffer);
}

@end






ttv::cam::IosVideoCaptureDevice::IosVideoCaptureDevice(ttv::cam::VideoCaptureSystem* parentSystem, AVCaptureDevice* device)
:	ttv::cam::VideoCaptureDevice(parentSystem)
,	mNativeDevice(device)
,	mCaptureSession(nil)
,	mCaptureDeviceProxy(nil)
{
}


ttv::cam::IosVideoCaptureDevice::IosVideoCaptureDevice()
:	ttv::cam::VideoCaptureDevice(nullptr)
,	mNativeDevice(nil)
,	mCaptureSession(nil)
,	mCaptureDeviceProxy(nil)
{
}


ttv::cam::IosVideoCaptureDevice::~IosVideoCaptureDevice()
{
	mNativeDevice = nullptr;
	mCaptureSession = nil;
	mCaptureDeviceProxy = nil;
}


TTV_ErrorCode ttv::cam::IosVideoCaptureDevice::InitializeDevice(const ttv::cam::InitializeDeviceMessage* /*message*/)
{
	assert(mNativeDevice);
	
	// fill in the device info
	const char* name = mNativeDevice.localizedName.UTF8String;
	const char* uniqueID = mNativeDevice.uniqueID.UTF8String;
	snprintf(mClientDeviceInfo.uniqueId, sizeof(mClientDeviceInfo.uniqueId), "%s", uniqueID);
	snprintf(mClientDeviceInfo.name, sizeof(mClientDeviceInfo.name), "%s", name);
	
	// setup the capabilities list

	NSString* presets[] =
	{
		//AVCaptureSessionPreset320x240,
		AVCaptureSessionPreset352x288,
		AVCaptureSessionPreset640x480,
		//AVCaptureSessionPreset960x540,
		AVCaptureSessionPreset1280x720,
		AVCaptureSessionPreset1920x1080,
	};
	
	// TODO: Is there a proper way to enumerate the supported capabilities?  Doesn't seem like it.
	
	for (unsigned int i=0; i<sizeof(presets)/sizeof(presets[0]); ++i)
	{
		BOOL supportsPreset = [mNativeDevice supportsAVCaptureSessionPreset:presets[i]];
		
		if (!supportsPreset)
		{
			continue;
		}
		
		mNativeCapabilities.push_back(presets[i]);
	}
	
	// handle the case where no presets are supported
	if (mNativeCapabilities.empty())
	{
		return TTV_EC_WEBCAM_NO_PLATFORM_SUPPORT;
	}
	
	unsigned int capabilityCount = static_cast<unsigned int> (mNativeCapabilities.size());
	mClientDeviceInfo.capabilityList.count = capabilityCount;
	mClientDeviceInfo.capabilityList.list = new TTV_WebCamDeviceCapability[capabilityCount];
	
	for (unsigned int i=0; i<capabilityCount; ++i)
	{
		NSString* preset = mNativeCapabilities[i];
		
		TTV_WebCamDeviceCapability& capability = mClientDeviceInfo.capabilityList.list[i];
		
		capability.capabilityIndex = i;
		capability.isTopToBottom = true;
		capability.frameRate = 30;						// TODO: get the framerate from somewhere?
		capability.format = TTV_WEBCAM_FORMAT_ARGB32;
		capability.isNative = true;
		
		// determine the resolution - there isn't really a good way to do it
		if ( [preset compare:AVCaptureSessionPreset352x288] == NSOrderedSame )
		{
			capability.resolution.width = 352;
			capability.resolution.height = 288;
		}
		else if ( [preset compare:AVCaptureSessionPreset640x480] == NSOrderedSame )
		{
			capability.resolution.width = 640;
			capability.resolution.height = 480;
		}
		else if ( [preset compare:AVCaptureSessionPreset1280x720] == NSOrderedSame )
		{
			capability.resolution.width = 1280;
			capability.resolution.height = 720;
		}
		else if ( [preset compare:AVCaptureSessionPreset1920x1080] == NSOrderedSame )
		{
			capability.resolution.width = 1920;
			capability.resolution.height = 1080;
		}
		else
		{
			assert(false);
		}
	}
	
	return TTV_EC_SUCCESS;
}


TTV_ErrorCode ttv::cam::IosVideoCaptureDevice::ShutdownDevice(const ttv::cam::ShutdownDeviceMessage* /*message*/)
{
	// cleanup the device list
	delete [] mClientDeviceInfo.capabilityList.list;
	mClientDeviceInfo.capabilityList.list = nullptr;
	
	// TODO: need to release the device?
	mNativeDevice = nil;
	
	// TODO: need to release the strings?
	mNativeCapabilities.clear();
	
	return TTV_EC_SUCCESS;
}


TTV_ErrorCode ttv::cam::IosVideoCaptureDevice::StartDevice(const ttv::cam::StartDeviceMessage* message)
{
	if (mStatus != TTV_WEBCAM_DEVICE_STOPPED)
	{
		return TTV_EC_WEBCAM_DEVICE_ALREADY_STARTED;
	}
	
	int listIndex = FindCapability(message->capabilityIndex);

	if (listIndex < 0)
	{
		// unsupported capability
		std::shared_ptr<DeviceStatusClientMessage> reply( new DeviceStatusClientMessage(mClientDeviceInfo.deviceIndex, TTV_WEBCAM_DEVICE_STOPPED, message->capabilityIndex, TTV_EC_WEBCAM_INVALID_CAPABILITY, message->callback, message->userdata) );
		SendEventToClient(reply);
		
		return TTV_EC_WEBCAM_INVALID_PARAMETER;
	}
	
	// allocate buffers
	const TTV_WebCamDeviceCapability& capability = mClientDeviceInfo.capabilityList.list[listIndex];
	const TTV_WebCamDeviceCapability& parentCapability = GetParentCapability(capability);

	mCaptureSession = [[AVCaptureSession alloc] init];
	
	[mCaptureSession beginConfiguration];
	{
		// configure the camera
		// TODO: make sure to change this to the proper capability
		mCaptureSession.sessionPreset = mNativeCapabilities[parentCapability.capabilityIndex];
		
		// configure the input to the session from the camera device
		NSError* error = nil;
		AVCaptureDeviceInput* videoIn = [AVCaptureDeviceInput deviceInputWithDevice:mNativeDevice error:&error];
		if (videoIn)
		{
			[mCaptureSession addInput:videoIn];
		}
		else
		{
			assert(false);
			
			[mCaptureSession release];
			mCaptureSession = nil;
			return TTV_EC_WEBCAM_FAILED_TO_START;
		}
		
		// configure the output of the session to call our delegate with the pixel data
		AVCaptureVideoDataOutput* videoOut = [[AVCaptureVideoDataOutput alloc] init];
		if (videoOut)
		{
			[videoOut setAlwaysDiscardsLateVideoFrames:YES];
			[videoOut setVideoSettings:[NSDictionary dictionaryWithObject:[NSNumber numberWithInt:kCVPixelFormatType_32BGRA] forKey:id(kCVPixelBufferPixelFormatTypeKey)]]; // BGRA is necessary for manual preview
		}
		else
		{
			assert(false);
			
			[mCaptureSession removeInput:videoIn];
			[mCaptureSession release];
			mCaptureSession = nil;
			return TTV_EC_WEBCAM_FAILED_TO_START;
		}
		
		// create the queue for capturing frames
		char buffer[32];
		sprintf(buffer, "twitch.cam.device.%d", mClientDeviceInfo.deviceIndex);
		dispatch_queue_t captureQueue = dispatch_queue_create(buffer, NULL);
		
		// create the proxy object
		mCaptureDeviceProxy = [[CaptureDeviceProxy alloc] init];
		[mCaptureDeviceProxy initializeProxy:this];
		
		if ([mCaptureSession canAddOutput:videoOut])
		{
			[mCaptureSession addOutput:videoOut];
		}
		else
		{
			assert(false);

			[mCaptureSession removeInput:videoIn];
			[mCaptureSession release];
			mCaptureSession = nil;
			return TTV_EC_WEBCAM_FAILED_TO_START;
		}
		
		// set the delegate for capturing the frames
		[videoOut setSampleBufferDelegate:mCaptureDeviceProxy queue:captureQueue];
		dispatch_release( captureQueue );
	}
	[mCaptureSession commitConfiguration];
	
	int bufferSize = GetBufferSize(parentCapability.format, parentCapability.resolution.width, parentCapability.resolution.height);
	
	for (size_t i=0; i<mFrames.max_size(); ++i)
	{
		mFrames[i].deviceIndex = mClientDeviceInfo.deviceIndex;
		mFrames[i].bufferSize = bufferSize;
		mFrames[i].capability = capability;
		mFrames[i].imageBuffer = new uint8_t[bufferSize];
		
		memset(mFrames[i].imageBuffer, 0, bufferSize);
	}
	
	// start capturing
	[mCaptureSession startRunning];
	
	mCaptureFrame = 0;
	mStatus = TTV_WEBCAM_DEVICE_STARTED;
	
	// notify the client
	std::shared_ptr<DeviceStatusClientMessage> reply( new DeviceStatusClientMessage(mClientDeviceInfo.deviceIndex, TTV_WEBCAM_DEVICE_STARTED, capability, message->callback, message->userdata) );
	SendEventToClient(reply);
	
	return TTV_EC_SUCCESS;
}


TTV_ErrorCode ttv::cam::IosVideoCaptureDevice::StopDevice(const ttv::cam::StopDeviceMessage* message)
{
	assert(mDeviceThreadRunning);

	if (mStatus != TTV_WEBCAM_DEVICE_STARTED)
	{
		return TTV_EC_WEBCAM_DEVICE_NOT_STARTED;
	}

	mStatus = TTV_WEBCAM_DEVICE_STOPPED;

	[mCaptureSession stopRunning];
	
	// remove the input to the session from the camera device
	NSError* error = nil;
	AVCaptureDeviceInput* videoIn = [AVCaptureDeviceInput deviceInputWithDevice:mNativeDevice error:&error];
	[mCaptureSession removeInput:videoIn];
	
	[mCaptureSession release];
	mCaptureSession = nil;
	
	// free the proxy object
	if (mCaptureDeviceProxy)
	{
		[mCaptureDeviceProxy dealloc];
		mCaptureDeviceProxy = nil;
	}
	
	// stop the client thread from receiving frame updates
	ClearFrames();

	// deallocate buffers
	(void)mMutex.Lock();
	for (size_t i = 0; i < mFrames.max_size(); ++i)
	{
		delete [] mFrames[i].imageBuffer;
		mFrames[i].imageBuffer = nullptr;
	}
	(void)mMutex.Unlock();

	// notify the client
	if (message != nullptr)
	{
		int listIndex = FindCapability(mFrames[0].capability.capabilityIndex);
		assert(listIndex >= 0);

		const TTV_WebCamDeviceCapability& capability = mClientDeviceInfo.capabilityList.list[listIndex];

		SendEventToClient(std::make_shared<ttv::cam::DeviceStatusClientMessage>(mClientDeviceInfo.deviceIndex, TTV_WEBCAM_DEVICE_STOPPED, capability, message->callback, message->userdata));
	}

	return TTV_EC_SUCCESS;
}


void ttv::cam::IosVideoCaptureDevice::OnReceivePixelBuffer(CVImageBufferRef pixelBuffer)
{
	TTV_WebCamFrame* frame = &mFrames[mCaptureFrame];
	assert(frame->imageBuffer);

	CVReturn ret = CVPixelBufferLockBaseAddress(pixelBuffer, 0);
	if (ret == kCVReturnSuccess)
	{
		void* src = CVPixelBufferGetBaseAddress(pixelBuffer);
		size_t expectedPitch = frame->capability.resolution.width * GetPixelSize(frame->capability.format);
		
		size_t width = CVPixelBufferGetWidth(pixelBuffer);
		size_t height = CVPixelBufferGetHeight(pixelBuffer);
		size_t bufferSize = CVPixelBufferGetDataSize(pixelBuffer);
		size_t pitch = CVPixelBufferGetBytesPerRow(pixelBuffer);
		
		assert(width == frame->capability.resolution.width);
		assert(height == frame->capability.resolution.height);
		assert(bufferSize >= frame->bufferSize);			// The buffer supplied by the device has 8 extra bytes at the end
		assert(pitch == expectedPitch);
		
		size_t copySize = frame->bufferSize < bufferSize ? frame->bufferSize : bufferSize;
		
		memcpy(frame->imageBuffer, src, copySize);

		CVPixelBufferUnlockBaseAddress(pixelBuffer, 0);
		
		SwapFrames();
	}
	else
	{
		assert(false);
	}
}


void ttv::cam::IosVideoCaptureDevice::Update()
{
}
