/********************************************************************************************
* Twitch Broadcasting SDK
*
* This software is supplied under the terms of a license agreement with Twitch Interactive, Inc. and
* may not be copied or used except in accordance with the terms of that agreement
* Copyright (c) 2012-2016 Twitch Interactive, Inc.
*********************************************************************************************/

#include "internal/pch.h"
#include "internal/webcam/testvideocapturesystem.h"
#include "internal/webcam/testvideocapturedevice.h"
#include "internal/webcam/webcamformat.h"
#include "twitchcore/systemclock.h"

#pragma clang diagnostic ignored "-Wold-style-cast"


ttv::cam::TestVideoCaptureDevice::TestVideoCaptureDevice(VideoCaptureSystem* captureSystem)
:	VideoCaptureDevice(captureSystem)
,	mLastCaptureTime(0)
,	mBaseColor(0)
{
}


//-----------------------------------------------------------------------
ttv::cam::TestVideoCaptureDevice::~TestVideoCaptureDevice()
{
}


void ttv::cam::TestVideoCaptureDevice::SetTestDeviceData(const utf8char* uniqueId, const utf8char* name, const TTV_WebCamDeviceCapability* capabilities, unsigned int capabilityCount, unsigned int color)
{
	assert(!mDeviceThreadRunning);

	CleanupTestDeviceData();

	sprintf(mClientDeviceInfo.uniqueId, "%s", uniqueId);
	sprintf(mClientDeviceInfo.name, "%s", name);
	mBaseColor = color;

	mTestCapabilities.clear();

	for (size_t i=0; i<capabilityCount; ++i)
	{
		mTestCapabilities.push_back(capabilities[i]);
	}
}


void ttv::cam::TestVideoCaptureDevice::CleanupTestDeviceData()
{
	if (mClientDeviceInfo.capabilityList.list != nullptr)
	{
		delete [] mClientDeviceInfo.capabilityList.list;
		mClientDeviceInfo.capabilityList.list = nullptr;
	}
}


TTV_ErrorCode ttv::cam::TestVideoCaptureDevice::InitializeDevice(const InitializeDeviceMessage* pMessage)
{
	UNUSED(pMessage);

	// fill in with converted types as needed
	ExtendCapabilities(mTestCapabilities);

	// allocate the device info
	mClientDeviceInfo.capabilityList.count = (unsigned int)mTestCapabilities.size();
	mClientDeviceInfo.capabilityList.list = new TTV_WebCamDeviceCapability[mTestCapabilities.size()];

	for (size_t i=0; i<mClientDeviceInfo.capabilityList.count; ++i)
	{
#pragma warning(suppress: 6386) // Definitely no buffer overrun
							    // Todo Replease loop with std::copy
		mClientDeviceInfo.capabilityList.list[i] = mTestCapabilities[i];
	}

	mTestCapabilities.clear();

	return TTV_EC_SUCCESS;
}


TTV_ErrorCode ttv::cam::TestVideoCaptureDevice::ShutdownDevice(const ShutdownDeviceMessage* pMessage)
{
	UNUSED(pMessage);

	CleanupTestDeviceData();

	return TTV_EC_SUCCESS;
}


TTV_ErrorCode ttv::cam::TestVideoCaptureDevice::StartDevice(const StartDeviceMessage* pMessage)
{
	assert(mDeviceThreadRunning);

	// TODO: report already started error?
	if (mStatus == TTV_WEBCAM_DEVICE_STARTED)
	{
		return TTV_EC_SUCCESS;
	}

	// shutting down so don't start capturing
	if (!mDeviceThreadRunning)
	{
		return TTV_EC_WEBCAM_NOT_INITIALIZED;
	}

	int listIndex = FindCapability(pMessage->capabilityIndex);

	// make sure we support the capability
	if (listIndex < 0)
	{
		// unsupported capability
		std::shared_ptr<DeviceStatusClientMessage> reply( new DeviceStatusClientMessage(mClientDeviceInfo.deviceIndex, TTV_WEBCAM_DEVICE_STOPPED, pMessage->capabilityIndex, TTV_EC_WEBCAM_INVALID_CAPABILITY, pMessage->callback, pMessage->userdata) );
		SendEventToClient(reply);

		return TTV_EC_WEBCAM_INVALID_CAPABILITY;
	}

	const TTV_WebCamDeviceCapability& capability = mClientDeviceInfo.capabilityList.list[listIndex];
	const TTV_WebCamDeviceCapability& parentCapability = GetParentCapability(capability);

	int bufferSize = GetBufferSize(parentCapability.format, parentCapability.resolution.width, parentCapability.resolution.height);

	// allocate buffers
	for (size_t i=0; i<mFrames.size(); ++i)
	{
		mFrames[i].deviceIndex = mClientDeviceInfo.deviceIndex;
		mFrames[i].bufferSize = bufferSize;
		mFrames[i].capability = capability;
		mFrames[i].imageBuffer = new uint8_t[bufferSize];

		memset(mFrames[i].imageBuffer, 0, bufferSize);
	}

	mCaptureFrame = 0;
	mStatus = TTV_WEBCAM_DEVICE_STARTED;

	// notify the client
	std::shared_ptr<DeviceStatusClientMessage> reply( new DeviceStatusClientMessage(mClientDeviceInfo.deviceIndex, TTV_WEBCAM_DEVICE_STARTED, capability, pMessage->callback, pMessage->userdata) );
	SendEventToClient(reply);

	return TTV_EC_SUCCESS;
}


TTV_ErrorCode ttv::cam::TestVideoCaptureDevice::StopDevice(const StopDeviceMessage* pMessage)
{
	assert(mDeviceThreadRunning);

	if (mStatus != TTV_WEBCAM_DEVICE_STARTED)
	{
		return TTV_EC_WEBCAM_DEVICE_NOT_STARTED;
	}

	mStatus = TTV_WEBCAM_DEVICE_STOPPED;

	// stop the client thread from receiving frame updates
	ClearFrames();

	// deallocate buffers
	(void)mMutex->Lock();
	for (size_t i=0; i<mFrames.size(); ++i)
	{
		delete [] mFrames[i].imageBuffer;
		mFrames[i].imageBuffer = nullptr;
	}
	(void)mMutex->Unlock();

	// notify the client
	if (pMessage != nullptr)
	{
		int listIndex = FindCapability(mFrames[0].capability.capabilityIndex);
		assert(listIndex >= 0);

		const TTV_WebCamDeviceCapability& capability = mClientDeviceInfo.capabilityList.list[listIndex];

		std::shared_ptr<DeviceStatusClientMessage> reply( new DeviceStatusClientMessage(mClientDeviceInfo.deviceIndex, TTV_WEBCAM_DEVICE_STOPPED, capability, pMessage->callback, pMessage->userdata) );
		SendEventToClient(reply);
	}

	return TTV_EC_SUCCESS;
}


float CalculateTimeModulation()
{
	unsigned int rate = 1000;
	float timeModulation = (float)(ttv::SystemTimeToMs( ttv::GetSystemClockTime() ) % (rate+1)) / rate;
	return timeModulation*0.5f + 0.5f;
}


void GenerateXRGB32(TTV_WebCamFrame* frame, int baseColor)
{
	float timeModulation = CalculateTimeModulation();

	int index = 0;
	unsigned int* p = reinterpret_cast<unsigned int*>(frame->imageBuffer);
	for (unsigned int y=0; y<frame->capability.resolution.height; ++y)
	{
		float rowModulation = (float)y / (float)(frame->capability.resolution.height-1);
		if (!frame->capability.isTopToBottom)
		{
			rowModulation = 1 - rowModulation;
		}

		float modulation = timeModulation * rowModulation;
		unsigned int rowColor = baseColor;
		unsigned char* c = reinterpret_cast<unsigned char*>(&rowColor);
		c[0] = (unsigned char)((float)c[0] * modulation);
		c[1] = (unsigned char)((float)c[1] * modulation);
		c[2] = (unsigned char)((float)c[2] * modulation);
		c[3] = (unsigned char)(rand() % 256);

		for (unsigned int x=0; x<frame->capability.resolution.width; ++x)
		{
			p[index] = rowColor;
			index++;
		}
	}
}

void GenerateARGB32(TTV_WebCamFrame* frame, int baseColor)
{
	float timeModulation = CalculateTimeModulation();

	int index = 0;
	unsigned int* p = reinterpret_cast<unsigned int*>(frame->imageBuffer);
	for (unsigned int y=0; y<frame->capability.resolution.height; ++y)
	{
		float rowModulation = (float)y / (float)(frame->capability.resolution.height-1);
		if (!frame->capability.isTopToBottom)
		{
			rowModulation = 1 - rowModulation;
		}

		float modulation = timeModulation * rowModulation;
		unsigned int rowColor = baseColor;
		unsigned char* c = reinterpret_cast<unsigned char*>(&rowColor);
		c[0] = (unsigned char)((float)c[0] * modulation);
		c[1] = (unsigned char)((float)c[1] * modulation);
		c[2] = (unsigned char)((float)c[2] * modulation);
		c[3] = (unsigned char)((float)c[3] * modulation);

		for (unsigned int x=0; x<frame->capability.resolution.width; ++x)
		{
			p[index] = rowColor;
			index++;
		}
	}
}


void GenerateRGB24(TTV_WebCamFrame* frame, int baseColor)
{
	float timeModulation = CalculateTimeModulation();

	uint8_t* p = frame->imageBuffer;
	for (unsigned int y=0; y<frame->capability.resolution.height; ++y)
	{
		float rowModulation = (float)y / (float)(frame->capability.resolution.height-1);
		if (!frame->capability.isTopToBottom)
		{
			rowModulation = 1 - rowModulation;
		}

		float modulation = timeModulation * rowModulation;
		unsigned int rowColor = baseColor;
		unsigned char* c = reinterpret_cast<unsigned char*>(&rowColor);
		c[0] = (unsigned char)((float)c[0] * modulation);
		c[1] = (unsigned char)((float)c[1] * modulation);
		c[2] = (unsigned char)((float)c[2] * modulation);

		for (unsigned int x=0; x<frame->capability.resolution.width; ++x)
		{
			p[0] = c[0];
			p[1] = c[1];
			p[2] = c[2];
			p += 3;
		}
	}
}


void GenerateRGB565(TTV_WebCamFrame* frame, int baseColor)
{
	float timeModulation = CalculateTimeModulation();

	uint16_t* p = reinterpret_cast<uint16_t*>(frame->imageBuffer);
	for (unsigned int y=0; y<frame->capability.resolution.height; ++y)
	{
		float rowModulation = (float)y / (float)(frame->capability.resolution.height-1);
		if (!frame->capability.isTopToBottom)
		{
			rowModulation = 1 - rowModulation;
		}

		float modulation = timeModulation * rowModulation;
		unsigned int rowColor = baseColor;

		unsigned char* c = reinterpret_cast<unsigned char*>(&rowColor);
		c[0] = (unsigned char)((float)c[0] * modulation);
		c[1] = (unsigned char)((float)c[1] * modulation);
		c[2] = (unsigned char)((float)c[2] * modulation);

		unsigned char r = c[0] >> 3;
		unsigned char g = c[1] >> 2;
		unsigned char b = c[2] >> 3;
		uint16_t final = ((uint16_t)b << 11) | ((uint16_t)g << 5) | ((uint16_t)r << 0);

		for (unsigned int x=0; x<frame->capability.resolution.width; ++x)
		{
			*p = final;
			++p;
		}
	}
}


void GenerateRGB555(TTV_WebCamFrame* frame, int baseColor)
{
	float timeModulation = CalculateTimeModulation();

	uint16_t* p = reinterpret_cast<uint16_t*>(frame->imageBuffer);
	for (unsigned int y=0; y<frame->capability.resolution.height; ++y)
	{
		float rowModulation = (float)y / (float)(frame->capability.resolution.height-1);
		if (!frame->capability.isTopToBottom)
		{
			rowModulation = 1 - rowModulation;
		}

		float modulation = timeModulation * rowModulation;
		unsigned int rowColor = baseColor;

		unsigned char* c = reinterpret_cast<unsigned char*>(&rowColor);
		c[0] = (unsigned char)((float)c[0] * modulation);
		c[1] = (unsigned char)((float)c[1] * modulation);
		c[2] = (unsigned char)((float)c[2] * modulation);

		unsigned char r = c[0] >> 3;
		unsigned char g = c[1] >> 3;
		unsigned char b = c[2] >> 3;
		uint16_t final = (1 << 15) | ((uint16_t)b << 10) | ((uint16_t)g << 5) | ((uint16_t)r << 0);

		for (unsigned int x=0; x<frame->capability.resolution.width; ++x)
		{
			*p = final;
			++p;
		}
	}
}


void GenerateYUY2(TTV_WebCamFrame* frame, int baseColor)
{
	// TODO: this doesn't work right for colors other than red

	assert(frame->capability.resolution.width % 2 == 0);

	// constants for generating YUY2 values
	float wr = 0.299f;
	float wb = 0.144f;
	float wg = 0.587f;

	float umin = -0.436f;
	float umax = 0.436f;
	float vmin = -0.615f;
	float vmax = 0.615f;

	float timeModulation = CalculateTimeModulation();

	uint32_t* p = reinterpret_cast<uint32_t*>(frame->imageBuffer);
	for (unsigned int y=0; y<frame->capability.resolution.height; ++y)
	{
		float rowModulation = (float)y / (float)(frame->capability.resolution.height-1);
		if (!frame->capability.isTopToBottom)
		{
			rowModulation = 1 - rowModulation;
		}

		float modulation = timeModulation * rowModulation;
		unsigned int rowColor = baseColor;

		unsigned char* c = reinterpret_cast<unsigned char*>(&rowColor);
		c[0] = (unsigned char)((float)c[0] * modulation);
		c[1] = (unsigned char)((float)c[1] * modulation);
		c[2] = (unsigned char)((float)c[2] * modulation);

		// normalize r,g,b
		float r = (float)c[2] / 255.0f;
		float g = (float)c[1] / 255.0f;
		float b = (float)c[0] / 255.0f;

		// compute y,u,v
		float yf = wr*r + wg*g * wb*b;
		float uf = 0.492f * (b - yf);
		float vf = 0.877f * (r - yf);

		// normalize u and v
		uf = (uf-umin) / (umax-umin);
		vf = (vf-vmin) / (vmax-vmin);

		// scale to the range [0,255]
		unsigned char yc = (unsigned char)(yf * 255.0f);
		unsigned char uc = (unsigned char)(uf * 255.0f);
		unsigned char vc = (unsigned char)(vf * 255.0f);

		// since the whole row is the same color we assume y0 and y1 are the same for the neighboring pixels
		uint32_t final = ((uint32_t)yc << 0) | ((uint32_t)uc << 8) | ((uint32_t)yc << 16) | ((uint32_t)vc << 24);

		// write 2 pixels at a time
		for (unsigned int x=0; x<frame->capability.resolution.width; x+=2)
		{
			*p = final;
			++p;
		}
	}
}


void ttv::cam::TestVideoCaptureDevice::Update()
{
	// only generate a frame when capturing
	if (mStatus != TTV_WEBCAM_DEVICE_STARTED)
	{
		return;
	}

	// no frame to capture to
	if (mCaptureFrame < 0)
	{
		return;
	}

	TTV_WebCamFrame* frame = &mFrames[mCaptureFrame];
	assert(frame->imageBuffer);

	// pace captures to the capability frame rate
	uint64_t now = SystemTimeToMs( GetSystemClockTime() );
	uint64_t delta = now - mLastCaptureTime;

	if (delta < 1000 / (uint64_t)frame->capability.frameRate)
	{
		return;
	}

	mLastCaptureTime = now;

	switch (frame->capability.format)
	{
	case TTV_WEBCAM_FORMAT_XRGB32:
		GenerateXRGB32(frame, mBaseColor);
		break;
	case TTV_WEBCAM_FORMAT_ARGB32:
		GenerateARGB32(frame, mBaseColor);
		break;
	case TTV_WEBCAM_FORMAT_RGB24:
		GenerateRGB24(frame, mBaseColor);
		break;
	case TTV_WEBCAM_FORMAT_RGB565:
		GenerateRGB565(frame, mBaseColor);
		break;
	case TTV_WEBCAM_FORMAT_RGB555:
		GenerateRGB555(frame, mBaseColor);
		break;
	case TTV_WEBCAM_FORMAT_YUY2:
		GenerateYUY2(frame, mBaseColor);
		break;
	default:
		GenerateGarbageFrame(frame);
		break;
	}

	SwapFrames();
}
